from typing import Any, ClassVar, Dict, Sequence, List, Iterator, Optional
from typing import overload
import klayout.tl as tl
class Box:
    r"""
    @brief A box class with integer coordinates

    This object represents a box (a rectangular shape).

    The definition of the attributes is: p1 is the lower left point, p2 the 
    upper right one. If a box is constructed from two points (or four coordinates), the 
    coordinates are sorted accordingly.

    A box can be empty. An empty box represents no area
    (not even a point). Empty boxes behave neutral with respect to most operations. 
    Empty boxes return true on \empty?.

    A box can be a point or a single
    line. In this case, the area is zero but the box still
    can overlap other boxes for example and it is not empty. 

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    bottom: int
    r"""
    Getter:
    @brief Gets the bottom coordinate of the box

    Setter:
    @brief Sets the bottom coordinate of the box
    """
    left: int
    r"""
    Getter:
    @brief Gets the left coordinate of the box

    Setter:
    @brief Sets the left coordinate of the box
    """
    p1: Point
    r"""
    Getter:
    @brief Gets the lower left point of the box

    Setter:
    @brief Sets the lower left point of the box
    """
    p2: Point
    r"""
    Getter:
    @brief Gets the upper right point of the box

    Setter:
    @brief Sets the upper right point of the box
    """
    right: int
    r"""
    Getter:
    @brief Gets the right coordinate of the box

    Setter:
    @brief Sets the right coordinate of the box
    """
    top: int
    r"""
    Getter:
    @brief Gets the top coordinate of the box

    Setter:
    @brief Sets the top coordinate of the box
    """
    @classmethod
    def from_dbox(cls, dbox: DBox) -> Box:
        r"""
        @brief Creates an integer coordinate box from a floating-point coordinate box

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dbox'.
        """
    @classmethod
    def from_s(cls, s: str) -> Box:
        r"""
        @brief Creates a box object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> Box:
        r"""
        @brief Creates an empty (invalid) box

        Empty boxes don't modify a box when joined with it. The intersection between an empty and any other box is also an empty box. The width, height, p1 and p2 attributes of an empty box are undefined. Use \empty? to get a value indicating whether the box is empty.
        """
    @overload
    @classmethod
    def new(cls, arg0: int) -> Box:
        r"""
        @brief Creates a square with the given dimensions centered around the origin

        Note that for integer-unit boxes, the dimension has to be an even number to avoid rounding.

        This convenience constructor has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, dbox: DBox) -> Box:
        r"""
        @brief Creates an integer coordinate box from a floating-point coordinate box

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dbox'.
        """
    @overload
    @classmethod
    def new(cls, arg0: int, arg1: int) -> Box:
        r"""
        @brief Creates a rectangle with given width and height, centered around the origin

        Note that for integer-unit boxes, the dimensions have to be an even number to avoid rounding.

        This convenience constructor has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, lower_left: Point, upper_right: Point) -> Box:
        r"""
        @brief Creates a box from two points


        Two points are given to create a new box. If the coordinates are not provided in the correct order (i.e. right < left), these are swapped.
        """
    @overload
    @classmethod
    def new(cls, left: int, bottom: int, right: int, top: int) -> Box:
        r"""
        @brief Creates a box with four coordinates


        Four coordinates are given to create a new box. If the coordinates are not provided in the correct order (i.e. right < left), these are swapped.
        """
    @classmethod
    def world(cls) -> Box:
        r"""
        @brief Gets the 'world' box
        The world box is the biggest box that can be represented. So it is basically 'all'. The world box behaves neutral on intersections for example. In other operations such as displacement or transformations, the world box may render unexpected results because of coordinate overflow.

        The world box can be used
        @ul
          @li for comparison ('==', '!=', '<') @/li
          @li in union and intersection ('+' and '&') @/li
          @li in relations (\contains?, \overlaps?, \touches?) @/li
          @li as 'all' argument in region queries @/li
        @/ul

        This method has been introduced in version 0.28.
        """
    @overload
    def __add__(self, box: Box) -> Box:
        r"""
        @brief Joins two boxes


        The + operator joins the first box with the one given as 
        the second argument. Joining constructs a box that encloses
        both boxes given. Empty boxes are neutral: they do not
        change another box when joining. Overwrites this box
        with the result.

        @param box The box to join with this box.

        @return The joined box
        """
    @overload
    def __add__(self, point: Point) -> Box:
        r"""
        @brief Joins box with a point


        The + operator joins a point with the box. The resulting box will enclose both the original box and the point.

        @param point The point to join with this box.

        @return The box joined with the point
        """
    def __and__(self, box: Box) -> Box:
        r"""
        @brief Returns the intersection of this box with another box


        The intersection of two boxes is the largest
        box common to both boxes. The intersection may be 
        empty if both boxes to not touch. If the boxes do
        not overlap but touch the result may be a single
        line or point with an area of zero. Overwrites this box
        with the result.

        @param box The box to take the intersection with

        @return The intersection box
        """
    def __copy__(self) -> Box:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> Box:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, box: object) -> bool:
        r"""
        @brief Returns true if this box is equal to the other box
        Returns true, if this box and the given box are equal 
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given box. This method enables boxes as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Creates an empty (invalid) box

        Empty boxes don't modify a box when joined with it. The intersection between an empty and any other box is also an empty box. The width, height, p1 and p2 attributes of an empty box are undefined. Use \empty? to get a value indicating whether the box is empty.
        """
    @overload
    def __init__(self, arg0: int) -> None:
        r"""
        @brief Creates a square with the given dimensions centered around the origin

        Note that for integer-unit boxes, the dimension has to be an even number to avoid rounding.

        This convenience constructor has been introduced in version 0.28.
        """
    @overload
    def __init__(self, dbox: DBox) -> None:
        r"""
        @brief Creates an integer coordinate box from a floating-point coordinate box

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dbox'.
        """
    @overload
    def __init__(self, arg0: int, arg1: int) -> None:
        r"""
        @brief Creates a rectangle with given width and height, centered around the origin

        Note that for integer-unit boxes, the dimensions have to be an even number to avoid rounding.

        This convenience constructor has been introduced in version 0.28.
        """
    @overload
    def __init__(self, lower_left: Point, upper_right: Point) -> None:
        r"""
        @brief Creates a box from two points


        Two points are given to create a new box. If the coordinates are not provided in the correct order (i.e. right < left), these are swapped.
        """
    @overload
    def __init__(self, left: int, bottom: int, right: int, top: int) -> None:
        r"""
        @brief Creates a box with four coordinates


        Four coordinates are given to create a new box. If the coordinates are not provided in the correct order (i.e. right < left), these are swapped.
        """
    def __lt__(self, box: Box) -> bool:
        r"""
        @brief Returns true if this box is 'less' than another box
        Returns true, if this box is 'less' with respect to first and second point (in this order)
        """
    @overload
    def __mul__(self, box: Box) -> Box:
        r"""
        @brief Returns the convolution product from this box with another box


        The * operator convolves the firstbox with the one given as 
        the second argument. The box resulting from "convolution" is the
        outer boundary of the union set formed by placing 
        the second box at every point of the first. In other words,
        the returned box of (p1,p2)*(q1,q2) is (p1+q1,p2+q2).

        @param box The box to convolve with this box.

        @return The convolved box
        """
    @overload
    def __mul__(self, scale_factor: float) -> Box:
        r"""
        @brief Returns the scaled box


        The * operator scales the box with the given factor and returns the result.

        This method has been introduced in version 0.22.

        @param scale_factor The scaling factor

        @return The scaled box
        """
    def __ne__(self, box: object) -> bool:
        r"""
        @brief Returns true if this box is not equal to the other box
        Returns true, if this box and the given box are not equal 
        """
    @overload
    def __rmul__(self, box: Box) -> Box:
        r"""
        @brief Returns the convolution product from this box with another box


        The * operator convolves the firstbox with the one given as 
        the second argument. The box resulting from "convolution" is the
        outer boundary of the union set formed by placing 
        the second box at every point of the first. In other words,
        the returned box of (p1,p2)*(q1,q2) is (p1+q1,p2+q2).

        @param box The box to convolve with this box.

        @return The convolved box
        """
    @overload
    def __rmul__(self, scale_factor: float) -> Box:
        r"""
        @brief Returns the scaled box


        The * operator scales the box with the given factor and returns the result.

        This method has been introduced in version 0.22.

        @param scale_factor The scaling factor

        @return The scaled box
        """
    def __str__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing this box

        This string can be turned into a box again by using \from_s
        . If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def area(self) -> float:
        r"""
        @brief Computes the box area

        Returns the box area or 0 if the box is empty
        """
    def assign(self, other: Box) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> Box:
        r"""
        @brief Returns the bounding box
        This method is provided for consistency of the shape API is returns the box itself.

        This method has been introduced in version 0.27.
        """
    def center(self) -> Point:
        r"""
        @brief Gets the center of the box
        """
    @overload
    def contains(self, point: Point) -> bool:
        r"""
        @brief Returns true if the box contains the given point


        Tests whether a point is inside the box.
        It also returns true if the point is exactly on the box contour.

        @param p The point to test against.

        @return true if the point is inside the box.
        """
    @overload
    def contains(self, x: int, y: int) -> bool:
        r"""
        @brief Returns true if the box contains the given point


        Tests whether a point (x, y) is inside the box.
        It also returns true if the point is exactly on the box contour.

        @return true if the point is inside the box.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> Box:
        r"""
        @brief Creates a copy of self
        """
    def empty(self) -> bool:
        r"""
        @brief Returns a value indicating whether the box is empty

        An empty box may be created with the default constructor for example. Such a box is neutral when combining it with other boxes and renders empty boxes if used in box intersections and false in geometrical relationship tests. 
        """
    @overload
    def enlarge(self, d: int) -> Box:
        r"""
        @brief Enlarges the box by a certain amount on all sides.

        This is a convenience method which takes one values instead of two values. It will apply the given enlargement in both directions.
        This method has been introduced in version 0.28.

        @return A reference to this box.
        """
    @overload
    def enlarge(self, enlargement: Vector) -> Box:
        r"""
        @brief Enlarges the box by a certain amount.


        Enlarges the box by x and y value specified in the vector
        passed. Positive values with grow the box, negative ones
        will shrink the box. The result may be an empty box if the
        box disappears. The amount specifies the grow or shrink
        per edge. The width and height will change by twice the
        amount.
        Does not check for coordinate
        overflows.

        @param enlargement The grow or shrink amount in x and y direction

        @return A reference to this box.
        """
    @overload
    def enlarge(self, dx: int, dy: int) -> Box:
        r"""
        @brief Enlarges the box by a certain amount.


        This is a convenience method which takes two values instead of a Vector object.
        This method has been introduced in version 0.23.

        @return A reference to this box.
        """
    @overload
    def enlarged(self, d: int) -> Box:
        r"""
        @brief Enlarges the box by a certain amount on all sides.

        This is a convenience method which takes one values instead of two values. It will apply the given enlargement in both directions.
        This method has been introduced in version 0.28.

        @return The enlarged box.
        """
    @overload
    def enlarged(self, enlargement: Vector) -> Box:
        r"""
        @brief Returns the enlarged box.


        Enlarges the box by x and y value specified in the vector
        passed. Positive values with grow the box, negative ones
        will shrink the box. The result may be an empty box if the
        box disappears. The amount specifies the grow or shrink
        per edge. The width and height will change by twice the
        amount.
        Does not modify this box. Does not check for coordinate
        overflows.

        @param enlargement The grow or shrink amount in x and y direction

        @return The enlarged box.
        """
    @overload
    def enlarged(self, dx: int, dy: int) -> Box:
        r"""
        @brief Enlarges the box by a certain amount.


        This is a convenience method which takes two values instead of a Vector object.
        This method has been introduced in version 0.23.

        @return The enlarged box.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given box. This method enables boxes as hash keys.

        This method has been introduced in version 0.25.
        """
    def height(self) -> int:
        r"""
        @brief Gets the height of the box
        """
    def inside(self, box: Box) -> bool:
        r"""
        @brief Tests if this box is inside the argument box


        Returns true, if this box is inside the given box, i.e. the box intersection renders this box
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_point(self) -> bool:
        r"""
        @brief Returns true, if the box is a single point
        """
    @overload
    def move(self, distance: Vector) -> Box:
        r"""
        @brief Moves the box by a certain distance


        Moves the box by a given offset and returns the moved
        box. Does not check for coordinate overflows.

        @param distance The offset to move the box.

        @return A reference to this box.
        """
    @overload
    def move(self, dx: int, dy: int) -> Box:
        r"""
        @brief Moves the box by a certain distance


        This is a convenience method which takes two values instead of a Point object.
        This method has been introduced in version 0.23.

        @return A reference to this box.
        """
    @overload
    def moved(self, distance: Vector) -> Box:
        r"""
        @brief Returns the box moved by a certain distance


        Moves the box by a given offset and returns the moved
        box. Does not modify this box. Does not check for coordinate
        overflows.

        @param distance The offset to move the box.

        @return The moved box.
        """
    @overload
    def moved(self, dx: int, dy: int) -> Box:
        r"""
        @brief Moves the box by a certain distance


        This is a convenience method which takes two values instead of a Point object.
        This method has been introduced in version 0.23.

        @return The moved box.
        """
    def overlaps(self, box: Box) -> bool:
        r"""
        @brief Tests if this box overlaps the argument box


        Returns true, if the intersection box of this box with the argument box exists and has a non-vanishing area
        """
    def perimeter(self) -> int:
        r"""
        @brief Returns the perimeter of the box

        This method is equivalent to 2*(width+height). For empty boxes, this method returns 0.

        This method has been introduced in version 0.23.
        """
    def to_dtype(self, dbu: Optional[float] = ...) -> DBox:
        r"""
        @brief Converts the box to a floating-point coordinate box

        The database unit can be specified to translate the integer-coordinate box into a floating-point coordinate box in micron units. The database unit is basically a scaling factor.

        This method has been introduced in version 0.25.
        """
    def to_s(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing this box

        This string can be turned into a box again by using \from_s
        . If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def touches(self, box: Box) -> bool:
        r"""
        @brief Tests if this box touches the argument box


        Two boxes touch if they overlap or their boundaries share at least one common point. Touching is equivalent to a non-empty intersection ('!(b1 & b2).empty?').
        """
    @overload
    def transformed(self, t: CplxTrans) -> DBox:
        r"""
        @brief Returns the box transformed with the given complex transformation


        @param t The magnifying transformation to apply
        @return The transformed box (a DBox now)
        """
    @overload
    def transformed(self, t: ICplxTrans) -> Box:
        r"""
        @brief Transforms the box with the given complex transformation


        @param t The magnifying transformation to apply
        @return The transformed box (in this case an integer coordinate box)

        This method has been introduced in version 0.18.
        """
    @overload
    def transformed(self, t: Trans) -> Box:
        r"""
        @brief Returns the box transformed with the given simple transformation


        @param t The transformation to apply
        @return The transformed box
        """
    def width(self) -> int:
        r"""
        @brief Gets the width of the box
        """

class DBox:
    r"""
    @brief A box class with floating-point coordinates

    This object represents a box (a rectangular shape).

    The definition of the attributes is: p1 is the lower left point, p2 the 
    upper right one. If a box is constructed from two points (or four coordinates), the 
    coordinates are sorted accordingly.

    A box can be empty. An empty box represents no area
    (not even a point). Empty boxes behave neutral with respect to most operations. 
    Empty boxes return true on \empty?.

    A box can be a point or a single
    line. In this case, the area is zero but the box still
    can overlap other boxes for example and it is not empty.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    bottom: float
    r"""
    Getter:
    @brief Gets the bottom coordinate of the box

    Setter:
    @brief Sets the bottom coordinate of the box
    """
    left: float
    r"""
    Getter:
    @brief Gets the left coordinate of the box

    Setter:
    @brief Sets the left coordinate of the box
    """
    p1: DPoint
    r"""
    Getter:
    @brief Gets the lower left point of the box

    Setter:
    @brief Sets the lower left point of the box
    """
    p2: DPoint
    r"""
    Getter:
    @brief Gets the upper right point of the box

    Setter:
    @brief Sets the upper right point of the box
    """
    right: float
    r"""
    Getter:
    @brief Gets the right coordinate of the box

    Setter:
    @brief Sets the right coordinate of the box
    """
    top: float
    r"""
    Getter:
    @brief Gets the top coordinate of the box

    Setter:
    @brief Sets the top coordinate of the box
    """
    @classmethod
    def from_ibox(cls, box: Box) -> DBox:
        r"""
        @brief Creates a floating-point coordinate box from an integer coordinate box

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ibox'.
        """
    @classmethod
    def from_s(cls, s: str) -> DBox:
        r"""
        @brief Creates a box object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> DBox:
        r"""
        @brief Creates an empty (invalid) box

        Empty boxes don't modify a box when joined with it. The intersection between an empty and any other box is also an empty box. The width, height, p1 and p2 attributes of an empty box are undefined. Use \empty? to get a value indicating whether the box is empty.
        """
    @overload
    @classmethod
    def new(cls, arg0: float) -> DBox:
        r"""
        @brief Creates a square with the given dimensions centered around the origin

        Note that for integer-unit boxes, the dimension has to be an even number to avoid rounding.

        This convenience constructor has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, box: Box) -> DBox:
        r"""
        @brief Creates a floating-point coordinate box from an integer coordinate box

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ibox'.
        """
    @overload
    @classmethod
    def new(cls, arg0: float, arg1: float) -> DBox:
        r"""
        @brief Creates a rectangle with given width and height, centered around the origin

        Note that for integer-unit boxes, the dimensions have to be an even number to avoid rounding.

        This convenience constructor has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, lower_left: DPoint, upper_right: DPoint) -> DBox:
        r"""
        @brief Creates a box from two points


        Two points are given to create a new box. If the coordinates are not provided in the correct order (i.e. right < left), these are swapped.
        """
    @overload
    @classmethod
    def new(cls, left: float, bottom: float, right: float, top: float) -> DBox:
        r"""
        @brief Creates a box with four coordinates


        Four coordinates are given to create a new box. If the coordinates are not provided in the correct order (i.e. right < left), these are swapped.
        """
    @classmethod
    def world(cls) -> DBox:
        r"""
        @brief Gets the 'world' box
        The world box is the biggest box that can be represented. So it is basically 'all'. The world box behaves neutral on intersections for example. In other operations such as displacement or transformations, the world box may render unexpected results because of coordinate overflow.

        The world box can be used
        @ul
          @li for comparison ('==', '!=', '<') @/li
          @li in union and intersection ('+' and '&') @/li
          @li in relations (\contains?, \overlaps?, \touches?) @/li
          @li as 'all' argument in region queries @/li
        @/ul

        This method has been introduced in version 0.28.
        """
    @overload
    def __add__(self, box: DBox) -> DBox:
        r"""
        @brief Joins two boxes


        The + operator joins the first box with the one given as 
        the second argument. Joining constructs a box that encloses
        both boxes given. Empty boxes are neutral: they do not
        change another box when joining. Overwrites this box
        with the result.

        @param box The box to join with this box.

        @return The joined box
        """
    @overload
    def __add__(self, point: DPoint) -> DBox:
        r"""
        @brief Joins box with a point


        The + operator joins a point with the box. The resulting box will enclose both the original box and the point.

        @param point The point to join with this box.

        @return The box joined with the point
        """
    def __and__(self, box: DBox) -> DBox:
        r"""
        @brief Returns the intersection of this box with another box


        The intersection of two boxes is the largest
        box common to both boxes. The intersection may be 
        empty if both boxes to not touch. If the boxes do
        not overlap but touch the result may be a single
        line or point with an area of zero. Overwrites this box
        with the result.

        @param box The box to take the intersection with

        @return The intersection box
        """
    def __copy__(self) -> DBox:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DBox:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, box: object) -> bool:
        r"""
        @brief Returns true if this box is equal to the other box
        Returns true, if this box and the given box are equal 
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given box. This method enables boxes as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Creates an empty (invalid) box

        Empty boxes don't modify a box when joined with it. The intersection between an empty and any other box is also an empty box. The width, height, p1 and p2 attributes of an empty box are undefined. Use \empty? to get a value indicating whether the box is empty.
        """
    @overload
    def __init__(self, arg0: float) -> None:
        r"""
        @brief Creates a square with the given dimensions centered around the origin

        Note that for integer-unit boxes, the dimension has to be an even number to avoid rounding.

        This convenience constructor has been introduced in version 0.28.
        """
    @overload
    def __init__(self, box: Box) -> None:
        r"""
        @brief Creates a floating-point coordinate box from an integer coordinate box

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ibox'.
        """
    @overload
    def __init__(self, arg0: float, arg1: float) -> None:
        r"""
        @brief Creates a rectangle with given width and height, centered around the origin

        Note that for integer-unit boxes, the dimensions have to be an even number to avoid rounding.

        This convenience constructor has been introduced in version 0.28.
        """
    @overload
    def __init__(self, lower_left: DPoint, upper_right: DPoint) -> None:
        r"""
        @brief Creates a box from two points


        Two points are given to create a new box. If the coordinates are not provided in the correct order (i.e. right < left), these are swapped.
        """
    @overload
    def __init__(self, left: float, bottom: float, right: float, top: float) -> None:
        r"""
        @brief Creates a box with four coordinates


        Four coordinates are given to create a new box. If the coordinates are not provided in the correct order (i.e. right < left), these are swapped.
        """
    def __lt__(self, box: DBox) -> bool:
        r"""
        @brief Returns true if this box is 'less' than another box
        Returns true, if this box is 'less' with respect to first and second point (in this order)
        """
    @overload
    def __mul__(self, box: DBox) -> DBox:
        r"""
        @brief Returns the convolution product from this box with another box


        The * operator convolves the firstbox with the one given as 
        the second argument. The box resulting from "convolution" is the
        outer boundary of the union set formed by placing 
        the second box at every point of the first. In other words,
        the returned box of (p1,p2)*(q1,q2) is (p1+q1,p2+q2).

        @param box The box to convolve with this box.

        @return The convolved box
        """
    @overload
    def __mul__(self, scale_factor: float) -> DBox:
        r"""
        @brief Returns the scaled box


        The * operator scales the box with the given factor and returns the result.

        This method has been introduced in version 0.22.

        @param scale_factor The scaling factor

        @return The scaled box
        """
    def __ne__(self, box: object) -> bool:
        r"""
        @brief Returns true if this box is not equal to the other box
        Returns true, if this box and the given box are not equal 
        """
    @overload
    def __rmul__(self, box: DBox) -> DBox:
        r"""
        @brief Returns the convolution product from this box with another box


        The * operator convolves the firstbox with the one given as 
        the second argument. The box resulting from "convolution" is the
        outer boundary of the union set formed by placing 
        the second box at every point of the first. In other words,
        the returned box of (p1,p2)*(q1,q2) is (p1+q1,p2+q2).

        @param box The box to convolve with this box.

        @return The convolved box
        """
    @overload
    def __rmul__(self, scale_factor: float) -> DBox:
        r"""
        @brief Returns the scaled box


        The * operator scales the box with the given factor and returns the result.

        This method has been introduced in version 0.22.

        @param scale_factor The scaling factor

        @return The scaled box
        """
    def __str__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing this box

        This string can be turned into a box again by using \from_s
        . If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def area(self) -> float:
        r"""
        @brief Computes the box area

        Returns the box area or 0 if the box is empty
        """
    def assign(self, other: DBox) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> DBox:
        r"""
        @brief Returns the bounding box
        This method is provided for consistency of the shape API is returns the box itself.

        This method has been introduced in version 0.27.
        """
    def center(self) -> DPoint:
        r"""
        @brief Gets the center of the box
        """
    @overload
    def contains(self, point: DPoint) -> bool:
        r"""
        @brief Returns true if the box contains the given point


        Tests whether a point is inside the box.
        It also returns true if the point is exactly on the box contour.

        @param p The point to test against.

        @return true if the point is inside the box.
        """
    @overload
    def contains(self, x: float, y: float) -> bool:
        r"""
        @brief Returns true if the box contains the given point


        Tests whether a point (x, y) is inside the box.
        It also returns true if the point is exactly on the box contour.

        @return true if the point is inside the box.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DBox:
        r"""
        @brief Creates a copy of self
        """
    def empty(self) -> bool:
        r"""
        @brief Returns a value indicating whether the box is empty

        An empty box may be created with the default constructor for example. Such a box is neutral when combining it with other boxes and renders empty boxes if used in box intersections and false in geometrical relationship tests. 
        """
    @overload
    def enlarge(self, d: float) -> DBox:
        r"""
        @brief Enlarges the box by a certain amount on all sides.

        This is a convenience method which takes one values instead of two values. It will apply the given enlargement in both directions.
        This method has been introduced in version 0.28.

        @return A reference to this box.
        """
    @overload
    def enlarge(self, enlargement: DVector) -> DBox:
        r"""
        @brief Enlarges the box by a certain amount.


        Enlarges the box by x and y value specified in the vector
        passed. Positive values with grow the box, negative ones
        will shrink the box. The result may be an empty box if the
        box disappears. The amount specifies the grow or shrink
        per edge. The width and height will change by twice the
        amount.
        Does not check for coordinate
        overflows.

        @param enlargement The grow or shrink amount in x and y direction

        @return A reference to this box.
        """
    @overload
    def enlarge(self, dx: float, dy: float) -> DBox:
        r"""
        @brief Enlarges the box by a certain amount.


        This is a convenience method which takes two values instead of a Vector object.
        This method has been introduced in version 0.23.

        @return A reference to this box.
        """
    @overload
    def enlarged(self, d: float) -> DBox:
        r"""
        @brief Enlarges the box by a certain amount on all sides.

        This is a convenience method which takes one values instead of two values. It will apply the given enlargement in both directions.
        This method has been introduced in version 0.28.

        @return The enlarged box.
        """
    @overload
    def enlarged(self, enlargement: DVector) -> DBox:
        r"""
        @brief Returns the enlarged box.


        Enlarges the box by x and y value specified in the vector
        passed. Positive values with grow the box, negative ones
        will shrink the box. The result may be an empty box if the
        box disappears. The amount specifies the grow or shrink
        per edge. The width and height will change by twice the
        amount.
        Does not modify this box. Does not check for coordinate
        overflows.

        @param enlargement The grow or shrink amount in x and y direction

        @return The enlarged box.
        """
    @overload
    def enlarged(self, dx: float, dy: float) -> DBox:
        r"""
        @brief Enlarges the box by a certain amount.


        This is a convenience method which takes two values instead of a Vector object.
        This method has been introduced in version 0.23.

        @return The enlarged box.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given box. This method enables boxes as hash keys.

        This method has been introduced in version 0.25.
        """
    def height(self) -> float:
        r"""
        @brief Gets the height of the box
        """
    def inside(self, box: DBox) -> bool:
        r"""
        @brief Tests if this box is inside the argument box


        Returns true, if this box is inside the given box, i.e. the box intersection renders this box
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_point(self) -> bool:
        r"""
        @brief Returns true, if the box is a single point
        """
    @overload
    def move(self, distance: DVector) -> DBox:
        r"""
        @brief Moves the box by a certain distance


        Moves the box by a given offset and returns the moved
        box. Does not check for coordinate overflows.

        @param distance The offset to move the box.

        @return A reference to this box.
        """
    @overload
    def move(self, dx: float, dy: float) -> DBox:
        r"""
        @brief Moves the box by a certain distance


        This is a convenience method which takes two values instead of a Point object.
        This method has been introduced in version 0.23.

        @return A reference to this box.
        """
    @overload
    def moved(self, distance: DVector) -> DBox:
        r"""
        @brief Returns the box moved by a certain distance


        Moves the box by a given offset and returns the moved
        box. Does not modify this box. Does not check for coordinate
        overflows.

        @param distance The offset to move the box.

        @return The moved box.
        """
    @overload
    def moved(self, dx: float, dy: float) -> DBox:
        r"""
        @brief Moves the box by a certain distance


        This is a convenience method which takes two values instead of a Point object.
        This method has been introduced in version 0.23.

        @return The moved box.
        """
    def overlaps(self, box: DBox) -> bool:
        r"""
        @brief Tests if this box overlaps the argument box


        Returns true, if the intersection box of this box with the argument box exists and has a non-vanishing area
        """
    def perimeter(self) -> float:
        r"""
        @brief Returns the perimeter of the box

        This method is equivalent to 2*(width+height). For empty boxes, this method returns 0.

        This method has been introduced in version 0.23.
        """
    def to_itype(self, dbu: Optional[float] = ...) -> Box:
        r"""
        @brief Converts the box to an integer coordinate box

        The database unit can be specified to translate the floating-point coordinate box in micron units to an integer-coordinate box in database units. The boxes coordinates will be divided by the database unit.

        This method has been introduced in version 0.25.
        """
    def to_s(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing this box

        This string can be turned into a box again by using \from_s
        . If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def touches(self, box: DBox) -> bool:
        r"""
        @brief Tests if this box touches the argument box


        Two boxes touch if they overlap or their boundaries share at least one common point. Touching is equivalent to a non-empty intersection ('!(b1 & b2).empty?').
        """
    @overload
    def transformed(self, t: DCplxTrans) -> DBox:
        r"""
        @brief Returns the box transformed with the given complex transformation


        @param t The magnifying transformation to apply
        @return The transformed box (a DBox now)
        """
    @overload
    def transformed(self, t: DTrans) -> DBox:
        r"""
        @brief Returns the box transformed with the given simple transformation


        @param t The transformation to apply
        @return The transformed box
        """
    @overload
    def transformed(self, t: VCplxTrans) -> Box:
        r"""
        @brief Transforms the box with the given complex transformation


        @param t The magnifying transformation to apply
        @return The transformed box (in this case an integer coordinate box)

        This method has been introduced in version 0.25.
        """
    def width(self) -> float:
        r"""
        @brief Gets the width of the box
        """

class Cell:
    r"""
    @brief A cell

    A cell object consists of a set of shape containers (called layers),
    a set of child cell instances and auxiliary information such as
    the parent instance list.
    A cell is identified through an index given to the cell upon instantiation.
    Cell instances refer to single instances or array instances. Both are encapsulated in the
    same object, the \CellInstArray object. In the simple case, this object refers to a single instance.
    In the general case, this object may refer to a regular array of cell instances as well.

    Starting from version 0.16, the child_inst and erase_inst methods are no longer available since
    they were using index addressing which is no longer supported. Instead, instances are now addressed
    with the \Instance reference objects.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects like the Cell class.
    """
    ghost_cell: bool
    r"""
    Getter:
    @brief Returns a value indicating whether the cell is a "ghost cell"

    The ghost cell flag is used by the GDS reader for example to indicate that
    the cell is not located inside the file. Upon writing the reader can determine
    whether to write the cell or not.
    To satisfy the references inside the layout, a dummy cell is created in this case
    which has the "ghost cell" flag set to true.

    This method has been introduced in version 0.20.

    Setter:
    @brief Sets the "ghost cell" flag

    See \is_ghost_cell? for a description of this property.

    This method has been introduced in version 0.20.
    """
    name: str
    r"""
    Getter:
    @brief Gets the cell's name

    This may be an internal name for proxy cells. See \basic_name for the formal name (PCell name or library cell name).

    This method has been introduced in version 0.22.

    Setter:
    @brief Renames the cell
    Renaming a cell may cause name clashes, i.e. the name may be identical to the name
    of another cell. This does not have any immediate effect, but the cell needs to be renamed, for example when writing the layout to a GDS file.

    This method has been introduced in version 0.22.
    """
    prop_id: int
    r"""
    Getter:
    @brief Gets the properties ID associated with the cell

    This method has been introduced in version 0.23.
    Setter:
    @brief Sets the properties ID associated with the cell
    This method is provided, if a properties ID has been derived already. Usually it's more convenient to use \delete_property, \set_property or \property.

    This method has been introduced in version 0.23.
    """
    @classmethod
    def new(cls) -> Cell:
        r"""
        @brief Creates a new object of this class
        """
    def __copy__(self) -> Cell:
        r"""
        @brief Creates a copy of the cell

        This method will create a copy of the cell. The new cell will be member of the same layout the original cell was member of. The copy will inherit all shapes and instances, but get a different cell_index and a modified name as duplicate cell names are not allowed in the same layout.

        This method has been introduced in version 0.27.
        """
    def __deepcopy__(self) -> Cell:
        r"""
        @brief Creates a copy of the cell

        This method will create a copy of the cell. The new cell will be member of the same layout the original cell was member of. The copy will inherit all shapes and instances, but get a different cell_index and a modified name as duplicate cell names are not allowed in the same layout.

        This method has been introduced in version 0.27.
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def basic_name(self) -> str:
        r"""
        @brief Returns the name of the library or PCell or the real name of the cell
        For non-proxy cells (see \is_proxy?), this method simply returns the cell name.
        For proxy cells, this method returns the PCells definition name or the library
        cell name. This name may differ from the actual cell's name because to ensure
        that cell names are unique, KLayout may assign different names to the actual 
        cell compared to the source cell.

        This method has been introduced in version 0.22.
        """
    @overload
    def bbox(self) -> Box:
        r"""
        @brief Gets the bounding box of the cell

        @return The bounding box of the cell

        The bounding box is computed over all layers. To compute the bounding box over single layers, use \bbox with a layer index argument.
        """
    @overload
    def bbox(self, layer_index: int) -> Box:
        r"""
        @brief Gets the per-layer bounding box of the cell

        @return The bounding box of the cell considering only the given layer

        The bounding box is the box enclosing all shapes on the given layer.

        'bbox' is the preferred synonym since version 0.28.
        """
    def bbox_per_layer(self, layer_index: int) -> Box:
        r"""
        @brief Gets the per-layer bounding box of the cell

        @return The bounding box of the cell considering only the given layer

        The bounding box is the box enclosing all shapes on the given layer.

        'bbox' is the preferred synonym since version 0.28.
        """
    def begin_instances_rec(self) -> RecursiveInstanceIterator:
        r"""
        @brief Delivers a recursive instance iterator for the instances below the cell
        @return A suitable iterator

        For details see the description of the \RecursiveInstanceIterator class.

        This method has been added in version 0.27.
        """
    @overload
    def begin_instances_rec_overlapping(self, region: Box) -> RecursiveInstanceIterator:
        r"""
        @brief Delivers a recursive instance iterator for the instances below the cell using a region search
        @param region The search region
        @return A suitable iterator

        For details see the description of the \RecursiveInstanceIterator class.
        This version gives an iterator delivering instances whose bounding box overlaps the given region.

        This method has been added in version 0.27.
        """
    @overload
    def begin_instances_rec_overlapping(self, region: DBox) -> RecursiveInstanceIterator:
        r"""
        @brief Delivers a recursive instance iterator for the instances below the cell using a region search, with the region given in micrometer units
        @param region The search region as \DBox object in micrometer units
        @return A suitable iterator

        For details see the description of the \RecursiveInstanceIterator class.
        This version gives an iterator delivering instances whose bounding box overlaps the given region.

        This variant has been added in version 0.27.
        """
    @overload
    def begin_instances_rec_touching(self, region: Box) -> RecursiveInstanceIterator:
        r"""
        @brief Delivers a recursive instance iterator for the instances below the cell
        @param region The search region
        @return A suitable iterator

        For details see the description of the \RecursiveInstanceIterator class.
        This version gives an iterator delivering instances whose bounding box touches the given region.

        This method has been added in version 0.27.
        """
    @overload
    def begin_instances_rec_touching(self, region: DBox) -> RecursiveInstanceIterator:
        r"""
        @brief Delivers a recursive instance iterator for the instances below the cell using a region search, with the region given in micrometer units
        @param region The search region as \DBox object in micrometer units
        @return A suitable iterator

        For details see the description of the \RecursiveInstanceIterator class.
        This version gives an iterator delivering instances whose bounding box touches the given region.

        This variant has been added in version 0.27.
        """
    def begin_shapes_rec(self, layer: int) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the cell on the given layer
        @param layer The layer from which to get the shapes
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.

        This method has been added in version 0.23.
        """
    @overload
    def begin_shapes_rec_overlapping(self, layer: int, region: Box) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the cell on the given layer using a region search
        @param layer The layer from which to get the shapes
        @param region The search region
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.
        This version gives an iterator delivering shapes whose bounding box overlaps the given region.

        This method has been added in version 0.23.
        """
    @overload
    def begin_shapes_rec_overlapping(self, layer: int, region: DBox) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the cell on the given layer using a region search, with the region given in micrometer units
        @param layer The layer from which to get the shapes
        @param region The search region as \DBox object in micrometer units
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.
        This version gives an iterator delivering shapes whose bounding box overlaps the given region.

        This variant has been added in version 0.25.
        """
    @overload
    def begin_shapes_rec_touching(self, layer: int, region: Box) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the cell on the given layer using a region search
        @param layer The layer from which to get the shapes
        @param region The search region
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.
        This version gives an iterator delivering shapes whose bounding box touches the given region.

        This method has been added in version 0.23.
        """
    @overload
    def begin_shapes_rec_touching(self, layer: int, region: DBox) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the cell on the given layer using a region search, with the region given in micrometer units
        @param layer The layer from which to get the shapes
        @param region The search region as \DBox object in micrometer units
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.
        This version gives an iterator delivering shapes whose bounding box touches the given region.

        This variant has been added in version 0.25.
        """
    def called_cells(self) -> List[int]:
        r"""
        @brief Gets a list of all called cells

        This method determines all cells which are called either directly or indirectly by the cell.
        It returns an array of cell indexes. Use the 'cell' method of \Layout to retrieve the corresponding Cell object.

        This method has been introduced in version 0.19.

        @return A list of cell indices.
        """
    def caller_cells(self) -> List[int]:
        r"""
        @brief Gets a list of all caller cells

        This method determines all cells which call this cell either directly or indirectly.
        It returns an array of cell indexes. Use the 'cell' method of \Layout to retrieve the corresponding Cell object.

        This method has been introduced in version 0.19.

        @return A list of cell indices.
        """
    def cell_index(self) -> int:
        r"""
        @brief Gets the cell index

        @return The cell index of the cell
        """
    def change_pcell_parameter(self, instance: Instance, name: str, value: Any) -> Instance:
        r"""
        @brief Changes a single parameter for an individual PCell instance given by name
        @return The new instance (the old may be invalid)
        This will set the PCell parameter named 'name' to the given value for the instance addressed by 'instance'. If no parameter with that name exists, the method will do nothing.

        This method has been introduced in version 0.23.
        """
    @overload
    def change_pcell_parameters(self, instance: Instance, parameters: Sequence[Any]) -> Instance:
        r"""
        @brief Changes the parameters for an individual PCell instance
        @return The new instance (the old may be invalid)
        If necessary, this method creates a new variant and replaces the given instance
        by an instance of this variant.

        The parameters are given in the order the parameters are declared. Use \pcell_declaration on the instance to get the PCell declaration object of the cell. That PCellDeclaration object delivers the parameter declaration with it's 'get_parameters' method.
        Each parameter in the variant list passed to the second list of values corresponds to one parameter declaration.

        There is a more convenient method (\change_pcell_parameter) that changes a single parameter by name.

        This method has been introduced in version 0.22.
        """
    @overload
    def change_pcell_parameters(self, instance: Instance, dict: Dict[str, Any]) -> Instance:
        r"""
        @brief Changes the given parameter for an individual PCell instance
        @return The new instance (the old may be invalid)
        This version receives a dictionary of names and values. It will change the parameters given by the names to the values given by the values of the dictionary. The functionality is similar to the same function with an array, but more convenient to use.
        Values with unknown names are ignored.

        This method has been introduced in version 0.24.
        """
    def child_cells(self) -> int:
        r"""
        @brief Gets the number of child cells

        The number of child cells (not child instances!) is returned.
        CAUTION: this method is SLOW, in particular if many instances are present.
        """
    def child_instances(self) -> int:
        r"""
        @brief Gets the number of child instances

        @return Returns the number of cell instances
        """
    @overload
    def clear(self) -> None:
        r"""
        @brief Clears the cell (deletes shapes and instances)
        This method has been introduced in version 0.23.
        """
    @overload
    def clear(self, layer_index: int) -> None:
        r"""
        @brief Clears the shapes on the given layer
        """
    def clear_insts(self) -> None:
        r"""
        @brief Clears the instance list
        """
    def clear_shapes(self) -> None:
        r"""
        @brief Clears all shapes in the cell
        """
    @overload
    def copy(self, src: int, dest: int) -> None:
        r"""
        @brief Copies the shapes from the source to the target layer

        The destination layer is not overwritten. Instead, the shapes are added to the shapes of the destination layer.
        If source are target layer are identical, this method does nothing.
        This method will copy shapes within the cell. To copy shapes from another cell to this cell, use the copy method with the cell parameter.

        This method has been introduced in version 0.19.

        @param src The layer index of the source layer
        @param dest The layer index of the destination layer
        """
    @overload
    def copy(self, src_cell: Cell, src_layer: int, dest: int) -> None:
        r"""
        @brief Copies shapes from another cell to the target layer in this cell

        This method will copy all shapes on layer 'src_layer' of cell 'src_cell' to the layer 'dest' of this cell.
        The destination layer is not overwritten. Instead, the shapes are added to the shapes of the destination layer.
        If the source cell lives in a layout with a different database unit than that current cell is in, the shapes will be transformed accordingly. The same way, shape properties are transformed as well. Note that the shape transformation may require rounding to smaller coordinates. This may result in a slight distortion of the original shapes, in particular when transforming into a layout with a bigger database unit.
        @param src_cell The cell where to take the shapes from
        @param src_layer The layer index of the layer from which to take the shapes
        @param dest The layer index of the destination layer
        """
    def copy_instances(self, source_cell: Cell) -> None:
        r"""
        @brief Copies the instances of child cells in the source cell to this cell
        @param source_cell The cell where the instances are copied from
        The source cell must reside in the same layout than this cell. The instances of child cells inside the source cell are copied to this cell. No new cells are created, just new instances are created to already existing cells in the target cell.

        The instances will be added to any existing instances in the cell.

        More elaborate methods of copying hierarchy trees between layouts or duplicating trees are provided through the \copy_tree_shapes (in cooperation with the \CellMapping class) or \copy_tree methods.

        This method has been added in version 0.23.
        """
    @overload
    def copy_shapes(self, source_cell: Cell) -> None:
        r"""
        @brief Copies the shapes from the given cell into this cell
        @param source_cell The cell from where to copy shapes
        All shapes are copied from the source cell to this cell. Instances are not copied.

        The source cell can reside in a different layout. In this case, the shapes are copied over from the other layout into this layout. Database unit conversion is done automatically if the database units differ between the layouts. Note that this may lead to grid snapping effects if the database unit of the target layout is not an integer fraction of the source layout.

        If source and target layout are different, the layers of the source and target layout are identified by their layer/datatype number or name (if no layer/datatype is present).
        The shapes will be added to any shapes already in the cell.

        This method has been added in version 0.23.
        """
    @overload
    def copy_shapes(self, source_cell: Cell, layer_mapping: LayerMapping) -> None:
        r"""
        @brief Copies the shapes from the given cell into this cell
        @param source_cell The cell from where to copy shapes
        @param layer_mapping A \LayerMapping object that specifies which layers are copied and where
        All shapes on layers specified in the layer mapping object are copied from the source cell to this cell. Instances are not copied.
        The target layer is taken from the mapping table.

        The shapes will be added to any shapes already in the cell.

        This method has been added in version 0.23.
        """
    def copy_tree(self, source_cell: Cell) -> List[int]:
        r"""
        @brief Copies the cell tree of the given cell into this cell
        @param source_cell The cell from where to copy the cell tree
        @return A list of indexes of newly created cells
        The complete cell tree of the source cell is copied to the target cell plus all shapes in that tree are copied as well. This method will basically duplicate the cell tree of the source cell.

        The source cell may reside in a separate layout. This method therefore provides a way to copy over complete cell trees from one layout to another.

        The shapes and instances will be added to any shapes or instances already in the cell.

        This method has been added in version 0.23.
        """
    @overload
    def copy_tree_shapes(self, source_cell: Cell, cell_mapping: CellMapping) -> None:
        r"""
        @brief Copies the shapes from the given cell and the cell tree below into this cell or subcells of this cell
        @param source_cell The starting cell from where to copy shapes
        @param cell_mapping The cell mapping object that determines how cells are identified between source and target layout

        This method is provided if source and target cell reside in different layouts. If will copy the shapes from all cells below the given source cell, but use a cell mapping object that provides a specification how cells are identified between the layouts. Cells in the source tree, for which no mapping is provided, will be flattened - their shapes will be propagated into parent cells for which a mapping is provided.

        The cell mapping object provides various methods to map cell trees between layouts. See the \CellMapping class for details about the mapping methods available. The cell mapping object is also responsible for creating a proper hierarchy of cells in the target layout if that is required.

        Layers are identified between the layouts by the layer/datatype number of name if no layer/datatype number is present.

        The shapes copied will be added to any shapes already in the cells.

        This method has been added in version 0.23.
        """
    @overload
    def copy_tree_shapes(self, source_cell: Cell, cell_mapping: CellMapping, layer_mapping: LayerMapping) -> None:
        r"""
        @brief Copies the shapes from the given cell and the cell tree below into this cell or subcells of this cell with layer mapping
        @param source_cell The cell from where to copy shapes and instances
        @param cell_mapping The cell mapping object that determines how cells are identified between source and target layout

        This method is provided if source and target cell reside in different layouts. If will copy the shapes from all cells below the given source cell, but use a cell mapping object that provides a specification how cells are identified between the layouts. Cells in the source tree, for which no mapping is provided, will be flattened - their shapes will be propagated into parent cells for which a mapping is provided.

        The cell mapping object provides various methods to map cell trees between layouts. See the \CellMapping class for details about the mapping methods available. The cell mapping object is also responsible for creating a proper hierarchy of cells in the target layout if that is required.

        In addition, the layer mapping object can be specified which maps source to target layers. This feature can be used to restrict the copy operation to a subset of layers or to convert shapes to different layers in that step.

        The shapes copied will be added to any shapes already in the cells.

        This method has been added in version 0.23.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    @overload
    def dbbox(self) -> DBox:
        r"""
        @brief Gets the bounding box of the cell in micrometer units

        @return The bounding box of the cell

        The bounding box is computed over all layers. To compute the bounding box over single layers, use \dbbox with a layer index argument.

        This method has been introduced in version 0.25.
        """
    @overload
    def dbbox(self, layer_index: int) -> DBox:
        r"""
        @brief Gets the per-layer bounding box of the cell in micrometer units

        @return The bounding box of the cell considering only the given layer

        The bounding box is the box enclosing all shapes on the given layer.

        This method has been introduced in version 0.25. 'dbbox' is the preferred synonym since version 0.28.
        """
    def dbbox_per_layer(self, layer_index: int) -> DBox:
        r"""
        @brief Gets the per-layer bounding box of the cell in micrometer units

        @return The bounding box of the cell considering only the given layer

        The bounding box is the box enclosing all shapes on the given layer.

        This method has been introduced in version 0.25. 'dbbox' is the preferred synonym since version 0.28.
        """
    def delete(self) -> None:
        r"""
        @brief Deletes this cell 

        This deletes the cell but not the sub cells of the cell.
        These subcells will likely become new top cells unless they are used
        otherwise.
        All instances of this cell are deleted as well.
        Hint: to delete multiple cells, use "delete_cells" which is 
        far more efficient in this case.

        After the cell has been deleted, the Cell object becomes invalid. Do not access methods or attributes of this object after deleting the cell.

        This method has been introduced in version 0.23.
        """
    def delete_property(self, key: Any) -> None:
        r"""
        @brief Deletes the user property with the given key
        This method is a convenience method that deletes the property with the given key. It does nothing if no property with that key exists. Using that method is more convenient than creating a new property set with a new ID and assigning that properties ID.
        This method may change the properties ID.

        This method has been introduced in version 0.23.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def display_title(self) -> str:
        r"""
        @brief Returns a nice looking name for display purposes

        For example, this name include PCell parameters for PCell proxy cells.

        This method has been introduced in version 0.22.
        """
    def dump_mem_statistics(self, detailed: Optional[bool] = ...) -> None:
        r"""
        @hide
        """
    def dup(self) -> Cell:
        r"""
        @brief Creates a copy of the cell

        This method will create a copy of the cell. The new cell will be member of the same layout the original cell was member of. The copy will inherit all shapes and instances, but get a different cell_index and a modified name as duplicate cell names are not allowed in the same layout.

        This method has been introduced in version 0.27.
        """
    def each_child_cell(self) -> Iterator[int]:
        r"""
        @brief Iterates over all child cells

        This iterator will report the child cell indices, not every instance.
        """
    def each_inst(self) -> Iterator[Instance]:
        r"""
        @brief Iterates over all child instances (which may actually be instance arrays)

        Starting with version 0.15, this iterator delivers \Instance objects rather than \CellInstArray objects.
        """
    @overload
    def each_overlapping_inst(self, b: Box) -> Iterator[Instance]:
        r"""
        @brief Gets the instances overlapping the given rectangle

        This will iterate over all child cell
        instances overlapping with the given rectangle b. 

        @param b The region to iterate over

        Starting with version 0.15, this iterator delivers \Instance objects rather than \CellInstArray objects.
        """
    @overload
    def each_overlapping_inst(self, b: DBox) -> Iterator[Instance]:
        r"""
        @brief Gets the instances overlapping the given rectangle, with the rectangle in micrometer units

        This will iterate over all child cell
        instances overlapping with the given rectangle b. This method is identical to the \each_overlapping_inst version that takes a \Box object, but instead of taking database unit coordinates in will take a micrometer unit \DBox object.

        @param b The region to iterate over

        This variant has been introduced in version 0.25.
        """
    @overload
    def each_overlapping_shape(self, layer_index: int, box: Box) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer that overlap the given box

        @param box The box by which to query the shapes
        @param layer_index The layer on which to run the query

        This call is equivalent to each_overlapping_shape(layer_index,box,RBA::Shapes::SAll).
        This convenience method has been introduced in version 0.16.
        """
    @overload
    def each_overlapping_shape(self, layer_index: int, box: DBox) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer that overlap the given box, with the box given in micrometer units

        @param box The box by which to query the shapes as a \DBox object in micrometer units
        @param layer_index The layer on which to run the query

        This call is equivalent to each_overlapping_shape(layer_index,box,RBA::Shapes::SAll).
        This convenience method has been introduced in version 0.16.
        """
    @overload
    def each_overlapping_shape(self, layer_index: int, box: Box, flags: int) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer that overlap the given box

        @param flags An "or"-ed combination of the S.. constants of the \Shapes class
        @param box The box by which to query the shapes
        @param layer_index The layer on which to run the query
        """
    @overload
    def each_overlapping_shape(self, layer_index: int, box: DBox, flags: int) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer that overlap the given box, with the box given in micrometer units

        @param flags An "or"-ed combination of the S.. constants of the \Shapes class
        @param box The box by which to query the shapes as a \DBox object in micrometer units
        @param layer_index The layer on which to run the query
        """
    def each_parent_cell(self) -> Iterator[int]:
        r"""
        @brief Iterates over all parent cells

        This iterator will iterate over the parent cells, just returning their
        cell index.
        """
    def each_parent_inst(self) -> Iterator[ParentInstArray]:
        r"""
        @brief Iterates over the parent instance list (which may actually be instance arrays)

        The parent instances are basically inversions of the instances. Using parent instances it is possible to determine how a specific cell is called from where.
        """
    @overload
    def each_shape(self, layer_index: int) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer

        @param layer_index The layer on which to run the query

        This call is equivalent to each_shape(layer_index,RBA::Shapes::SAll).
        This convenience method has been introduced in version 0.16.
        """
    @overload
    def each_shape(self, layer_index: int, flags: int) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer

        @param flags An "or"-ed combination of the S.. constants of the \Shapes class
        @param layer_index The layer on which to run the query

        This iterator is equivalent to 'shapes(layer).each'.
        """
    @overload
    def each_touching_inst(self, b: Box) -> Iterator[Instance]:
        r"""
        @brief Gets the instances touching the given rectangle

        This will iterate over all child cell
        instances overlapping with the given rectangle b. 

        @param b The region to iterate over

        Starting with version 0.15, this iterator delivers \Instance objects rather than \CellInstArray objects.
        """
    @overload
    def each_touching_inst(self, b: DBox) -> Iterator[Instance]:
        r"""
        @brief Gets the instances touching the given rectangle, with the rectangle in micrometer units

        This will iterate over all child cell
        instances touching the given rectangle b. This method is identical to the \each_touching_inst version that takes a \Box object, but instead of taking database unit coordinates in will take a micrometer unit \DBox object.

        @param b The region to iterate over

        This variant has been introduced in version 0.25.
        """
    @overload
    def each_touching_shape(self, layer_index: int, box: Box) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer that touch the given box

        @param box The box by which to query the shapes
        @param layer_index The layer on which to run the query

        This call is equivalent to each_touching_shape(layer_index,box,RBA::Shapes::SAll).
        This convenience method has been introduced in version 0.16.
        """
    @overload
    def each_touching_shape(self, layer_index: int, box: DBox) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer that touch the given box, with the box given in micrometer units

        @param box The box by which to query the shapes as a \DBox object in micrometer units
        @param layer_index The layer on which to run the query

        This call is equivalent to each_touching_shape(layer_index,box,RBA::Shapes::SAll).
        This convenience method has been introduced in version 0.16.
        """
    @overload
    def each_touching_shape(self, layer_index: int, box: Box, flags: int) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer that touch the given box

        @param flags An "or"-ed combination of the S.. constants of the \Shapes class
        @param box The box by which to query the shapes
        @param layer_index The layer on which to run the query
        """
    @overload
    def each_touching_shape(self, layer_index: int, box: DBox, flags: int) -> Iterator[Shape]:
        r"""
        @brief Iterates over all shapes of a given layer that touch the given box, with the box given in micrometer units

        @param flags An "or"-ed combination of the S.. constants of the \Shapes class
        @param box The box by which to query the shapes as a \DBox object in micrometer units
        @param layer_index The layer on which to run the query
        """
    def erase(self, inst: Instance) -> None:
        r"""
        @brief Erases the instance given by the Instance object

        This method has been introduced in version 0.16. It can only be used in editable mode.
        """
    @overload
    def fill_region(self, region: Region, fill_cell_index: int, fc_box: Box, origin: Optional[Point] = ..., remaining_parts: Optional[Region] = ..., fill_margin: Optional[Vector] = ..., remaining_polygons: Optional[Region] = ..., glue_box: Optional[Box] = ...) -> None:
        r"""
        @brief Fills the given region with cells of the given type (extended version)
        @param region The region to fill
        @param fill_cell_index The fill cell to place
        @param fc_box The fill cell's footprint
        @param origin The global origin of the fill pattern or nil to allow local (per-polygon) optimization
        @param remaining_parts See explanation below
        @param fill_margin See explanation below
        @param remaining_polygons See explanation below
        @param glue_box Guarantees fill cell compatibility to neighbor regions in enhanced mode

        This method creates a regular pattern of fill cells to cover the interior of the given region as far as possible. This process is also known as tiling. This implementation supports rectangular (not necessarily square) tile cells. The tile cell's footprint is given by the fc_box parameter and the cells will be arranged with their footprints forming a seamless array.

        The algorithm supports a global fill raster as well as local (per-polygon) origin optimization. In the latter case the origin of the regular raster is optimized per individual polygon of the fill region. To enable optimization, pass 'nil' to the 'origin' argument.

        The implementation will basically try to find a repetition pattern of the tile cell's footprint and produce instances which fit entirely into the fill region.

        There is also a version available which offers skew step vectors as a generalization of the orthogonal ones.

        If the 'remaining_parts' argument is non-nil, the corresponding region will receive the parts of the polygons which are not covered by tiles. Basically the tiles are subtracted from the original polygons. A margin can be specified which is applied separately in x and y direction before the subtraction is done ('fill_margin' parameter).

        If the 'remaining_polygons' argument is non-nil, the corresponding region will receive all polygons from the input region which could not be filled and where there is no chance of filling because not a single tile will fit into them.

        'remaining_parts' and 'remaining_polygons' can be identical with the input. In that case the input will be overwritten with the respective output. Otherwise, the respective polygons are added to these regions.

        This allows setting up a more elaborate fill scheme using multiple iterations and local origin-optimization ('origin' is nil):

        @code
        r = ...        # region to fill
        c = ...        # cell in which to produce the fill cells
        fc_index = ... # fill cell index
        fc_box = ...   # fill cell footprint

        fill_margin = RBA::Point::new(0, 0)   # x/y distance between tile cells with different origin

        # Iteration: fill a region and fill the remaining parts as long as there is anything left.
        # Polygons not worth being considered further are dropped (last argument is nil).
        while !r.is_empty?
          c.fill_region(r, fc_index, fc_box, nil, r, fill_margin, nil)
        end
        @/code

        The glue box parameter supports fill cell array compatibility with neighboring regions. This is specifically useful when putting the fill_cell method into a tiling processor. Fill cell array compatibility means that the fill cell array continues over tile boundaries. This is easy with an origin: you can chose the origin identically over all tiles which is sufficient to guarantee fill cell array compatibility across the tiles. However there is no freedom of choice of the origin then and fill cell placement may not be optimal. To enable the origin for the tile boundary only, a glue box can given. The origin will then be used only when the polygons to fill not entirely inside and not at the border of the glue box. Hence, while a certain degree of freedom is present for the placement of fill cells inside the glue box, the fill cells are guaranteed to be placed at the raster implied by origin at the glue box border and beyond. To ensure fill cell compatibility inside the tiling processor, it is sufficient to use the tile box as the glue box.

        This method has been introduced in version 0.23 and enhanced in version 0.27.
        """
    @overload
    def fill_region(self, region: Region, fill_cell_index: int, fc_bbox: Box, row_step: Vector, column_step: Vector, origin: Optional[Point] = ..., remaining_parts: Optional[Region] = ..., fill_margin: Optional[Vector] = ..., remaining_polygons: Optional[Region] = ..., glue_box: Optional[Box] = ...) -> None:
        r"""
        @brief Fills the given region with cells of the given type (skew step version)
        @param region The region to fill
        @param fill_cell_index The fill cell to place
        @param fc_bbox The fill cell's box to place
        @param row_step The 'rows' step vector
        @param column_step The 'columns' step vector
        @param origin The global origin of the fill pattern or nil to allow local (per-polygon) optimization
        @param remaining_parts See explanation in other version
        @param fill_margin See explanation in other version
        @param remaining_polygons See explanation in other version

        This version is similar to the version providing an orthogonal fill, but it offers more generic stepping of the fill cell.
        The step pattern is defined by an origin and two vectors (row_step and column_step) which span the axes of the fill cell pattern.

        The fill box and the step vectors are decoupled which means the fill box can be larger or smaller than the step pitch - it can be overlapping and there can be space between the fill box instances. Fill boxes are placed where they fit entirely into a polygon of the region. The fill boxes lower left corner is the reference for the fill pattern and aligns with the origin if given.

        This variant has been introduced in version 0.27.
        """
    def fill_region_multi(self, region: Region, fill_cell_index: int, fc_bbox: Box, row_step: Vector, column_step: Vector, fill_margin: Optional[Vector] = ..., remaining_polygons: Optional[Region] = ..., glue_box: Optional[Box] = ...) -> None:
        r"""
        @brief Fills the given region with cells of the given type in enhanced mode with iterations
        This version operates like \fill_region, but repeats the fill generation until no further fill cells can be placed. As the fill pattern origin changes between the iterations, narrow regions can be filled which cannot with a fixed fill pattern origin. The \fill_margin parameter is important as it controls the distance between fill cells with a different origin and therefore introduces a safety distance between pitch-incompatible arrays.

        The origin is ignored unless a glue box is given. See \fill_region for a description of this concept.

        This method has been introduced in version 0.27.
        """
    @overload
    def flatten(self, prune: bool) -> None:
        r"""
        @brief Flattens the given cell

        This method propagates all shapes from the hierarchy below into the given cell.
        It also removes the instances of the cells from which the shapes came from, but does not remove the cells themselves if prune is set to false.
        If prune is set to true, these cells are removed if not used otherwise.

        A version of this method exists which allows one to specify the number of hierarchy levels to which subcells are considered.

        @param prune Set to true to remove orphan cells.

        This method has been introduced in version 0.23.
        """
    @overload
    def flatten(self, levels: int, prune: bool) -> None:
        r"""
        @brief Flattens the given cell

        This method propagates all shapes from the specified number of hierarchy levels below into the given cell.
        It also removes the instances of the cells from which the shapes came from, but does not remove the cells themselves if prune is set to false.
        If prune is set to true, these cells are removed if not used otherwise.

        @param levels The number of hierarchy levels to flatten (-1: all, 0: none, 1: one level etc.)
        @param prune Set to true to remove orphan cells.

        This method has been introduced in version 0.23.
        """
    def has_prop_id(self) -> bool:
        r"""
        @brief Returns true, if the cell has user properties

        This method has been introduced in version 0.23.
        """
    def hierarchy_levels(self) -> int:
        r"""
        @brief Returns the number of hierarchy levels below

        This method returns the number of call levels below the current cell. If there are no child cells, this method will return 0, if there are only direct children, it will return 1.

        CAUTION: this method may be expensive!
        """
    @overload
    def insert(self, cell_inst_array: CellInstArray) -> Instance:
        r"""
        @brief Inserts a cell instance (array)
        @return An Instance object representing the new instance
        With version 0.16, this method returns an Instance object that represents the new instance.
        It's use is discouraged in readonly mode, since it invalidates other Instance references.
        """
    @overload
    def insert(self, cell_inst_array: DCellInstArray) -> Instance:
        r"""
        @brief Inserts a cell instance (array) given in micron units
        @return An Instance object representing the new instance
        This method inserts an instance array, similar to \insert with a \CellInstArray parameter. But in this version, the argument is a cell instance array given in micrometer units. It is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    @overload
    def insert(self, inst: Instance) -> Instance:
        r"""
        @brief Inserts a cell instance given by another reference
        @return An Instance object representing the new instance
        This method allows one to copy instances taken from a reference (an \Instance object).
        This method is not suited to inserting instances from other Layouts into this cell. For this purpose, the hierarchical copy methods of \Layout have to be used.

        It has been added in version 0.16.
        """
    @overload
    def insert(self, cell_inst_array: CellInstArray, property_id: int) -> Instance:
        r"""
        @brief Inserts a cell instance (array) with properties
        @return An \Instance object representing the new instance
        The property Id must be obtained from the \Layout object's property_id method which associates a property set with a property Id.
        With version 0.16, this method returns an Instance object that represents the new instance.
        It's use is discouraged in readonly mode, since it invalidates other Instance references.
        """
    @overload
    def insert(self, cell_inst_array: DCellInstArray, property_id: int) -> Instance:
        r"""
        @brief Inserts a cell instance (array) given in micron units with properties
        @return An Instance object representing the new instance
        This method inserts an instance array, similar to \insert with a \CellInstArray parameter and a property set ID. But in this version, the argument is a cell instance array given in micrometer units. It is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_empty(self) -> bool:
        r"""
        @brief Returns a value indicating whether the cell is empty

        An empty cell is a cell not containing instances nor any shapes.

        This method has been introduced in version 0.20.
        """
    def is_ghost_cell(self) -> bool:
        r"""
        @brief Returns a value indicating whether the cell is a "ghost cell"

        The ghost cell flag is used by the GDS reader for example to indicate that
        the cell is not located inside the file. Upon writing the reader can determine
        whether to write the cell or not.
        To satisfy the references inside the layout, a dummy cell is created in this case
        which has the "ghost cell" flag set to true.

        This method has been introduced in version 0.20.
        """
    def is_leaf(self) -> bool:
        r"""
        @brief Gets a value indicating whether the cell is a leaf cell

        A cell is a leaf cell if there are no child instantiations.
        """
    def is_library_cell(self) -> bool:
        r"""
        @brief Returns true, if the cell is a proxy cell pointing to a library cell
        If the cell is imported from some library, this attribute returns true.
        Please note, that this attribute can combine with \is_pcell? for PCells imported from
        a library.

        This method has been introduced in version 0.22.
        """
    @overload
    def is_pcell_variant(self) -> bool:
        r"""
        @brief Returns true, if this cell is a pcell variant
        this method returns true, if this cell represents a pcell with a distinct
        set of parameters (a PCell proxy). This also is true, if the PCell is imported from a library.

        Technically, PCells imported from a library are library proxies which are 
        pointing to PCell variant proxies. This scheme can even proceed over multiple
        indirections, i.e. a library using PCells from another library.

        This method has been introduced in version 0.22.
        """
    @overload
    def is_pcell_variant(self, instance: Instance) -> bool:
        r"""
        @brief Returns true, if this instance is a PCell variant
        This method returns true, if this instance represents a PCell with a distinct
        set of parameters. This method also returns true, if it is a PCell imported from a library.

        This method has been introduced in version 0.22.
        """
    def is_proxy(self) -> bool:
        r"""
        @brief Returns true, if the cell presents some external entity   
        A cell may represent some data which is imported from some other source, i.e.
        a library. Such cells are called "proxy cells". For a library reference, the
        proxy cell is some kind of pointer to the library and the cell within the library.

        For PCells, this data can even be computed through some script.
        A PCell proxy represents all instances with a given set of parameters.

        Proxy cells cannot be modified, except that pcell parameters can be modified
        and PCell instances can be recomputed.

        This method has been introduced in version 0.22.
        """
    def is_top(self) -> bool:
        r"""
        @brief Gets a value indicating whether the cell is a top-level cell

        A cell is a top-level cell if there are no parent instantiations.
        """
    def is_valid(self, instance: Instance) -> bool:
        r"""
        @brief Tests if the given \Instance object is still pointing to a valid object
        This method has been introduced in version 0.16.
        If the instance represented by the given reference has been deleted, this method returns false. If however, another instance has been inserted already that occupies the original instances position, this method will return true again.
        """
    @overload
    def layout(self) -> Layout:
        r"""
        @brief Returns a reference to the layout where the cell resides

        this method has been introduced in version 0.22.
        """
    @overload
    def layout(self) -> Layout:
        r"""
        @brief Returns a reference to the layout where the cell resides (const references)

        this method has been introduced in version 0.22.
        """
    def library(self) -> Library:
        r"""
        @brief Returns a reference to the library from which the cell is imported
        if the cell is not imported from a library, this reference is nil.

        this method has been introduced in version 0.22.
        """
    def library_cell_index(self) -> int:
        r"""
        @brief Returns the index of the cell in the layout of the library (if it's a library proxy)
        Together with the \library method, it is possible to locate the source cell of
        a library proxy. The source cell can be retrieved from a cell "c" with 

        @code
        c.library.layout.cell(c.library_cell_index)
        @/code

        This cell may be itself a proxy,
        i.e. for pcell libraries, where the library cells are pcell variants which itself
        are proxies to a pcell.

        This method has been introduced in version 0.22.
        """
    @overload
    def move(self, src: int, dest: int) -> None:
        r"""
        @brief Moves the shapes from the source to the target layer

        The destination layer is not overwritten. Instead, the shapes are added to the shapes of the destination layer.
        This method will move shapes within the cell. To move shapes from another cell to this cell, use the copy method with the cell parameter.

        This method has been introduced in version 0.19.

        @param src The layer index of the source layer
        @param dest The layer index of the destination layer
        """
    @overload
    def move(self, src_cell: Cell, src_layer: int, dest: int) -> None:
        r"""
        @brief Moves shapes from another cell to the target layer in this cell

        This method will move all shapes on layer 'src_layer' of cell 'src_cell' to the layer 'dest' of this cell.
        The destination layer is not overwritten. Instead, the shapes are added to the shapes of the destination layer.
        If the source cell lives in a layout with a different database unit than that current cell is in, the shapes will be transformed accordingly. The same way, shape properties are transformed as well. Note that the shape transformation may require rounding to smaller coordinates. This may result in a slight distortion of the original shapes, in particular when transforming into a layout with a bigger database unit.
        @param src_cell The cell where to take the shapes from
        @param src_layer The layer index of the layer from which to take the shapes
        @param dest The layer index of the destination layer
        """
    def move_instances(self, source_cell: Cell) -> None:
        r"""
        @brief Moves the instances of child cells in the source cell to this cell
        @param source_cell The cell where the instances are moved from
        The source cell must reside in the same layout than this cell. The instances of child cells inside the source cell are moved to this cell. No new cells are created, just new instances are created to already existing cells in the target cell.

        The instances will be added to any existing instances in the cell.

        More elaborate methods of moving hierarchy trees between layouts are provided through the \move_tree_shapes (in cooperation with the \CellMapping class) or \move_tree methods.

        This method has been added in version 0.23.
        """
    @overload
    def move_shapes(self, source_cell: Cell) -> None:
        r"""
        @brief Moves the shapes from the given cell into this cell
        @param source_cell The cell from where to move shapes
        All shapes are moved from the source cell to this cell. Instances are not moved.

        The source cell can reside in a different layout. In this case, the shapes are moved over from the other layout into this layout. Database unit conversion is done automatically if the database units differ between the layouts. Note that this may lead to grid snapping effects if the database unit of the target layout is not an integer fraction of the source layout.

        If source and target layout are different, the layers of the source and target layout are identified by their layer/datatype number or name (if no layer/datatype is present).
        The shapes will be added to any shapes already in the cell.

        This method has been added in version 0.23.
        """
    @overload
    def move_shapes(self, source_cell: Cell, layer_mapping: LayerMapping) -> None:
        r"""
        @brief Moves the shapes from the given cell into this cell
        @param source_cell The cell from where to move shapes
        @param layer_mapping A \LayerMapping object that specifies which layers are moved and where
        All shapes on layers specified in the layer mapping object are moved from the source cell to this cell. Instances are not moved.
        The target layer is taken from the mapping table.

        The shapes will be added to any shapes already in the cell.

        This method has been added in version 0.23.
        """
    def move_tree(self, source_cell: Cell) -> List[int]:
        r"""
        @brief Moves the cell tree of the given cell into this cell
        @param source_cell The cell from where to move the cell tree
        @return A list of indexes of newly created cells
        The complete cell tree of the source cell is moved to the target cell plus all shapes in that tree are moved as well. This method will basically rebuild the cell tree of the source cell and empty the source cell.

        The source cell may reside in a separate layout. This method therefore provides a way to move over complete cell trees from one layout to another.

        The shapes and instances will be added to any shapes or instances already in the cell.

        This method has been added in version 0.23.
        """
    @overload
    def move_tree_shapes(self, source_cell: Cell, cell_mapping: CellMapping) -> None:
        r"""
        @brief Moves the shapes from the given cell and the cell tree below into this cell or subcells of this cell
        @param source_cell The starting cell from where to move shapes
        @param cell_mapping The cell mapping object that determines how cells are identified between source and target layout

        This method is provided if source and target cell reside in different layouts. If will move the shapes from all cells below the given source cell, but use a cell mapping object that provides a specification how cells are identified between the layouts. Cells in the source tree, for which no mapping is provided, will be flattened - their shapes will be propagated into parent cells for which a mapping is provided.

        The cell mapping object provides various methods to map cell trees between layouts. See the \CellMapping class for details about the mapping methods available. The cell mapping object is also responsible for creating a proper hierarchy of cells in the target layout if that is required.

        Layers are identified between the layouts by the layer/datatype number of name if no layer/datatype number is present.

        The shapes moved will be added to any shapes already in the cells.

        This method has been added in version 0.23.
        """
    @overload
    def move_tree_shapes(self, source_cell: Cell, cell_mapping: CellMapping, layer_mapping: LayerMapping) -> None:
        r"""
        @brief Moves the shapes from the given cell and the cell tree below into this cell or subcells of this cell with layer mapping
        @param source_cell The cell from where to move shapes and instances
        @param cell_mapping The cell mapping object that determines how cells are identified between source and target layout

        This method is provided if source and target cell reside in different layouts. If will move the shapes from all cells below the given source cell, but use a cell mapping object that provides a specification how cells are identified between the layouts. Cells in the source tree, for which no mapping is provided, will be flattened - their shapes will be propagated into parent cells for which a mapping is provided.

        The cell mapping object provides various methods to map cell trees between layouts. See the \CellMapping class for details about the mapping methods available. The cell mapping object is also responsible for creating a proper hierarchy of cells in the target layout if that is required.

        In addition, the layer mapping object can be specified which maps source to target layers. This feature can be used to restrict the move operation to a subset of layers or to convert shapes to different layers in that step.

        The shapes moved will be added to any shapes already in the cells.

        This method has been added in version 0.23.
        """
    def parent_cells(self) -> int:
        r"""
        @brief Gets the number of parent cells 

        The number of parent cells (cells which reference our cell) is reported.
        """
    @overload
    def pcell_declaration(self) -> PCellDeclaration_Native:
        r"""
        @brief Returns a reference to the PCell declaration
        If this cell is not a PCell variant, this method returns nil.
        PCell variants are proxy cells which are PCell incarnations for a specific parameter set.
        The \PCellDeclaration object allows one to retrieve PCell parameter definitions for example.

        This method has been introduced in version 0.22.
        """
    @overload
    def pcell_declaration(self, instance: Instance) -> PCellDeclaration_Native:
        r"""
        @brief Returns the PCell declaration of a pcell instance
        If the instance is not a PCell instance, this method returns nil.
        The \PCellDeclaration object allows one to retrieve PCell parameter definitions for example.

        This method has been introduced in version 0.22.
        """
    def pcell_id(self) -> int:
        r"""
        @brief Returns the PCell ID if the cell is a pcell variant
        This method returns the ID which uniquely identifies the PCell within the 
        layout where it's declared. It can be used to retrieve the PCell declaration 
        or to create new PCell variants.

        The method will be rarely used. It's more convenient to use \pcell_declaration to directly retrieve the PCellDeclaration object for example.

        This method has been introduced in version 0.22.
        """
    def pcell_library(self) -> Library:
        r"""
        @brief Returns the library where the PCell is declared if this cell is a PCell and it is not defined locally.
        A PCell often is not declared within the current layout but in some library. 
        This method returns a reference to that library, which technically is the last of the 
        chained library proxies. If this cell is not a PCell or it is not located in a 
        library, this method returns nil.

        This method has been introduced in version 0.22.
        """
    @overload
    def pcell_parameter(self, name: str) -> Any:
        r"""
        @brief Gets a PCell parameter by name if the cell is a PCell variant
        If the cell is a PCell variant, this method returns the parameter with the given name.
        If the cell is not a PCell variant or the name is not a valid PCell parameter name, the return value is nil.

        This method has been introduced in version 0.25.
        """
    @overload
    def pcell_parameter(self, instance: Instance, name: str) -> Any:
        r"""
        @brief Returns a PCell parameter by name for a pcell instance

        If the given instance is a PCell instance, this method returns the value of the PCell parameter with the given name.
        If the instance is not a PCell instance or the name is not a valid PCell parameter name, this
        method returns nil.

        This method has been introduced in version 0.25.
        """
    @overload
    def pcell_parameters(self) -> List[Any]:
        r"""
        @brief Returns the PCell parameters for a pcell variant
        If the cell is a PCell variant, this method returns a list of
        values for the PCell parameters. If the cell is not a PCell variant, this
        method returns an empty list. This method also returns the PCell parameters if
        the cell is a PCell imported from a library.

        This method has been introduced in version 0.22.
        """
    @overload
    def pcell_parameters(self, instance: Instance) -> List[Any]:
        r"""
        @brief Returns the PCell parameters for a pcell instance
        If the given instance is a PCell instance, this method returns a list of
        values for the PCell parameters. If the instance is not a PCell instance, this
        method returns an empty list.

        This method has been introduced in version 0.22.
        """
    @overload
    def pcell_parameters_by_name(self) -> Dict[str, Any]:
        r"""
        @brief Returns the PCell parameters for a pcell variant as a name to value dictionary
        If the cell is a PCell variant, this method returns a dictionary of
        values for the PCell parameters with the parameter names as the keys. If the cell is not a PCell variant, this
        method returns an empty dictionary. This method also returns the PCell parameters if
        the cell is a PCell imported from a library.

        This method has been introduced in version 0.24.
        """
    @overload
    def pcell_parameters_by_name(self, instance: Instance) -> Dict[str, Any]:
        r"""
        @brief Returns the PCell parameters for a pcell instance as a name to value dictionary
        If the given instance is a PCell instance, this method returns a dictionary of
        values for the PCell parameters with the parameter names as the keys. If the instance is not a PCell instance, this
        method returns an empty dictionary.

        This method has been introduced in version 0.24.
        """
    def property(self, key: Any) -> Any:
        r"""
        @brief Gets the user property with the given key
        This method is a convenience method that gets the property with the given key. If no property with that key exists, it will return nil. Using that method is more convenient than using the layout object and the properties ID to retrieve the property value. 
        This method has been introduced in version 0.23.
        """
    @overload
    def prune_cell(self) -> None:
        r"""
        @brief Deletes the cell plus subcells not used otherwise

        This deletes the cell and also all sub cells of the cell which are not used otherwise.
        All instances of this cell are deleted as well.
        A version of this method exists which allows one to specify the number of hierarchy levels to which subcells are considered.

        After the cell has been deleted, the Cell object becomes invalid. Do not access methods or attributes of this object after deleting the cell.

        This method has been introduced in version 0.23.
        """
    @overload
    def prune_cell(self, levels: int) -> None:
        r"""
        @brief Deletes the cell plus subcells not used otherwise

        This deletes the cell and also all sub cells of the cell which are not used otherwise.
        The number of hierarchy levels to consider can be specified as well. One level of hierarchy means that only the direct children of the cell are deleted with the cell itself.
        All instances of this cell are deleted as well.

        After the cell has been deleted, the Cell object becomes invalid. Do not access methods or attributes of this object after deleting the cell.

        @param levels The number of hierarchy levels to consider (-1: all, 0: none, 1: one level etc.)

        This method has been introduced in version 0.23.
        """
    @overload
    def prune_subcells(self) -> None:
        r"""
        @brief Deletes all sub cells of the cell which are not used otherwise

        This deletes all sub cells of the cell which are not used otherwise.
        All instances of the deleted cells are deleted as well.
        A version of this method exists which allows one to specify the number of hierarchy levels to which subcells are considered.

        This method has been introduced in version 0.23.
        """
    @overload
    def prune_subcells(self, levels: int) -> None:
        r"""
        @brief Deletes all sub cells of the cell which are not used otherwise down to the specified level of hierarchy

        This deletes all sub cells of the cell which are not used otherwise.
        All instances of the deleted cells are deleted as well.
        It is possible to specify how many levels of hierarchy below the given root cell are considered.

        @param levels The number of hierarchy levels to consider (-1: all, 0: none, 1: one level etc.)

        This method has been introduced in version 0.23.
        """
    def qname(self) -> str:
        r"""
        @brief Returns the library-qualified name

        Library cells will be indicated by returning a qualified name composed of the library name, a dot and the basic cell name. For example: "Basic.TEXT" will be the qname of the TEXT cell of the Basic library. For non-library cells, the qname is identical to the basic name (see \name).

        This method has been introduced in version 0.25.
        """
    @overload
    def read(self, file_name: str) -> List[int]:
        r"""
        @brief Reads a layout file into this cell
        This version uses the default options for reading the file.

        This method has been introduced in version 0.28.
        """
    @overload
    def read(self, file_name: str, options: LoadLayoutOptions) -> List[int]:
        r"""
        @brief Reads a layout file into this cell

        @param file_name The path of the file to read
        @param options The reader options to use
        @return The indexes of the cells created during the reading (new child cells)

        The format of the file will be determined from the file name. The layout will be read into the cell, potentially creating new layers and a subhierarchy of cells below this cell.

        This feature is equivalent to the following code:

        @code
        def Cell.read(file_name, options)
          layout = RBA::Layout::new
          layout.read(file_name, options)
          cm = RBA::CellMapping::new
          cm.for_single_cell_full(self, layout.top_cell)
          self.move_tree_shapes(layout.top_cell)
        end
        @/code

        See \move_tree_shapes and \CellMapping for more details and how to implement more elaborate schemes.

        This method has been introduced in version 0.28.
        """
    def refresh(self) -> None:
        r"""
        @brief Refreshes a proxy cell

        If the cell is a PCell variant, this method recomputes the PCell.
        If the cell is a library proxy, this method reloads the information from the library, but not the library itself.
        Note that if the cell is an PCell variant for a PCell coming from a library, this method will not recompute the PCell. Instead, you can use \Library#refresh to recompute all PCells from that library.

        You can use \Layout#refresh to refresh all cells from a layout.

        This method has been introduced in version 0.22.
        """
    @overload
    def replace(self, instance: Instance, cell_inst_array: CellInstArray) -> Instance:
        r"""
        @brief Replaces a cell instance (array) with a different one
        @return An \Instance object representing the new instance
        This method has been introduced in version 0.16. It can only be used in editable mode.
        The instance given by the instance object (first argument) is replaced by the given instance (second argument). The new object will not have any properties.
        """
    @overload
    def replace(self, instance: Instance, cell_inst_array: DCellInstArray) -> Instance:
        r"""
        @brief Replaces a cell instance (array) with a different one, given in micrometer units
        @return An \Instance object representing the new instance
        This method is identical to the corresponding \replace variant with a \CellInstArray argument. It however accepts a micrometer-unit \DCellInstArray object which is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    @overload
    def replace(self, instance: Instance, cell_inst_array: CellInstArray, property_id: int) -> Instance:
        r"""
        @brief Replaces a cell instance (array) with a different one with properties
        @return An \Instance object representing the new instance
        This method has been introduced in version 0.16. It can only be used in editable mode.
        The instance given by the instance object (first argument) is replaced by the given instance (second argument) with the given properties Id.
        The property Id must be obtained from the \Layout object's property_id method which associates a property set with a property Id.
        The new object will not have any properties.
        """
    @overload
    def replace(self, instance: Instance, cell_inst_array: DCellInstArray, property_id: int) -> Instance:
        r"""
        @brief Replaces a cell instance (array) with a different one and new properties, where the cell instance is given in micrometer units
        @return An \Instance object representing the new instance
        This method is identical to the corresponding \replace variant with a \CellInstArray argument and a property ID. It however accepts a micrometer-unit \DCellInstArray object which is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    def replace_prop_id(self, instance: Instance, property_id: int) -> Instance:
        r"""
        @brief Replaces (or install) the properties of a cell
        @return An Instance object representing the new instance
        This method has been introduced in version 0.16. It can only be used in editable mode.
        Changes the properties Id of the given instance or install a properties Id on that instance if it does not have one yet.
        The property Id must be obtained from the \Layout object's property_id method which associates a property set with a property Id.
        """
    def set_property(self, key: Any, value: Any) -> None:
        r"""
        @brief Sets the user property with the given key to the given value
        This method is a convenience method that sets the property with the given key to the given value. If no property with that key exists, it will create one. Using that method is more convenient than creating a new property set with a new ID and assigning that properties ID.
        This method may change the properties ID. Note: GDS only supports integer keys. OASIS supports numeric and string keys. 
        This method has been introduced in version 0.23.
        """
    @overload
    def shapes(self, layer_index: int) -> Shapes:
        r"""
        @brief Returns the shapes list of the given layer

        This method gives access to the shapes list on a certain layer.
        If the layer does not exist yet, it is created.

        @param index The layer index of the shapes list to retrieve

        @return A reference to the shapes list
        """
    @overload
    def shapes(self, layer_index: int) -> Shapes:
        r"""
        @brief Returns the shapes list of the given layer (const version)

        This method gives access to the shapes list on a certain layer. This is the const version - only const (reading) methods can be called on the returned object.

        @param index The layer index of the shapes list to retrieve

        @return A reference to the shapes list

        This variant has been introduced in version 0.26.4.
        """
    def swap(self, layer_index1: int, layer_index2: int) -> None:
        r"""
        @brief Swaps the layers given

        This method swaps two layers inside this cell.
        """
    @overload
    def transform(self, trans: DCplxTrans) -> None:
        r"""
        @brief Transforms the cell by the given, micrometer-unit transformation

        This method transforms all instances and all shapes by the given transformation. There is a variant called \transform_into which applies the transformation to instances in a way such that it can be applied recursively to the child cells. The difference is important in the presence of magnifications: "transform" will leave magnified instances while "transform_into" will not do so but expect the magnification to be applied inside the called cells too.

        This method has been introduced in version 0.26.7.
        """
    @overload
    def transform(self, trans: DTrans) -> None:
        r"""
        @brief Transforms the cell by the given, micrometer-unit transformation

        This method transforms all instances and all shapes by the given transformation. There is a variant called \transform_into which applies the transformation to instances in a way such that it can be applied recursively to the child cells.

        This method has been introduced in version 0.26.7.
        """
    @overload
    def transform(self, trans: ICplxTrans) -> None:
        r"""
        @brief Transforms the cell by the given complex integer transformation

        This method transforms all instances and all shapes by the given transformation. There is a variant called \transform_into which applies the transformation to instances in a way such that it can be applied recursively to the child cells. The difference is important in the presence of magnifications: "transform" will leave magnified instances while "transform_into" will not do so but expect the magnification to be applied inside the called cells too.

        This method has been introduced in version 0.26.7.
        """
    @overload
    def transform(self, trans: Trans) -> None:
        r"""
        @brief Transforms the cell by the given integer transformation

        This method transforms all instances and all shapes by the given transformation. There is a variant called \transform_into which applies the transformation to instances in a way such that it can be applied recursively to the child cells.

        This method has been introduced in version 0.26.7.
        """
    @overload
    def transform(self, instance: Instance, trans: DCplxTrans) -> Instance:
        r"""
        @brief Transforms the instance with the given complex floating-point transformation given in micrometer units
        @return A reference (an \Instance object) to the new instance
        This method is identical to the corresponding \transform method with a \ICplxTrans argument. For this variant however, the transformation is given in micrometer units and is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    @overload
    def transform(self, instance: Instance, trans: DTrans) -> Instance:
        r"""
        @brief Transforms the instance with the transformation given in micrometer units
        @return A reference (an \Instance object) to the new instance
        This method is identical to the corresponding \transform method with a \Trans argument. For this variant however, the transformation is given in micrometer units and is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    @overload
    def transform(self, instance: Instance, trans: ICplxTrans) -> Instance:
        r"""
        @brief Transforms the instance with the given complex integer transformation
        @return A reference (an \Instance object) to the new instance
        This method has been introduced in version 0.23.
        The original instance may be deleted and re-inserted by this method. Therefore, a new reference is returned.
        It is permitted in editable mode only.
        """
    @overload
    def transform(self, instance: Instance, trans: Trans) -> Instance:
        r"""
        @brief Transforms the instance with the given transformation
        @return A reference (an \Instance object) to the new instance
        This method has been introduced in version 0.16.
        The original instance may be deleted and re-inserted by this method. Therefore, a new reference is returned.
        It is permitted in editable mode only.
        """
    @overload
    def transform_into(self, trans: DCplxTrans) -> None:
        r"""
        @brief Transforms the cell into a new coordinate system with the given complex integer transformation where the transformation is in micrometer units
        This method is identical to the corresponding \transform_into method with a \ICplxTrans argument. For this variant however, the transformation is given in micrometer units and is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    @overload
    def transform_into(self, trans: DTrans) -> None:
        r"""
        @brief Transforms the cell into a new coordinate system with the given transformation where the transformation is in micrometer units
        This method is identical to the corresponding \transform_into method with a \Trans argument. For this variant however, the transformation is given in micrometer units and is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    @overload
    def transform_into(self, trans: ICplxTrans) -> None:
        r"""
        @brief Transforms the cell into a new coordinate system with the given complex integer transformation

        See the comments for the simple-transformation version for a description of this method.
        This method has been introduced in version 0.23.
        """
    @overload
    def transform_into(self, trans: Trans) -> None:
        r"""
        @brief Transforms the cell into a new coordinate system with the given transformation

        This method transforms all instances and all shapes. The instances are transformed in a way that allows propagation of the transformation into child cells. For this, it applies just a part of the given transformation to the instance such that when transforming the shapes of the cell instantiated, the result will reflect the desired transformation. Mathematically spoken, the transformation of the instance (A) is transformed with the given transformation T using "A' = T * A * Tinv" where Tinv is the inverse of T. In effect, the transformation T commutes with the new instance transformation A' and can be applied to child cells as well. This method is therefore useful to transform a hierarchy of cells.

        It has been introduced in version 0.23.
        """
    @overload
    def transform_into(self, instance: Instance, trans: DCplxTrans) -> Instance:
        r"""
        @brief Transforms the instance into a new coordinate system with the given complex transformation where the transformation is in micrometer units
        @return A reference (an \Instance object) to the new instance
        This method is identical to the corresponding \transform_into method with a \ICplxTrans argument. For this variant however, the transformation is given in micrometer units and is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    @overload
    def transform_into(self, instance: Instance, trans: DTrans) -> Instance:
        r"""
        @brief Transforms the instance into a new coordinate system with the given transformation where the transformation is in micrometer units
        @return A reference (an \Instance object) to the new instance
        This method is identical to the corresponding \transform_into method with a \Trans argument. For this variant however, the transformation is given in micrometer units and is translated to database units internally.

        This variant has been introduced in version 0.25.
        """
    @overload
    def transform_into(self, instance: Instance, trans: ICplxTrans) -> Instance:
        r"""
        @brief Transforms the instance into a new coordinate system with the given complex integer transformation
        @return A reference (an \Instance object) to the new instance

        See the comments for the simple-transformation version for a description of this method.
        This method has been introduced in version 0.23.
        The original instance may be deleted and re-inserted by this method. Therefore, a new reference is returned.
        It is permitted in editable mode only.
        """
    @overload
    def transform_into(self, instance: Instance, trans: Trans) -> Instance:
        r"""
        @brief Transforms the instance into a new coordinate system with the given transformation
        @return A reference (an \Instance object) to the new instance

        In contrast to the \transform method, this method allows propagation of the transformation into child cells. More precisely: it applies just a part of the given transformation to the instance, such that when transforming the cell instantiated and it's shapes with the same transformation, the result will reflect the desired transformation. Mathematically spoken, the transformation of the instance (A) is transformed with the given transformation T using "A' = T * A * Tinv" where Tinv is the inverse of T. In effect, the transformation T commutes with the new instance transformation A' and can be applied to child cells as well. This method is therefore useful to transform a hierarchy of cells.

        This method has been introduced in version 0.23.
        The original instance may be deleted and re-inserted by this method. Therefore, a new reference is returned.
        It is permitted in editable mode only.
        """
    @overload
    def write(self, file_name: str) -> None:
        r"""
        @brief Writes the cell to a layout file
        The format of the file will be determined from the file name. Only the cell and it's subtree below will be saved.

        This method has been introduced in version 0.23.
        """
    @overload
    def write(self, file_name: str, options: SaveLayoutOptions) -> None:
        r"""
        @brief Writes the cell to a layout file
        The format of the file will be determined from the file name. Only the cell and it's subtree below will be saved.
        In contrast to the other 'write' method, this version allows one to specify save options, i.e. scaling etc.

        This method has been introduced in version 0.23.
        """

class Instance:
    r"""
    @brief An instance proxy

    An instance proxy is basically a pointer to an instance of different kinds, 
    similar to \Shape, the shape proxy. \Instance objects can be duplicated without
    creating copies of the instances itself: the copy will still point to the same instance
    than the original.

    When the \Instance object is modified, the actual instance behind it is modified. The \Instance object acts as a simplified interface for single and array instances with or without properties.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    a: Vector
    r"""
    Getter:
    @brief Returns the displacement vector for the 'a' axis

    Starting with version 0.25 the displacement is of vector type.
    Setter:
    @brief Sets the displacement vector for the 'a' axis

    If the instance was not an array instance before it is made one.

    This method has been introduced in version 0.23. Starting with version 0.25 the displacement is of vector type.
    """
    b: Vector
    r"""
    Getter:
    @brief Returns the displacement vector for the 'b' axis

    Starting with version 0.25 the displacement is of vector type.
    Setter:
    @brief Sets the displacement vector for the 'b' axis in micrometer units

    Like \b= with an integer displacement, this method will set the displacement vector but it accepts a vector in micrometer units that is of \DVector type. The vector will be translated to database units internally.

    This method has been introduced in version 0.25.
    """
    cell: Cell
    r"""
    Getter:
    @brief Gets the \Cell object of the cell this instance refers to

    Please note that before version 0.23 this method returned the cell the instance is contained in. For consistency, this method has been renamed \parent_cell.

    This method has been introduced in version 0.23.
    Setter:
    @brief Sets the \Cell object this instance refers to

    Setting the cell object to nil is equivalent to deleting the instance.

    This method has been introduced in version 0.23.
    """
    cell_index: int
    r"""
    Getter:
    @brief Get the index of the cell this instance refers to

    Setter:
    @brief Sets the index of the cell this instance refers to

    This method has been introduced in version 0.23.
    """
    cell_inst: CellInstArray
    r"""
    Getter:
    @brief Gets the basic \CellInstArray object associated with this instance reference.
    Setter:
    @brief Returns the basic cell instance array object by giving a micrometer unit object.
    This method replaces the instance by the given CellInstArray object and it internally transformed into database units.

    This method has been introduced in version 0.25
    """
    cplx_trans: ICplxTrans
    r"""
    Getter:
    @brief Gets the complex transformation of the instance or the first instance in the array
    This method is always valid compared to \trans, since simple transformations can be expressed as complex transformations as well.
    Setter:
    @brief Sets the complex transformation of the instance or the first instance in the array

    This method has been introduced in version 0.23.
    """
    da: DVector
    r"""
    Getter:
    @brief Returns the displacement vector for the 'a' axis in micrometer units

    Like \a, this method returns the displacement, but it will be translated to database units internally.

    This method has been introduced in version 0.25.
    Setter:
    @brief Sets the displacement vector for the 'a' axis in micrometer units

    Like \a= with an integer displacement, this method will set the displacement vector but it accepts a vector in micrometer units that is of \DVector type. The vector will be translated to database units internally.

    This method has been introduced in version 0.25.
    """
    db: DVector
    r"""
    Getter:
    @brief Returns the displacement vector for the 'b' axis in micrometer units

    Like \b, this method returns the displacement, but it will be translated to database units internally.

    This method has been introduced in version 0.25.
    Setter:
    @brief Sets the displacement vector for the 'b' axis in micrometer units

    Like \b= with an integer displacement, this method will set the displacement vector but it accepts a vector in micrometer units that is of \DVector type. The vector will be translated to database units internally.

    This method has been introduced in version 0.25.
    """
    dcell_inst: DCellInstArray
    r"""
    Getter:
    @brief Returns the micrometer unit version of the basic cell instance array object.

    This method has been introduced in version 0.25
    Setter:
    @brief Returns the basic cell instance array object by giving a micrometer unit object.
    This method replaces the instance by the given CellInstArray object and it internally transformed into database units.

    This method has been introduced in version 0.25
    """
    dcplx_trans: DCplxTrans
    r"""
    Getter:
    @brief Gets the complex transformation of the instance or the first instance in the array (in micrometer units)
    This method returns the same transformation as \cplx_trans, but the displacement of this transformation is given in micrometer units. It is internally translated from database units into micrometers.

    This method has been introduced in version 0.25.

    Setter:
    @brief Sets the complex transformation of the instance or the first instance in the array (in micrometer units)
    This method sets the transformation the same way as \cplx_trans=, but the displacement of this transformation is given in micrometer units. It is internally translated into database units.

    This method has been introduced in version 0.25.
    """
    dtrans: DTrans
    r"""
    Getter:
    @brief Gets the transformation of the instance or the first instance in the array (in micrometer units)
    This method returns the same transformation as \cplx_trans, but the displacement of this transformation is given in micrometer units. It is internally translated from database units into micrometers.

    This method has been introduced in version 0.25.

    Setter:
    @brief Sets the transformation of the instance or the first instance in the array (in micrometer units)
    This method sets the transformation the same way as \cplx_trans=, but the displacement of this transformation is given in micrometer units. It is internally translated into database units.

    This method has been introduced in version 0.25.
    """
    na: int
    r"""
    Getter:
    @brief Returns the number of instances in the 'a' axis

    Setter:
    @brief Sets the number of instances in the 'a' axis

    If the instance was not an array instance before it is made one.

    This method has been introduced in version 0.23.
    """
    nb: int
    r"""
    Getter:
    @brief Returns the number of instances in the 'b' axis

    Setter:
    @brief Sets the number of instances in the 'b' axis

    If the instance was not an array instance before it is made one.

    This method has been introduced in version 0.23.
    """
    parent_cell: Cell
    r"""
    Getter:
    @brief Gets the cell this instance is contained in

    Returns nil if the instance does not live inside a cell.
    This method was named "cell" previously which lead to confusion with \cell_index.
    It was renamed to "parent_cell" in version 0.23.

    Setter:
    @brief Moves the instance to a different cell

    Both the current and the target cell must live in the same layout.

    This method has been introduced in version 0.23.
    """
    prop_id: int
    r"""
    Getter:
    @brief Gets the properties ID associated with the instance

    Setter:
    @brief Sets the properties ID associated with the instance
    This method is provided, if a properties ID has been derived already. Usually it's more convenient to use \delete_property, \set_property or \property.

    This method has been introduced in version 0.22.
    """
    trans: Trans
    r"""
    Getter:
    @brief Gets the transformation of the instance or the first instance in the array
    The transformation returned is only valid if the array does not represent a complex transformation array
    Setter:
    @brief Sets the transformation of the instance or the first instance in the array (in micrometer units)
    This method sets the transformation the same way as \cplx_trans=, but the displacement of this transformation is given in micrometer units. It is internally translated into database units.

    This method has been introduced in version 0.25.
    """
    @classmethod
    def new(cls) -> Instance:
        r"""
        @brief Creates a new object of this class
        """
    def __copy__(self) -> Instance:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> Instance:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, b: object) -> bool:
        r"""
        @brief Tests for equality of two Instance objects
        See the hint on the < operator.
        """
    def __getitem__(self, key: Any) -> Any:
        r"""
        @brief Gets the user property with the given key or, if available, the PCell parameter with the name given by the key
        Getting the PCell parameter has priority over the user property.
        This method has been introduced in version 0.25.
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def __len__(self) -> int:
        r"""
        @brief Gets the number of single instances in the instance array
        If the instance represents a single instance, the count is 1. Otherwise it is na*nb.
        """
    def __lt__(self, b: Instance) -> bool:
        r"""
        @brief Provides an order criterion for two Instance objects
        Warning: this operator is just provided to establish any order, not a particular one.
        """
    def __ne__(self, b: object) -> bool:
        r"""
        @brief Tests for inequality of two Instance objects
        Warning: this operator returns true if both objects refer to the same instance, not just identical ones.
        """
    def __setitem__(self, key: Any, value: Any) -> None:
        r"""
        @brief Sets the user property with the given key or, if available, the PCell parameter with the name given by the key
        Setting the PCell parameter has priority over the user property.
        This method has been introduced in version 0.25.
        """
    def __str__(self) -> str:
        r"""
        @brief Creates a string showing the contents of the reference

        This method has been introduced with version 0.16.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: Instance) -> None:
        r"""
        @brief Assigns another object to self
        """
    @overload
    def bbox(self) -> Box:
        r"""
        @brief Gets the bounding box of the instance
        The bounding box incorporates all instances that the array represents. It gives the overall extension of the child cell as seen in the calling cell (or all array members if the instance forms an array). 
        This method has been introduced in version 0.23.
        """
    @overload
    def bbox(self, layer_index: int) -> Box:
        r"""
        @brief Gets the bounding box of the instance for a given layer
        @param layer_index The index of the layer the bounding box will be computed for.
        The bounding box incorporates all instances that the array represents. It gives the overall extension of the child cell as seen in the calling cell (or all array members if the instance forms an array) for the given layer. If the layer is empty in this cell and all it's children', an empty bounding box will be returned. 
        This method has been introduced in version 0.25. 'bbox' is the preferred synonym for it since version 0.28.
        """
    def bbox_per_layer(self, layer_index: int) -> Box:
        r"""
        @brief Gets the bounding box of the instance for a given layer
        @param layer_index The index of the layer the bounding box will be computed for.
        The bounding box incorporates all instances that the array represents. It gives the overall extension of the child cell as seen in the calling cell (or all array members if the instance forms an array) for the given layer. If the layer is empty in this cell and all it's children', an empty bounding box will be returned. 
        This method has been introduced in version 0.25. 'bbox' is the preferred synonym for it since version 0.28.
        """
    def change_pcell_parameter(self, name: str, value: Any) -> None:
        r"""
        @brief Changes a single parameter of a PCell instance to the given value

        This method changes a parameter of a PCell instance to the given value. The name identifies the PCell parameter and must correspond to one parameter listed in the PCell declaration.

        This method has been introduced in version 0.24.
        """
    @overload
    def change_pcell_parameters(self, params: Sequence[Any]) -> None:
        r"""
        @brief Changes the parameters of a PCell instance to the list of parameters

        This method changes the parameters of a PCell instance to the given list of parameters. The list must correspond to the parameters listed in the pcell declaration.
        A more convenient method is provided with the same name which accepts a dictionary of names and values
        .
        This method has been introduced in version 0.24.
        """
    @overload
    def change_pcell_parameters(self, dict: Dict[str, Any]) -> None:
        r"""
        @brief Changes the parameters of a PCell instance to the dictionary of parameters

        This method changes the parameters of a PCell instance to the given values. The values are specifies as a dictionary of names (keys) vs. values.
        Unknown names are ignored and only the parameters listed in the dictionary are changed.

        This method has been introduced in version 0.24.
        """
    def convert_to_static(self) -> None:
        r"""
        @brief Converts a PCell instance to a static cell

        If the instance is a PCell instance, this method will convert the cell into a static cell and remove the PCell variant if required. A new cell will be created containing the PCell content but being a static cell. If the instance is not a PCell instance, this method won't do anything.

        This method has been introduced in version 0.24.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    @overload
    def dbbox(self) -> DBox:
        r"""
        @brief Gets the bounding box of the instance in micron units
        Gets the bounding box (see \bbox) of the instance, but will compute the micrometer unit box by multiplying \bbox with the database unit.

        This method has been introduced in version 0.25.
        """
    @overload
    def dbbox(self, layer_index: int) -> DBox:
        r"""
        @brief Gets the bounding box of the instance in micron units
        @param layer_index The index of the layer the bounding box will be computed for.
        Gets the bounding box (see \bbox) of the instance, but will compute the micrometer unit box by multiplying \bbox with the database unit.

        This method has been introduced in version 0.25. 'dbbox' is the preferred synonym for it since version 0.28.
        """
    def dbbox_per_layer(self, layer_index: int) -> DBox:
        r"""
        @brief Gets the bounding box of the instance in micron units
        @param layer_index The index of the layer the bounding box will be computed for.
        Gets the bounding box (see \bbox) of the instance, but will compute the micrometer unit box by multiplying \bbox with the database unit.

        This method has been introduced in version 0.25. 'dbbox' is the preferred synonym for it since version 0.28.
        """
    def delete(self) -> None:
        r"""
        @brief Deletes this instance

        After this method was called, the instance object is pointing to nothing.

        This method has been introduced in version 0.23.
        """
    def delete_property(self, key: Any) -> None:
        r"""
        @brief Deletes the user property with the given key
        This method is a convenience method that deletes the property with the given key. It does nothing if no property with that key exists. Using that method is more convenient than creating a new property set with a new ID and assigning that properties ID.
        This method may change the properties ID. Calling this method may invalidate any iterators. It should not be called inside a loop iterating over instances.

        This method has been introduced in version 0.22.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> Instance:
        r"""
        @brief Creates a copy of self
        """
    def explode(self) -> None:
        r"""
        @brief Explodes the instance array

        This method does nothing if the instance was not an array before.
        The instance object will point to the first instance of the array afterwards.

        This method has been introduced in version 0.23.
        """
    @overload
    def flatten(self) -> None:
        r"""
        @brief Flattens the instance

        This method will convert the instance to a number of shapes which are equivalent to the content of the cell. The instance itself will be removed.
        There is another variant of this method which allows specification of the number of hierarchy levels to flatten.

        This method has been introduced in version 0.24.
        """
    @overload
    def flatten(self, levels: int) -> None:
        r"""
        @brief Flattens the instance

        This method will convert the instance to a number of shapes which are equivalent to the content of the cell. The instance itself will be removed.
        This version of the method allows specification of the number of hierarchy levels to remove. Specifying 1 for 'levels' will remove the instance and replace it by the contents of the cell. Specifying a negative value or zero for the number of levels will flatten the instance completely.

        This method has been introduced in version 0.24.
        """
    def has_prop_id(self) -> bool:
        r"""
        @brief Returns true, if the instance has properties
        """
    def is_complex(self) -> bool:
        r"""
        @brief Tests, if the array is a complex array

        Returns true if the array represents complex instances (that is, with magnification and 
        arbitrary rotation angles).
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_null(self) -> bool:
        r"""
        @brief Checks, if the instance is a valid one
        """
    def is_pcell(self) -> bool:
        r"""
        @brief Returns a value indicating whether the instance is a PCell instance

        This method has been introduced in version 0.24.
        """
    def is_regular_array(self) -> bool:
        r"""
        @brief Tests, if this instance is a regular array
        """
    def is_valid(self) -> bool:
        r"""
        @brief Tests if the \Instance object is still pointing to a valid instance
        If the instance represented by the given reference has been deleted, this method returns false. If however, another instance has been inserted already that occupies the original instances position, this method will return true again.

        This method has been introduced in version 0.23 and is a shortcut for "inst.cell.is_valid?(inst)".
        """
    @overload
    def layout(self) -> Layout:
        r"""
        @brief Gets the layout this instance is contained in

        This method has been introduced in version 0.22.
        """
    @overload
    def layout(self) -> Layout:
        r"""
        @brief Gets the layout this instance is contained in

        This const version of the method has been introduced in version 0.25.
        """
    def pcell_declaration(self) -> PCellDeclaration_Native:
        r"""
        @brief Returns the PCell declaration object

        If the instance is a PCell instance, this method returns the PCell declaration object for that PCell. If not, this method will return nil.
        This method has been introduced in version 0.24.
        """
    def pcell_parameter(self, name: str) -> Any:
        r"""
        @brief Gets a PCell parameter by the name of the parameter
        @return The parameter value or nil if the instance is not a PCell or does not have a parameter with given name

        This method has been introduced in version 0.25.
        """
    def pcell_parameters(self) -> List[Any]:
        r"""
        @brief Gets the parameters of a PCell instance as a list of values
        @return A list of values

        If the instance is a PCell instance, this method will return an array of values where each value corresponds to one parameter. The order of the values is the order the parameters are declared in the PCell declaration.
        If the instance is not a PCell instance, this list returned will be empty.

        This method has been introduced in version 0.24.
        """
    def pcell_parameters_by_name(self) -> Dict[str, Any]:
        r"""
        @brief Gets the parameters of a PCell instance as a dictionary of values vs. names
        @return A dictionary of values by parameter name

        If the instance is a PCell instance, this method will return a map of values vs. parameter names. The names are the ones defined in the PCell declaration.If the instance is not a PCell instance, the dictionary returned will be empty.

        This method has been introduced in version 0.24.
        """
    def property(self, key: Any) -> Any:
        r"""
        @brief Gets the user property with the given key
        This method is a convenience method that gets the property with the given key. If no property with that key exists, it will return nil. Using that method is more convenient than using the layout object and the properties ID to retrieve the property value. 
        This method has been introduced in version 0.22.
        """
    def set_property(self, key: Any, value: Any) -> None:
        r"""
        @brief Sets the user property with the given key to the given value
        This method is a convenience method that sets the property with the given key to the given value. If no property with that key exists, it will create one. Using that method is more convenient than creating a new property set with a new ID and assigning that properties ID.
        This method may change the properties ID. Note: GDS only supports integer keys. OASIS supports numeric and string keys. Calling this method may invalidate any iterators. It should not be called inside a loop iterating over instances.

        This method has been introduced in version 0.22.
        """
    def size(self) -> int:
        r"""
        @brief Gets the number of single instances in the instance array
        If the instance represents a single instance, the count is 1. Otherwise it is na*nb.
        """
    @overload
    def to_s(self) -> str:
        r"""
        @brief Creates a string showing the contents of the reference

        This method has been introduced with version 0.16.
        """
    @overload
    def to_s(self, with_cellname: bool) -> str:
        r"""
        @brief Creates a string showing the contents of the reference

        Passing true to with_cellname makes the string contain the cellname instead of the cell index

        This method has been introduced with version 0.23.
        """
    @overload
    def transform(self, t: DCplxTrans) -> None:
        r"""
        @brief Transforms the instance array with the given complex transformation (given in micrometer units)
        Transforms the instance like \transform does, but with a transformation given in micrometer units. The displacement of this transformation is given in micrometers and is internally translated to database units.

        This method has been introduced in version 0.25.
        """
    @overload
    def transform(self, t: DTrans) -> None:
        r"""
        @brief Transforms the instance array with the given transformation (given in micrometer units)
        Transforms the instance like \transform does, but with a transformation given in micrometer units. The displacement of this transformation is given in micrometers and is internally translated to database units.

        This method has been introduced in version 0.25.
        """
    @overload
    def transform(self, t: ICplxTrans) -> None:
        r"""
        @brief Transforms the instance array with the given complex transformation
        See \Cell#transform for a description of this method.

        This method has been introduced in version 0.23.
        """
    @overload
    def transform(self, t: Trans) -> None:
        r"""
        @brief Transforms the instance array with the given transformation
        See \Cell#transform for a description of this method.

        This method has been introduced in version 0.23.
        """
    @overload
    def transform_into(self, t: DCplxTrans) -> None:
        r"""
        @brief Transforms the instance array with the given complex transformation (given in micrometer units)
        Transforms the instance like \transform_into does, but with a transformation given in micrometer units. The displacement of this transformation is given in micrometers and is internally translated to database units.

        This method has been introduced in version 0.25.
        """
    @overload
    def transform_into(self, t: DTrans) -> None:
        r"""
        @brief Transforms the instance array with the given transformation (given in micrometer units)
        Transforms the instance like \transform_into does, but with a transformation given in micrometer units. The displacement of this transformation is given in micrometers and is internally translated to database units.

        This method has been introduced in version 0.25.
        """
    @overload
    def transform_into(self, t: ICplxTrans) -> None:
        r"""
        @brief Transforms the instance array with the given transformation
        See \Cell#transform_into for a description of this method.

        This method has been introduced in version 0.23.
        """
    @overload
    def transform_into(self, t: Trans) -> None:
        r"""
        @brief Transforms the instance array with the given transformation
        See \Cell#transform_into for a description of this method.

        This method has been introduced in version 0.23.
        """

class ParentInstArray:
    r"""
    @brief A parent instance

    A parent instance is basically an inverse instance: instead of pointing
    to the child cell, it is pointing to the parent cell and the transformation
    is representing the shift of the parent cell relative to the child cell.
    For memory performance, a parent instance is not stored as a instance but
    rather as a reference to a child instance and a reference to the cell which
    is the parent.
    The parent instance itself is computed on the fly. It is representative for
    a set of instances belonging to the same cell index. The special parent instance
    iterator takes care of producing the right sequence (\Cell#each_parent_inst).

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    @classmethod
    def new(cls) -> ParentInstArray:
        r"""
        @brief Creates a new object of this class
        """
    def __copy__(self) -> ParentInstArray:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> ParentInstArray:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: ParentInstArray) -> None:
        r"""
        @brief Assigns another object to self
        """
    def child_inst(self) -> Instance:
        r"""
        @brief Retrieve the child instance associated with this parent instance

        Starting with version 0.15, this method returns an \Instance object rather than a \CellInstArray reference.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dinst(self) -> DCellInstArray:
        r"""
        @brief Compute the inverse instance by which the parent is seen from the child in micrometer units

        This convenience method has been introduced in version 0.28.
        """
    def dup(self) -> ParentInstArray:
        r"""
        @brief Creates a copy of self
        """
    def inst(self) -> CellInstArray:
        r"""
        @brief Compute the inverse instance by which the parent is seen from the child
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def parent_cell_index(self) -> int:
        r"""
        @brief Gets the index of the parent cell
        """

class CellInstArray:
    r"""
    @brief A single or array cell instance
    This object represents either single or array cell instances. A cell instance array is a regular array, described by two displacement vectors (a, b) and the instance count along that axes (na, nb). 

    In addition, this object represents either instances with simple transformations or instances with complex transformations. The latter includes magnified instances and instances rotated by an arbitrary angle.

    The cell which is instantiated is given by a cell index. The cell index can be converted to a cell pointer by using \Layout#cell. The cell index of a cell can be obtained using \Cell#cell_index.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    a: Vector
    r"""
    Getter:
    @brief Gets the displacement vector for the 'a' axis

    Starting with version 0.25 the displacement is of vector type.

    Setter:
    @brief Sets the displacement vector for the 'a' axis

    If the instance was not regular before this property is set, it will be initialized to a regular instance.

    This method was introduced in version 0.22. Starting with version 0.25 the displacement is of vector type.
    """
    b: Vector
    r"""
    Getter:
    @brief Gets the displacement vector for the 'b' axis

    Starting with version 0.25 the displacement is of vector type.

    Setter:
    @brief Sets the displacement vector for the 'b' axis

    If the instance was not regular before this property is set, it will be initialized to a regular instance.

    This method was introduced in version 0.22. Starting with version 0.25 the displacement is of vector type.
    """
    @property
    def cell(self) -> None:
        r"""
        WARNING: This variable can only be set, not retrieved.
        @brief Sets the cell this instance refers to
        This is a convenience method and equivalent to 'cell_index = cell.cell_index()'. There is no getter for the cell pointer because the \CellInstArray object only knows about cell indexes.

        This convenience method has been introduced in version 0.28.
        """
    cell_index: int
    r"""
    Getter:
    @brief Gets the cell index of the cell instantiated 
    Use \Layout#cell to get the \Cell object from the cell index.
    Setter:
    @brief Sets the index of the cell this instance refers to
    """
    cplx_trans: ICplxTrans
    r"""
    Getter:
    @brief Gets the complex transformation of the first instance in the array
    This method is always applicable, compared to \trans, since simple transformations can be expressed as complex transformations as well.
    Setter:
    @brief Sets the complex transformation of the instance or the first instance in the array

    This method was introduced in version 0.22.
    """
    na: int
    r"""
    Getter:
    @brief Gets the number of instances in the 'a' axis

    Setter:
    @brief Sets the number of instances in the 'a' axis

    If the instance was not regular before this property is set to a value larger than zero, it will be initialized to a regular instance.
    To make an instance a single instance, set na or nb to 0.

    This method was introduced in version 0.22.
    """
    nb: int
    r"""
    Getter:
    @brief Gets the number of instances in the 'b' axis

    Setter:
    @brief Sets the number of instances in the 'b' axis

    If the instance was not regular before this property is set to a value larger than zero, it will be initialized to a regular instance.
    To make an instance a single instance, set na or nb to 0.

    This method was introduced in version 0.22.
    """
    trans: Trans
    r"""
    Getter:
    @brief Gets the transformation of the first instance in the array
    The transformation returned is only valid if the array does not represent a complex transformation array
    Setter:
    @brief Sets the transformation of the instance or the first instance in the array

    This method was introduced in version 0.22.
    """
    @overload
    @classmethod
    def new(cls) -> CellInstArray:
        r"""
        @brief Creates en empty cell instance with size 0
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, disp: Vector) -> CellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param disp The displacement

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, trans: ICplxTrans) -> CellInstArray:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, trans: Trans) -> CellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param trans The transformation by which to instantiate the cell

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, disp: Vector) -> CellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param disp The displacement
        This convenience initializer has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, trans: ICplxTrans) -> CellInstArray:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell_index The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, trans: Trans) -> CellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, disp: Vector, a: Vector, b: Vector, na: int, nb: int) -> CellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param disp The basic displacement of the first instance
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, trans: ICplxTrans, a: Vector, b: Vector, na: int, nb: int) -> CellInstArray:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, trans: Trans, a: Vector, b: Vector, na: int, nb: int) -> CellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, disp: Vector, a: Vector, b: Vector, na: int, nb: int) -> CellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param disp The basic displacement of the first instance
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience initializer has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, trans: ICplxTrans, a: Vector, b: Vector, na: int, nb: int) -> CellInstArray:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell_index The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        Starting with version 0.25 the displacements are of vector type.
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, trans: Trans, a: Vector, b: Vector, na: int, nb: int) -> CellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        Starting with version 0.25 the displacements are of vector type.
        """
    def __copy__(self) -> CellInstArray:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> CellInstArray:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, other: object) -> bool:
        r"""
        @brief Compares two arrays for equality
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given cell instance. This method enables cell instances as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Creates en empty cell instance with size 0
        """
    @overload
    def __init__(self, cell: Cell, disp: Vector) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param disp The displacement

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell: Cell, trans: ICplxTrans) -> None:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell: Cell, trans: Trans) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param trans The transformation by which to instantiate the cell

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell_index: int, disp: Vector) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param disp The displacement
        This convenience initializer has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell_index: int, trans: ICplxTrans) -> None:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell_index The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        """
    @overload
    def __init__(self, cell_index: int, trans: Trans) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        """
    @overload
    def __init__(self, cell: Cell, disp: Vector, a: Vector, b: Vector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param disp The basic displacement of the first instance
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell: Cell, trans: ICplxTrans, a: Vector, b: Vector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell: Cell, trans: Trans, a: Vector, b: Vector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell_index: int, disp: Vector, a: Vector, b: Vector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param disp The basic displacement of the first instance
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience initializer has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell_index: int, trans: ICplxTrans, a: Vector, b: Vector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell_index The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        Starting with version 0.25 the displacements are of vector type.
        """
    @overload
    def __init__(self, cell_index: int, trans: Trans, a: Vector, b: Vector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        Starting with version 0.25 the displacements are of vector type.
        """
    def __len__(self) -> int:
        r"""
        @brief Gets the number of single instances in the array
        If the instance represents a single instance, the count is 1. Otherwise it is na*nb. Starting with version 0.27, there may be iterated instances for which the size is larger than 1, but \is_regular_array? will return false. In this case, use \each_trans or \each_cplx_trans to retrieve the individual placements of the iterated instance.
        """
    def __lt__(self, other: CellInstArray) -> bool:
        r"""
        @brief Compares two arrays for 'less'
        The comparison provides an arbitrary sorting criterion and not specific sorting order. It is guaranteed that if an array a is less than b, b is not less than a. In addition, it a is not less than b and b is not less than a, then a is equal to b.
        """
    def __ne__(self, other: object) -> bool:
        r"""
        @brief Compares two arrays for inequality
        """
    def __str__(self) -> str:
        r"""
        @brief Converts the array to a string

        This method was introduced in version 0.22.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: CellInstArray) -> None:
        r"""
        @brief Assigns another object to self
        """
    @overload
    def bbox(self, layout: Layout) -> Box:
        r"""
        @brief Gets the bounding box of the array
        The bounding box incorporates all instances that the array represents. It needs the layout object to access the actual cell from the cell index.
        """
    @overload
    def bbox(self, layout: Layout, layer_index: int) -> Box:
        r"""
        @brief Gets the bounding box of the array with respect to one layer
        The bounding box incorporates all instances that the array represents. It needs the layout object to access the actual cell from the cell index.

        'bbox' is the preferred synonym since version 0.28.
        """
    def bbox_per_layer(self, layout: Layout, layer_index: int) -> Box:
        r"""
        @brief Gets the bounding box of the array with respect to one layer
        The bounding box incorporates all instances that the array represents. It needs the layout object to access the actual cell from the cell index.

        'bbox' is the preferred synonym since version 0.28.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> CellInstArray:
        r"""
        @brief Creates a copy of self
        """
    def each_cplx_trans(self) -> Iterator[ICplxTrans]:
        r"""
        @brief Gets the complex transformations represented by this instance
        For a single instance, this iterator will deliver the single, complex transformation. For array instances, the iterator will deliver each complex transformation of the expanded array.
        This iterator is a generalization of \each_trans for general complex transformations.

        This method has been introduced in version 0.25.
        """
    def each_trans(self) -> Iterator[Trans]:
        r"""
        @brief Gets the simple transformations represented by this instance
        For a single instance, this iterator will deliver the single, simple transformation. For array instances, the iterator will deliver each simple transformation of the expanded array.

        This iterator will only deliver valid transformations if the instance array is not of complex type (see \is_complex?). A more general iterator that delivers the complex transformations is \each_cplx_trans.

        This method has been introduced in version 0.25.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given cell instance. This method enables cell instances as hash keys.

        This method has been introduced in version 0.25.
        """
    def invert(self) -> None:
        r"""
        @brief Inverts the array reference

        The inverted array reference describes in which transformations the parent cell is
        seen from the current cell.
        """
    def is_complex(self) -> bool:
        r"""
        @brief Gets a value indicating whether the array is a complex array

        Returns true if the array represents complex instances (that is, with magnification and 
        arbitrary rotation angles).
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_regular_array(self) -> bool:
        r"""
        @brief Gets a value indicating whether this instance is a regular array
        """
    def size(self) -> int:
        r"""
        @brief Gets the number of single instances in the array
        If the instance represents a single instance, the count is 1. Otherwise it is na*nb. Starting with version 0.27, there may be iterated instances for which the size is larger than 1, but \is_regular_array? will return false. In this case, use \each_trans or \each_cplx_trans to retrieve the individual placements of the iterated instance.
        """
    def to_s(self) -> str:
        r"""
        @brief Converts the array to a string

        This method was introduced in version 0.22.
        """
    @overload
    def transform(self, trans: ICplxTrans) -> None:
        r"""
        @brief Transforms the cell instance with the given complex transformation

        This method has been introduced in version 0.20.
        """
    @overload
    def transform(self, trans: Trans) -> None:
        r"""
        @brief Transforms the cell instance with the given transformation

        This method has been introduced in version 0.20.
        """
    @overload
    def transformed(self, trans: ICplxTrans) -> CellInstArray:
        r"""
        @brief Gets the transformed cell instance (complex transformation)

        This method has been introduced in version 0.20.
        """
    @overload
    def transformed(self, trans: Trans) -> CellInstArray:
        r"""
        @brief Gets the transformed cell instance

        This method has been introduced in version 0.20.
        """

class DCellInstArray:
    r"""
    @brief A single or array cell instance in micrometer units
    This object is identical to \CellInstArray, except that it holds coordinates in micron units instead of database units.

    This class has been introduced in version 0.25.
    """
    a: DVector
    r"""
    Getter:
    @brief Gets the displacement vector for the 'a' axis

    Setter:
    @brief Sets the displacement vector for the 'a' axis

    If the instance was not regular before this property is set, it will be initialized to a regular instance.
    """
    b: DVector
    r"""
    Getter:
    @brief Gets the displacement vector for the 'b' axis

    Setter:
    @brief Sets the displacement vector for the 'b' axis

    If the instance was not regular before this property is set, it will be initialized to a regular instance.
    """
    @property
    def cell(self) -> None:
        r"""
        WARNING: This variable can only be set, not retrieved.
        @brief Sets the cell this instance refers to
        This is a convenience method and equivalent to 'cell_index = cell.cell_index()'. There is no getter for the cell pointer because the \CellInstArray object only knows about cell indexes.

        This convenience method has been introduced in version 0.28.
        """
    cell_index: int
    r"""
    Getter:
    @brief Gets the cell index of the cell instantiated 
    Use \Layout#cell to get the \Cell object from the cell index.
    Setter:
    @brief Sets the index of the cell this instance refers to
    """
    cplx_trans: DCplxTrans
    r"""
    Getter:
    @brief Gets the complex transformation of the first instance in the array
    This method is always applicable, compared to \trans, since simple transformations can be expressed as complex transformations as well.
    Setter:
    @brief Sets the complex transformation of the instance or the first instance in the array
    """
    na: int
    r"""
    Getter:
    @brief Gets the number of instances in the 'a' axis

    Setter:
    @brief Sets the number of instances in the 'a' axis

    If the instance was not regular before this property is set to a value larger than zero, it will be initialized to a regular instance.
    To make an instance a single instance, set na or nb to 0.
    """
    nb: int
    r"""
    Getter:
    @brief Gets the number of instances in the 'b' axis

    Setter:
    @brief Sets the number of instances in the 'b' axis

    If the instance was not regular before this property is set to a value larger than zero, it will be initialized to a regular instance.
    To make an instance a single instance, set na or nb to 0.
    """
    trans: DTrans
    r"""
    Getter:
    @brief Gets the transformation of the first instance in the array
    The transformation returned is only valid if the array does not represent a complex transformation array
    Setter:
    @brief Sets the transformation of the instance or the first instance in the array
    """
    @overload
    @classmethod
    def new(cls) -> DCellInstArray:
        r"""
        @brief Creates en empty cell instance with size 0
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, disp: DVector) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param disp The displacement

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, trans: DCplxTrans) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, trans: DTrans) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param trans The transformation by which to instantiate the cell

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, disp: DVector) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param disp The displacement
        This convenience initializer has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, trans: DCplxTrans) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell_index The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, trans: DTrans) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, disp: DVector, a: DVector, b: DVector, na: int, nb: int) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param disp The basic displacement of the first instance
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, trans: DCplxTrans, a: DVector, b: DVector, na: int, nb: int) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell: Cell, trans: DTrans, a: DVector, b: DVector, na: int, nb: int) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, disp: DVector, a: DVector, b: DVector, na: int, nb: int) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param disp The basic displacement of the first instance
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience initializer has been introduced in version 0.28.
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, trans: DCplxTrans, a: DVector, b: DVector, na: int, nb: int) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell_index The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis
        """
    @overload
    @classmethod
    def new(cls, cell_index: int, trans: DTrans, a: DVector, b: DVector, na: int, nb: int) -> DCellInstArray:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis
        """
    def __copy__(self) -> DCellInstArray:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DCellInstArray:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, other: object) -> bool:
        r"""
        @brief Compares two arrays for equality
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given cell instance. This method enables cell instances as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Creates en empty cell instance with size 0
        """
    @overload
    def __init__(self, cell: Cell, disp: DVector) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param disp The displacement

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell: Cell, trans: DCplxTrans) -> None:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell: Cell, trans: DTrans) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param trans The transformation by which to instantiate the cell

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell_index: int, disp: DVector) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param disp The displacement
        This convenience initializer has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell_index: int, trans: DCplxTrans) -> None:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell_index The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        """
    @overload
    def __init__(self, cell_index: int, trans: DTrans) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        """
    @overload
    def __init__(self, cell: Cell, disp: DVector, a: DVector, b: DVector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param disp The basic displacement of the first instance
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell: Cell, trans: DCplxTrans, a: DVector, b: DVector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell: Cell, trans: DTrans, a: DVector, b: DVector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience variant takes a \Cell pointer and is equivalent to using 'cell.cell_index()'. It has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell_index: int, disp: DVector, a: DVector, b: DVector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param disp The basic displacement of the first instance
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis

        This convenience initializer has been introduced in version 0.28.
        """
    @overload
    def __init__(self, cell_index: int, trans: DCplxTrans, a: DVector, b: DVector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance with a complex transformation
        @param cell_index The cell to instantiate
        @param trans The complex transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis
        """
    @overload
    def __init__(self, cell_index: int, trans: DTrans, a: DVector, b: DVector, na: int, nb: int) -> None:
        r"""
        @brief Creates a single cell instance
        @param cell_index The cell to instantiate
        @param trans The transformation by which to instantiate the cell
        @param a The displacement vector of the array in the 'a' axis
        @param b The displacement vector of the array in the 'b' axis
        @param na The number of placements in the 'a' axis
        @param nb The number of placements in the 'b' axis
        """
    def __len__(self) -> int:
        r"""
        @brief Gets the number of single instances in the array
        If the instance represents a single instance, the count is 1. Otherwise it is na*nb. Starting with version 0.27, there may be iterated instances for which the size is larger than 1, but \is_regular_array? will return false. In this case, use \each_trans or \each_cplx_trans to retrieve the individual placements of the iterated instance.
        """
    def __lt__(self, other: DCellInstArray) -> bool:
        r"""
        @brief Compares two arrays for 'less'
        The comparison provides an arbitrary sorting criterion and not specific sorting order. It is guaranteed that if an array a is less than b, b is not less than a. In addition, it a is not less than b and b is not less than a, then a is equal to b.
        """
    def __ne__(self, other: object) -> bool:
        r"""
        @brief Compares two arrays for inequality
        """
    def __str__(self) -> str:
        r"""
        @brief Converts the array to a string
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: DCellInstArray) -> None:
        r"""
        @brief Assigns another object to self
        """
    @overload
    def bbox(self, layout: Layout) -> DBox:
        r"""
        @brief Gets the bounding box of the array
        The bounding box incorporates all instances that the array represents. It needs the layout object to access the actual cell from the cell index.
        """
    @overload
    def bbox(self, layout: Layout, layer_index: int) -> DBox:
        r"""
        @brief Gets the bounding box of the array with respect to one layer
        The bounding box incorporates all instances that the array represents. It needs the layout object to access the actual cell from the cell index.

        'bbox' is the preferred synonym since version 0.28.
        """
    def bbox_per_layer(self, layout: Layout, layer_index: int) -> DBox:
        r"""
        @brief Gets the bounding box of the array with respect to one layer
        The bounding box incorporates all instances that the array represents. It needs the layout object to access the actual cell from the cell index.

        'bbox' is the preferred synonym since version 0.28.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DCellInstArray:
        r"""
        @brief Creates a copy of self
        """
    def each_cplx_trans(self) -> Iterator[DCplxTrans]:
        r"""
        @brief Gets the complex transformations represented by this instance
        For a single instance, this iterator will deliver the single, complex transformation. For array instances, the iterator will deliver each complex transformation of the expanded array.
        This iterator is a generalization of \each_trans for general complex transformations.
        """
    def each_trans(self) -> Iterator[DTrans]:
        r"""
        @brief Gets the simple transformations represented by this instance
        For a single instance, this iterator will deliver the single, simple transformation. For array instances, the iterator will deliver each simple transformation of the expanded array.

        This iterator will only deliver valid transformations if the instance array is not of complex type (see \is_complex?). A more general iterator that delivers the complex transformations is \each_cplx_trans.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given cell instance. This method enables cell instances as hash keys.

        This method has been introduced in version 0.25.
        """
    def invert(self) -> None:
        r"""
        @brief Inverts the array reference

        The inverted array reference describes in which transformations the parent cell is
        seen from the current cell.
        """
    def is_complex(self) -> bool:
        r"""
        @brief Gets a value indicating whether the array is a complex array

        Returns true if the array represents complex instances (that is, with magnification and 
        arbitrary rotation angles).
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_regular_array(self) -> bool:
        r"""
        @brief Gets a value indicating whether this instance is a regular array
        """
    def size(self) -> int:
        r"""
        @brief Gets the number of single instances in the array
        If the instance represents a single instance, the count is 1. Otherwise it is na*nb. Starting with version 0.27, there may be iterated instances for which the size is larger than 1, but \is_regular_array? will return false. In this case, use \each_trans or \each_cplx_trans to retrieve the individual placements of the iterated instance.
        """
    def to_s(self) -> str:
        r"""
        @brief Converts the array to a string
        """
    @overload
    def transform(self, trans: DCplxTrans) -> None:
        r"""
        @brief Transforms the cell instance with the given complex transformation
        """
    @overload
    def transform(self, trans: DTrans) -> None:
        r"""
        @brief Transforms the cell instance with the given transformation
        """
    @overload
    def transformed(self, trans: DCplxTrans) -> DCellInstArray:
        r"""
        @brief Gets the transformed cell instance (complex transformation)
        """
    @overload
    def transformed(self, trans: DTrans) -> DCellInstArray:
        r"""
        @brief Gets the transformed cell instance
        """

class CellMapping:
    r"""
    @brief A cell mapping (source to target layout)

    A cell mapping is an association of cells in two layouts forming pairs of cells, i.e. one cell corresponds to another cell in the other layout. The CellMapping object describes the mapping of cells of a source layout B to a target layout A. The cell mapping object is basically a table associating a cell in layout B with a cell in layout A.

    The cell mapping is of particular interest for providing the cell mapping recipe in \Cell#copy_tree_shapes or \Cell#move_tree_shapes.

    The mapping object is used to create and hold that table. There are three basic modes in which a table can be generated:

    @ul
      @li Top-level identity (\for_single_cell and \for_single_cell_full) @/li
      @li Top-level identify for multiple cells (\for_multi_cells_full and \for_multi_cells_full) @/li
      @li Geometrical identity (\from_geometry and \from_geometry_full)@/li
      @li Name identity (\from_names and \from_names_full) @/li
    @/ul

    'full' refers to the way cells are treated which are not mentioned. In the 'full' versions, cells for which no mapping is established explicitly - specifically all child cells in top-level identity modes - are created in the target layout and instantiated according to their source layout hierarchy. Then, these new cells become targets of the respective source cells. In the plain version (without 'full' cells), no additional cells are created. For the case of \Layout#copy_tree_shapes cells not explicitly mapped are flattened. Hence for example, \for_single_cell will flatten all children of the source cell during \Layout#copy_tree_shapes or \Layout#move_tree_shapes.

    Top-level identity means that only one cell (the top cell) is regarded identical. All child cells are not considered identical. In full mode (see below), this will create a new, identical cell tree below the top cell in layout A.

    Geometrical identity is defined by the exact identity of the set of expanded instances in each starting cell. Therefore, when a cell is mapped to another cell, shapes can be transferred from one cell to another while effectively rendering the same flat geometry (in the context of the given starting cells). Location identity is basically the safest way to map cells from one hierarchy into another, because it preserves the flat shape geometry. However in some cases the algorithm may find multiple mapping candidates. In that case it will make a guess about what mapping to choose.

    Name identity means that cells are identified by their names - for a source cell in layer B, a target cell with the same name is looked up in the target layout A and a mapping is created if a cell with the same name is found. However, name identity does not mean that the cells are actually equivalent because they may be placed differently. Hence, cell mapping by name is not a good choice when it is important to preserve the shape geometry of a layer.

    A cell might not be mapped to another cell which basically means that there is no corresponding cell. In this case, flattening to the next mapped cell is an option to transfer geometries despite the missing mapping. You can enforce a mapping by using the mapping generator methods in 'full' mode, i.e. \from_names_full or \from_geometry_full. These versions will create new cells and their corresponding instances in the target layout if no suitable target cell is found.

    This is a simple example for a cell mapping preserving the hierarchy of the source cell and creating a hierarchy copy in the top cell of the target layout ('hierarchical merge'):

    @code
    cell_names = [ "A", "B", "C" ]

    source = RBA::Layout::new
    source.read("input.gds")

    target = RBA::Layout::new
    target_top = target.create_cell("IMPORTED")

    cm = RBA::CellMapping::new
    # Copies the source layout hierarchy into the target top cell:
    cm.for_single_cell_full(target_top, source.top_cell)
    target.copy_tree_shapes(source, cm)
    @/code

    Without 'full', the effect is move-with-flattening (note we're using 'move' in this example):

    @code
    cell_names = [ "A", "B", "C" ]

    source = RBA::Layout::new
    source.read("input.gds")

    target = RBA::Layout::new
    target_top = target.create_cell("IMPORTED")

    cm = RBA::CellMapping::new
    # Flattens the source layout hierarchy into the target top cell:
    cm.for_single_cell(target_top, source.top_cell)
    target.move_tree_shapes(source, cm)
    @/code

    This is another example for using \CellMapping in multiple top cell identity mode. It extracts cells 'A', 'B' and 'C' from one layout and copies them to another. It will also copy all shapes and all child cells. Child cells which are shared between the three initial cells will be shared in the target layout too.

    @code
    cell_names = [ "A", "B", "C" ]

    source = RBA::Layout::new
    source.read("input.gds")

    target = RBA::Layout::new

    source_cells = cell_names.collect { |n| source.cell_by_name(n) }
    target_cells = cell_names.collect { |n| target.create_cell(n) }

    cm = RBA::CellMapping::new
    cm.for_multi_cells_full(target_cells, source_cells)
    target.copy_tree_shapes(source, cm)
    @/code
    """
    DropCell: ClassVar[int]
    r"""
    @brief A constant indicating the request to drop a cell

    If used as a pseudo-target for the cell mapping, this index indicates that the cell shall be dropped rather than created on the target side or skipped by flattening. Instead, all shapes of this cell are discarded and it's children are not translated unless explicitly requested or if required are children for other cells.

    This constant has been introduced in version 0.25.
    """
    @classmethod
    def new(cls) -> CellMapping:
        r"""
        @brief Creates a new object of this class
        """
    def __copy__(self) -> CellMapping:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> CellMapping:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: CellMapping) -> None:
        r"""
        @brief Assigns another object to self
        """
    def cell_mapping(self, cell_index_b: int) -> int:
        r"""
        @brief Determines cell mapping of a layout_b cell to the corresponding layout_a cell.


        @param cell_index_b The index of the cell in layout_b whose mapping is requested.
        @return The cell index in layout_a.

        Note that the returned index can be \DropCell to indicate the cell shall be dropped.
        """
    def clear(self) -> None:
        r"""
        @brief Clears the mapping.

        This method has been introduced in version 0.23.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> CellMapping:
        r"""
        @brief Creates a copy of self
        """
    @overload
    def for_multi_cells(self, cell_a: Sequence[Cell], cell_b: Sequence[Cell]) -> None:
        r"""
        @brief Initializes the cell mapping for top-level identity

        @param cell_a A list of target cells.
        @param cell_b A list of source cells.
        @return A list of indexes of cells created.

        This is a convenience version which uses cell references instead of layout/cell index combinations. It has been introduced in version 0.28.
        """
    @overload
    def for_multi_cells(self, layout_a: Layout, cell_indexes_a: Sequence[int], layout_b: Layout, cell_indexes_b: Sequence[int]) -> None:
        r"""
        @brief Initializes the cell mapping for top-level identity

        @param layout_a The target layout.
        @param cell_indexes_a A list of cell indexes for the target cells.
        @param layout_b The source layout.
        @param cell_indexes_b A list of cell indexes for the source cells (same number of indexes than \cell_indexes_a).

        The cell mapping is created for cells from cell_indexes_b to cell from cell_indexes_a in the respective layouts. This method clears the mapping and creates one for each cell pair from cell_indexes_b vs. cell_indexes_a. If used for \Layout#copy_tree_shapes or \Layout#move_tree_shapes, this cell mapping will essentially flatten the source cells in the target layout.

        This method is equivalent to \clear, followed by \map(cell_index_a, cell_index_b) for each cell pair.

        This method has been introduced in version 0.27.
        """
    @overload
    def for_multi_cells_full(self, cell_a: Sequence[Cell], cell_b: Sequence[Cell]) -> List[int]:
        r"""
        @brief Initializes the cell mapping for top-level identity in full mapping mode

        @param cell_a A list of target cells.
        @param cell_b A list of source cells.
        @return A list of indexes of cells created.

        This is a convenience version which uses cell references instead of layout/cell index combinations. It has been introduced in version 0.28.
        """
    @overload
    def for_multi_cells_full(self, layout_a: Layout, cell_indexes_a: Sequence[int], layout_b: Layout, cell_indexes_b: Sequence[int]) -> List[int]:
        r"""
        @brief Initializes the cell mapping for top-level identity in full mapping mode

        @param layout_a The target layout.
        @param cell_indexes_a A list of cell indexes for the target cells.
        @param layout_b The source layout.
        @param cell_indexes_b A list of cell indexes for the source cells (same number of indexes than \cell_indexes_a).

        The cell mapping is created for cells from cell_indexes_b to cell from cell_indexes_a in the respective layouts. This method clears the mapping and creates one for each cell pair from cell_indexes_b vs. cell_indexes_a. In addition and in contrast to \for_multi_cells, this method completes the mapping by adding all the child cells of all cells in cell_indexes_b to layout_a and creating the proper instances.

        This method has been introduced in version 0.27.
        """
    @overload
    def for_single_cell(self, cell_a: Cell, cell_b: Cell) -> None:
        r"""
        @brief Initializes the cell mapping for top-level identity

        @param cell_a The target cell.
        @param cell_b The source cell.
        @return A list of indexes of cells created.

        This is a convenience version which uses cell references instead of layout/cell index combinations. It has been introduced in version 0.28.
        """
    @overload
    def for_single_cell(self, layout_a: Layout, cell_index_a: int, layout_b: Layout, cell_index_b: int) -> None:
        r"""
        @brief Initializes the cell mapping for top-level identity

        @param layout_a The target layout.
        @param cell_index_a The index of the target cell.
        @param layout_b The source layout.
        @param cell_index_b The index of the source cell.

        The cell mapping is created for cell_b to cell_a in the respective layouts. This method clears the mapping and creates one for the single cell pair. If used for \Cell#copy_tree or \Cell#move_tree, this cell mapping will essentially flatten the cell.

        This method is equivalent to \clear, followed by \map(cell_index_a, cell_index_b).

        This method has been introduced in version 0.23.
        """
    @overload
    def for_single_cell_full(self, cell_a: Cell, cell_b: Cell) -> List[int]:
        r"""
        @brief Initializes the cell mapping for top-level identity in full mapping mode

        @param cell_a The target cell.
        @param cell_b The source cell.
        @return A list of indexes of cells created.

        This is a convenience version which uses cell references instead of layout/cell index combinations. It has been introduced in version 0.28.
        """
    @overload
    def for_single_cell_full(self, layout_a: Layout, cell_index_a: int, layout_b: Layout, cell_index_b: int) -> List[int]:
        r"""
        @brief Initializes the cell mapping for top-level identity in full mapping mode

        @param layout_a The target layout.
        @param cell_index_a The index of the target cell.
        @param layout_b The source layout.
        @param cell_index_b The index of the source cell.

        The cell mapping is created for cell_b to cell_a in the respective layouts. This method clears the mapping and creates one for the single cell pair. In addition and in contrast to \for_single_cell, this method completes the mapping by adding all the child cells of cell_b to layout_a and creating the proper instances.

        This method has been introduced in version 0.23.
        """
    @overload
    def from_geometry(self, cell_a: Cell, cell_b: Cell) -> None:
        r"""
        @brief Initializes the cell mapping using the geometrical identity

        @param cell_a The target cell.
        @param cell_b The source cell.
        @return A list of indexes of cells created.

        This is a convenience version which uses cell references instead of layout/cell index combinations. It has been introduced in version 0.28.
        """
    @overload
    def from_geometry(self, layout_a: Layout, cell_index_a: int, layout_b: Layout, cell_index_b: int) -> None:
        r"""
        @brief Initializes the cell mapping using the geometrical identity

        @param layout_a The target layout.
        @param cell_index_a The index of the target starting cell.
        @param layout_b The source layout.
        @param cell_index_b The index of the source starting cell.

        The cell mapping is created for cells below cell_a and cell_b in the respective layouts. This method employs geometrical identity to derive mappings for the child cells of the starting cell in layout A and B.
        If the geometrical identity is ambiguous, the algorithm will make an arbitrary choice.

        This method has been introduced in version 0.23.
        """
    @overload
    def from_geometry_full(self, cell_a: Cell, cell_b: Cell) -> List[int]:
        r"""
        @brief Initializes the cell mapping using the geometrical identity in full mapping mode

        @param cell_a The target cell.
        @param cell_b The source cell.
        @return A list of indexes of cells created.

        This is a convenience version which uses cell references instead of layout/cell index combinations. It has been introduced in version 0.28.
        """
    @overload
    def from_geometry_full(self, layout_a: Layout, cell_index_a: int, layout_b: Layout, cell_index_b: int) -> List[int]:
        r"""
        @brief Initializes the cell mapping using the geometrical identity in full mapping mode

        @param layout_a The target layout.
        @param cell_index_a The index of the target starting cell.
        @param layout_b The source layout.
        @param cell_index_b The index of the source starting cell.
        @return A list of indexes of cells created.

        The cell mapping is created for cells below cell_a and cell_b in the respective layouts. This method employs geometrical identity to derive mappings for the child cells of the starting cell in layout A and B.
        If the geometrical identity is ambiguous, the algorithm will make an arbitrary choice.

        Full mapping means that cells which are not found in the target layout A are created there plus their corresponding instances are created as well. The returned list will contain the indexes of all cells created for that reason.

        This method has been introduced in version 0.23.
        """
    @overload
    def from_names(self, cell_a: Cell, cell_b: Cell) -> None:
        r"""
        @brief Initializes the cell mapping using the name identity

        @param cell_a The target cell.
        @param cell_b The source cell.
        @return A list of indexes of cells created.

        This is a convenience version which uses cell references instead of layout/cell index combinations. It has been introduced in version 0.28.
        """
    @overload
    def from_names(self, layout_a: Layout, cell_index_a: int, layout_b: Layout, cell_index_b: int) -> None:
        r"""
        @brief Initializes the cell mapping using the name identity

        @param layout_a The target layout.
        @param cell_index_a The index of the target starting cell.
        @param layout_b The source layout.
        @param cell_index_b The index of the source starting cell.

        The cell mapping is created for cells below cell_a and cell_b in the respective layouts.
        This method employs name identity to derive mappings for the child cells of the starting cell in layout A and B.

        This method has been introduced in version 0.23.
        """
    @overload
    def from_names_full(self, cell_a: Cell, cell_b: Cell) -> List[int]:
        r"""
        @brief Initializes the cell mapping using the name identity in full mapping mode

        @param cell_a The target cell.
        @param cell_b The source cell.
        @return A list of indexes of cells created.

        This is a convenience version which uses cell references instead of layout/cell index combinations. It has been introduced in version 0.28.
        """
    @overload
    def from_names_full(self, layout_a: Layout, cell_index_a: int, layout_b: Layout, cell_index_b: int) -> List[int]:
        r"""
        @brief Initializes the cell mapping using the name identity in full mapping mode

        @param layout_a The target layout.
        @param cell_index_a The index of the target starting cell.
        @param layout_b The source layout.
        @param cell_index_b The index of the source starting cell.
        @return A list of indexes of cells created.

        The cell mapping is created for cells below cell_a and cell_b in the respective layouts.
        This method employs name identity to derive mappings for the child cells of the starting cell in layout A and B.

        Full mapping means that cells which are not found in the target layout A are created there plus their corresponding instances are created as well. The returned list will contain the indexes of all cells created for that reason.

        This method has been introduced in version 0.23.
        """
    def has_mapping(self, cell_index_b: int) -> bool:
        r"""
        @brief Returns as value indicating whether a cell of layout_b has a mapping to a layout_a cell.

        @param cell_index_b The index of the cell in layout_b whose mapping is requested.
        @return true, if the cell has a mapping

        Note that if the cell is supposed to be dropped (see \DropCell), the respective source cell will also be regarded "mapped", so has_mapping? will return true in this case.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def map(self, cell_index_b: int, cell_index_a: int) -> None:
        r"""
        @brief Explicitly specifies a mapping.


        @param cell_index_b The index of the cell in layout B (the "source")
        @param cell_index_a The index of the cell in layout A (the "target") - this index can be \DropCell

        Beside using the mapping generator algorithms provided through \from_names and \from_geometry, it is possible to explicitly specify cell mappings using this method.

        This method has been introduced in version 0.23.
        """
    def table(self) -> Dict[int, int]:
        r"""
        @brief Returns the mapping table.

        The mapping table is a dictionary where the keys are source layout cell indexes and the values are the target layout cell indexes.
        Note that the target cell index can be \DropCell to indicate that a cell is supposed to be dropped.

        This method has been introduced in version 0.25.
        """

class CompoundRegionOperationNode:
    r"""
    @brief A base class for compound DRC operations

    This class is not intended to be used directly but rather provide a factory for various incarnations of compound operation nodes. Compound operations are a way to specify complex DRC operations put together by building a tree of operations. This operation tree then is executed with \Region#complex_op and will act on individual clusters of shapes and their interacting neighbors.

    A basic concept to the compound operations is the 'subject' (primary) and 'intruder' (secondary) input. The 'subject' is the Region, 'complex_op' with the operation tree is executed on. 'intruders' are regions inserted into the equation through secondary input nodes created with \new_secondary_node. The algorithm will execute the operation tree for every subject shape considering intruder shapes from the secondary inputs. The algorithm will only act on subject shapes primarily. As a consequence, 'lonely' intruder shapes without a subject shape are not considered at all. Only subject shapes trigger evaluation of the operation tree.

    The search distance for intruder shapes is determined by the operation and computed from the operation's requirements.

    NOTE: this feature is experimental and not deployed into the the DRC framework yet.

    This class has been introduced in version 0.27.
    """
    class LogicalOp:
        r"""
        @brief This class represents the CompoundRegionOperationNode::LogicalOp enum

        This enum has been introduced in version 0.27.
        """
        LogAnd: ClassVar[CompoundRegionOperationNode.LogicalOp]
        r"""
        @brief Indicates a logical '&&' (and).
        """
        LogOr: ClassVar[CompoundRegionOperationNode.LogicalOp]
        r"""
        @brief Indicates a logical '||' (or).
        """
        @overload
        @classmethod
        def new(cls, i: int) -> CompoundRegionOperationNode.LogicalOp:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        @classmethod
        def new(cls, s: str) -> CompoundRegionOperationNode.LogicalOp:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares two enums
            """
        @overload
        def __init__(self, i: int) -> None:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        def __init__(self, s: str) -> None:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __lt__(self, other: int) -> bool:
            r"""
            @brief Returns true if the enum is less (in the enum symbol order) than the integer value
            """
        @overload
        def __lt__(self, other: CompoundRegionOperationNode.LogicalOp) -> bool:
            r"""
            @brief Returns true if the first enum is less (in the enum symbol order) than the second
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer for inequality
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares two enums for inequality
            """
        def __repr__(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def __str__(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
        def inspect(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def to_i(self) -> int:
            r"""
            @brief Gets the integer value from the enum
            """
        def to_s(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
    class GeometricalOp:
        r"""
        @brief This class represents the CompoundRegionOperationNode::GeometricalOp enum

        This enum has been introduced in version 0.27.
        """
        And: ClassVar[CompoundRegionOperationNode.GeometricalOp]
        r"""
        @brief Indicates a geometrical '&' (and).
        """
        Not: ClassVar[CompoundRegionOperationNode.GeometricalOp]
        r"""
        @brief Indicates a geometrical '-' (not).
        """
        Or: ClassVar[CompoundRegionOperationNode.GeometricalOp]
        r"""
        @brief Indicates a geometrical '|' (or).
        """
        Xor: ClassVar[CompoundRegionOperationNode.GeometricalOp]
        r"""
        @brief Indicates a geometrical '^' (xor).
        """
        @overload
        @classmethod
        def new(cls, i: int) -> CompoundRegionOperationNode.GeometricalOp:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        @classmethod
        def new(cls, s: str) -> CompoundRegionOperationNode.GeometricalOp:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares two enums
            """
        @overload
        def __init__(self, i: int) -> None:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        def __init__(self, s: str) -> None:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __lt__(self, other: int) -> bool:
            r"""
            @brief Returns true if the enum is less (in the enum symbol order) than the integer value
            """
        @overload
        def __lt__(self, other: CompoundRegionOperationNode.GeometricalOp) -> bool:
            r"""
            @brief Returns true if the first enum is less (in the enum symbol order) than the second
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares two enums for inequality
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer for inequality
            """
        def __repr__(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def __str__(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
        def inspect(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def to_i(self) -> int:
            r"""
            @brief Gets the integer value from the enum
            """
        def to_s(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
    class ResultType:
        r"""
        @brief This class represents the CompoundRegionOperationNode::ResultType enum

        This enum has been introduced in version 0.27.
        """
        EdgePairs: ClassVar[CompoundRegionOperationNode.ResultType]
        r"""
        @brief Indicates edge pair result type.
        """
        Edges: ClassVar[CompoundRegionOperationNode.ResultType]
        r"""
        @brief Indicates edge result type.
        """
        Region: ClassVar[CompoundRegionOperationNode.ResultType]
        r"""
        @brief Indicates polygon result type.
        """
        @overload
        @classmethod
        def new(cls, i: int) -> CompoundRegionOperationNode.ResultType:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        @classmethod
        def new(cls, s: str) -> CompoundRegionOperationNode.ResultType:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares two enums
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer value
            """
        @overload
        def __init__(self, i: int) -> None:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        def __init__(self, s: str) -> None:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __lt__(self, other: int) -> bool:
            r"""
            @brief Returns true if the enum is less (in the enum symbol order) than the integer value
            """
        @overload
        def __lt__(self, other: CompoundRegionOperationNode.ResultType) -> bool:
            r"""
            @brief Returns true if the first enum is less (in the enum symbol order) than the second
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer for inequality
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares two enums for inequality
            """
        def __repr__(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def __str__(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
        def inspect(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def to_i(self) -> int:
            r"""
            @brief Gets the integer value from the enum
            """
        def to_s(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
    class ParameterType:
        r"""
        @brief This class represents the parameter type enum used in \CompoundRegionOperationNode#new_bbox_filter

        This enum has been introduced in version 0.27.
        """
        BoxAverageDim: ClassVar[CompoundRegionOperationNode.ParameterType]
        r"""
        @brief Measures the average of width and height of the bounding box
        """
        BoxHeight: ClassVar[CompoundRegionOperationNode.ParameterType]
        r"""
        @brief Measures the height of the bounding box
        """
        BoxMaxDim: ClassVar[CompoundRegionOperationNode.ParameterType]
        r"""
        @brief Measures the maximum dimension of the bounding box
        """
        BoxMinDim: ClassVar[CompoundRegionOperationNode.ParameterType]
        r"""
        @brief Measures the minimum dimension of the bounding box
        """
        BoxWidth: ClassVar[CompoundRegionOperationNode.ParameterType]
        r"""
        @brief Measures the width of the bounding box
        """
        @overload
        @classmethod
        def new(cls, i: int) -> CompoundRegionOperationNode.ParameterType:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        @classmethod
        def new(cls, s: str) -> CompoundRegionOperationNode.ParameterType:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares two enums
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer value
            """
        @overload
        def __init__(self, i: int) -> None:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        def __init__(self, s: str) -> None:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __lt__(self, other: int) -> bool:
            r"""
            @brief Returns true if the enum is less (in the enum symbol order) than the integer value
            """
        @overload
        def __lt__(self, other: CompoundRegionOperationNode.ParameterType) -> bool:
            r"""
            @brief Returns true if the first enum is less (in the enum symbol order) than the second
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares two enums for inequality
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer for inequality
            """
        def __repr__(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def __str__(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
        def inspect(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def to_i(self) -> int:
            r"""
            @brief Gets the integer value from the enum
            """
        def to_s(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
    class RatioParameterType:
        r"""
        @brief This class represents the parameter type enum used in \CompoundRegionOperationNode#new_ratio_filter

        This enum has been introduced in version 0.27.
        """
        AreaRatio: ClassVar[CompoundRegionOperationNode.RatioParameterType]
        r"""
        @brief Measures the area ratio (bounding box area / polygon area)
        """
        AspectRatio: ClassVar[CompoundRegionOperationNode.RatioParameterType]
        r"""
        @brief Measures the aspect ratio of the bounding box (larger / smaller dimension)
        """
        RelativeHeight: ClassVar[CompoundRegionOperationNode.RatioParameterType]
        r"""
        @brief Measures the relative height (height / width)
        """
        @overload
        @classmethod
        def new(cls, i: int) -> CompoundRegionOperationNode.RatioParameterType:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        @classmethod
        def new(cls, s: str) -> CompoundRegionOperationNode.RatioParameterType:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares two enums
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer value
            """
        @overload
        def __init__(self, i: int) -> None:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        def __init__(self, s: str) -> None:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __lt__(self, other: int) -> bool:
            r"""
            @brief Returns true if the enum is less (in the enum symbol order) than the integer value
            """
        @overload
        def __lt__(self, other: CompoundRegionOperationNode.RatioParameterType) -> bool:
            r"""
            @brief Returns true if the first enum is less (in the enum symbol order) than the second
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer for inequality
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares two enums for inequality
            """
        def __repr__(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def __str__(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
        def inspect(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def to_i(self) -> int:
            r"""
            @brief Gets the integer value from the enum
            """
        def to_s(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
    description: str
    r"""
    Getter:
    @brief Gets the description for this node
    Setter:
    @brief Sets the description for this node
    """
    distance: int
    r"""
    Getter:
    @brief Gets the distance value for this node
    Setter:
    @brief Sets the distance value for this nodeUsually it's not required to provide a distance because the nodes compute a distance based on their operation. If necessary you can supply a distance. The processor will use this distance or the computed one, whichever is larger.
    """
    @classmethod
    def new(cls) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a new object of this class
        """
    @classmethod
    def new_area_filter(cls, input: CompoundRegionOperationNode, inverse: Optional[bool] = ..., amin: Optional[int] = ..., amax: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering the input by area.
        This node renders the input if the area is between amin and amax (exclusively). If 'inverse' is set to true, the input shape is returned if the area is less than amin (exclusively) or larger than amax (inclusively).
        """
    @classmethod
    def new_area_sum_filter(cls, input: CompoundRegionOperationNode, inverse: Optional[bool] = ..., amin: Optional[int] = ..., amax: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering the input by area sum.
        Like \new_area_filter, but applies to the sum of all shapes in the current set.
        """
    @classmethod
    def new_bbox_filter(cls, input: CompoundRegionOperationNode, parameter: CompoundRegionOperationNode.ParameterType, inverse: Optional[bool] = ..., pmin: Optional[int] = ..., pmax: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering the input by bounding box parameters.
        This node renders the input if the specified bounding box parameter of the input shape is between pmin and pmax (exclusively). If 'inverse' is set to true, the input shape is returned if the parameter is less than pmin (exclusively) or larger than pmax (inclusively).
        """
    @classmethod
    def new_case(cls, inputs: Sequence[CompoundRegionOperationNode]) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a 'switch ladder' (case statement) compound operation node.

        The inputs are treated as a sequence of condition/result pairs: c1,r1,c2,r2 etc. If there is an odd number of inputs, the last element is taken as the default result. The implementation will evaluate c1 and if not empty, will render r1. Otherwise, c2 will be evaluated and r2 rendered if c2 isn't empty etc. If none of the conditions renders a non-empty set and a default result is present, the default will be returned. Otherwise, the result is empty.
        """
    @classmethod
    def new_centers(cls, input: CompoundRegionOperationNode, length: int, fraction: float) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node delivering a part at the center of each input edge.
        """
    @classmethod
    def new_convex_decomposition(cls, input: CompoundRegionOperationNode, mode: PreferredOrientation) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a composition into convex pieces.
        """
    @classmethod
    def new_corners_as_dots(cls, input: CompoundRegionOperationNode, angle_min: float, include_angle_min: bool, angle_max: float, include_angle_max: bool) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node turning corners into dots (single-point edges).
        """
    @classmethod
    def new_corners_as_edge_pairs(cls, input: CompoundRegionOperationNode, angle_min: float, include_angle_min: bool, angle_max: float, include_angle_max: bool) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node turning corners into edge pairs containing the two edges adjacent to the corner.
        The first edge will be the incoming edge and the second one the outgoing edge.

        This feature has been introduced in version 0.27.1.
        """
    @classmethod
    def new_corners_as_rectangles(cls, input: CompoundRegionOperationNode, angle_min: float, include_angle_min: bool, angle_max: float, include_angle_max: bool, dim: int) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node turning corners into rectangles.
        """
    @classmethod
    def new_count_filter(cls, inputs: CompoundRegionOperationNode, invert: Optional[bool] = ..., min_count: Optional[int] = ..., max_count: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node selecting results but their shape count.
        """
    @classmethod
    def new_edge_length_filter(cls, input: CompoundRegionOperationNode, inverse: Optional[bool] = ..., lmin: Optional[int] = ..., lmax: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering edges by their length.
        """
    @classmethod
    def new_edge_length_sum_filter(cls, input: CompoundRegionOperationNode, inverse: Optional[bool] = ..., lmin: Optional[int] = ..., lmax: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering edges by their length sum (over the local set).
        """
    @classmethod
    def new_edge_orientation_filter(cls, input: CompoundRegionOperationNode, inverse: bool, amin: float, include_amin: bool, amax: float, include_amax: bool) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering edges by their orientation.
        """
    @classmethod
    def new_edge_pair_to_first_edges(cls, input: CompoundRegionOperationNode) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node delivering the first edge of each edges pair.
        """
    @classmethod
    def new_edge_pair_to_second_edges(cls, input: CompoundRegionOperationNode) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node delivering the second edge of each edges pair.
        """
    @classmethod
    def new_edges(cls, input: CompoundRegionOperationNode) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node converting polygons into it's edges.
        """
    @classmethod
    def new_empty(cls, type: CompoundRegionOperationNode.ResultType) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node delivering an empty result of the given type
        """
    @classmethod
    def new_enclosed_check(cls, other: CompoundRegionOperationNode, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Region.Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ..., shielded: Optional[bool] = ..., opposite_filter: Optional[Region.OppositeFilter] = ..., rect_filter: Optional[Region.RectFilter] = ..., negative: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing an enclosed (secondary enclosing primary) check.

        This method has been added in version 0.27.5.
        """
    @classmethod
    def new_enclosing(cls, a: CompoundRegionOperationNode, b: CompoundRegionOperationNode, inverse: Optional[bool] = ..., min_count: Optional[int] = ..., max_count: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node representing an inside selection operation between the inputs.
        """
    @classmethod
    def new_enclosing_check(cls, other: CompoundRegionOperationNode, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Region.Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ..., shielded: Optional[bool] = ..., opposite_filter: Optional[Region.OppositeFilter] = ..., rect_filter: Optional[Region.RectFilter] = ..., negative: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing an inside (enclosure) check.
        """
    @classmethod
    def new_end_segments(cls, input: CompoundRegionOperationNode, length: int, fraction: float) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node delivering a part at the end of each input edge.
        """
    @classmethod
    def new_extended(cls, input: CompoundRegionOperationNode, ext_b: int, ext_e: int, ext_o: int, ext_i: int) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node delivering a polygonized version of the edges with the four extension parameters.
        """
    @classmethod
    def new_extended_in(cls, input: CompoundRegionOperationNode, e: int) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node delivering a polygonized, inside-extended version of the edges.
        """
    @classmethod
    def new_extended_out(cls, input: CompoundRegionOperationNode, e: int) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node delivering a polygonized, inside-extended version of the edges.
        """
    @classmethod
    def new_extents(cls, input: CompoundRegionOperationNode, e: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node returning the extents of the objects.
        The 'e' parameter provides a generic enlargement which is applied to the boxes. This is helpful to cover dot-like edges or edge pairs in the input.
        """
    @classmethod
    def new_foreign(cls) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node object representing the primary input without the current polygon
        """
    @classmethod
    def new_geometrical_boolean(cls, op: CompoundRegionOperationNode.GeometricalOp, a: CompoundRegionOperationNode, b: CompoundRegionOperationNode) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node representing a geometrical boolean operation between the inputs.
        """
    @classmethod
    def new_hole_count_filter(cls, input: CompoundRegionOperationNode, inverse: Optional[bool] = ..., hmin: Optional[int] = ..., hmax: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering the input by number of holes per polygon.
        This node renders the input if the hole count is between hmin and hmax (exclusively). If 'inverse' is set to true, the input shape is returned if the hole count is less than hmin (exclusively) or larger than hmax (inclusively).
        """
    @classmethod
    def new_holes(cls, input: CompoundRegionOperationNode) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node extracting the holes from polygons.
        """
    @classmethod
    def new_hulls(cls, input: CompoundRegionOperationNode) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node extracting the hulls from polygons.
        """
    @classmethod
    def new_inside(cls, a: CompoundRegionOperationNode, b: CompoundRegionOperationNode, inverse: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node representing an inside selection operation between the inputs.
        """
    @classmethod
    def new_interacting(cls, a: CompoundRegionOperationNode, b: CompoundRegionOperationNode, inverse: Optional[bool] = ..., min_count: Optional[int] = ..., max_count: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node representing an interacting selection operation between the inputs.
        """
    @classmethod
    def new_isolated_check(cls, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Region.Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ..., shielded: Optional[bool] = ..., opposite_filter: Optional[Region.OppositeFilter] = ..., rect_filter: Optional[Region.RectFilter] = ..., negative: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a isolated polygons (space between different polygons) check.
        """
    @classmethod
    def new_join(cls, inputs: Sequence[CompoundRegionOperationNode]) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node that joins the inputs.
        """
    @classmethod
    def new_logical_boolean(cls, op: CompoundRegionOperationNode.LogicalOp, invert: bool, inputs: Sequence[CompoundRegionOperationNode]) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node representing a logical boolean operation between the inputs.

        A logical AND operation will evaluate the arguments and render the subject shape when all arguments are non-empty. The logical OR operation will evaluate the arguments and render the subject shape when one argument is non-empty. Setting 'inverse' to true will reverse the result and return the subject shape when one argument is empty in the AND case and when all arguments are empty in the OR case.
        """
    @classmethod
    def new_merged(cls, input: CompoundRegionOperationNode, min_coherence: Optional[bool] = ..., min_wc: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing merged input polygons.
        """
    @overload
    @classmethod
    def new_minkowski_sum(cls, input: CompoundRegionOperationNode, e: Edge) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a Minkowski sum with an edge.
        """
    @overload
    @classmethod
    def new_minkowski_sum(cls, input: CompoundRegionOperationNode, p: Box) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a Minkowski sum with a box.
        """
    @overload
    @classmethod
    def new_minkowski_sum(cls, input: CompoundRegionOperationNode, p: Polygon) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a Minkowski sum with a polygon.
        """
    @overload
    @classmethod
    def new_minkowski_sum(cls, input: CompoundRegionOperationNode, p: Sequence[Point]) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a Minkowski sum with a point sequence forming a contour.
        """
    @overload
    @classmethod
    def new_minkowsky_sum(cls, input: CompoundRegionOperationNode, e: Edge) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a Minkowski sum with an edge.
        """
    @overload
    @classmethod
    def new_minkowsky_sum(cls, input: CompoundRegionOperationNode, p: Box) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a Minkowski sum with a box.
        """
    @overload
    @classmethod
    def new_minkowsky_sum(cls, input: CompoundRegionOperationNode, p: Polygon) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a Minkowski sum with a polygon.
        """
    @overload
    @classmethod
    def new_minkowsky_sum(cls, input: CompoundRegionOperationNode, p: Sequence[Point]) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a Minkowski sum with a point sequence forming a contour.
        """
    @classmethod
    def new_notch_check(cls, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Region.Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ..., shielded: Optional[bool] = ..., negative: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a intra-polygon space check.
        """
    @classmethod
    def new_outside(cls, a: CompoundRegionOperationNode, b: CompoundRegionOperationNode, inverse: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node representing an outside selection operation between the inputs.
        """
    @classmethod
    def new_overlap_check(cls, other: CompoundRegionOperationNode, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Region.Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ..., shielded: Optional[bool] = ..., opposite_filter: Optional[Region.OppositeFilter] = ..., rect_filter: Optional[Region.RectFilter] = ..., negative: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing an overlap check.
        """
    @classmethod
    def new_overlapping(cls, a: CompoundRegionOperationNode, b: CompoundRegionOperationNode, inverse: Optional[bool] = ..., min_count: Optional[int] = ..., max_count: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node representing an overlapping selection operation between the inputs.
        """
    @classmethod
    def new_perimeter_filter(cls, input: CompoundRegionOperationNode, inverse: Optional[bool] = ..., pmin: Optional[int] = ..., pmax: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering the input by perimeter.
        This node renders the input if the perimeter is between pmin and pmax (exclusively). If 'inverse' is set to true, the input shape is returned if the perimeter is less than pmin (exclusively) or larger than pmax (inclusively).
        """
    @classmethod
    def new_perimeter_sum_filter(cls, input: CompoundRegionOperationNode, inverse: Optional[bool] = ..., amin: Optional[int] = ..., amax: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering the input by area sum.
        Like \new_perimeter_filter, but applies to the sum of all shapes in the current set.
        """
    @classmethod
    def new_polygon_breaker(cls, input: CompoundRegionOperationNode, max_vertex_count: int, max_area_ratio: float) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a composition into parts with less than the given number of points and a smaller area ratio.
        """
    @classmethod
    def new_polygons(cls, input: CompoundRegionOperationNode, e: Optional[int] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node converting the input to polygons.
        @param e The enlargement parameter when converting edges or edge pairs to polygons.
        """
    @classmethod
    def new_primary(cls) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node object representing the primary input
        """
    @classmethod
    def new_ratio_filter(cls, input: CompoundRegionOperationNode, parameter: CompoundRegionOperationNode.RatioParameterType, inverse: Optional[bool] = ..., pmin: Optional[float] = ..., pmin_included: Optional[bool] = ..., pmax: Optional[float] = ..., pmax_included: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering the input by ratio parameters.
        This node renders the input if the specified ratio parameter of the input shape is between pmin and pmax. If 'pmin_included' is true, the range will include pmin. Same for 'pmax_included' and pmax. If 'inverse' is set to true, the input shape is returned if the parameter is not within the specified range.
        """
    @classmethod
    def new_rectangle_filter(cls, input: CompoundRegionOperationNode, is_square: Optional[bool] = ..., inverse: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering the input for rectangular or square shapes.
        If 'is_square' is true, only squares will be selected. If 'inverse' is true, the non-rectangle/non-square shapes are returned.
        """
    @classmethod
    def new_rectilinear_filter(cls, input: CompoundRegionOperationNode, inverse: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node filtering the input for rectilinear shapes (or non-rectilinear ones with 'inverse' set to 'true').
        """
    @classmethod
    def new_relative_extents(cls, input: CompoundRegionOperationNode, fx1: float, fy1: float, fx2: float, fy2: float, dx: int, dy: int) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node returning markers at specified locations of the extent (e.g. at the center).
        """
    @classmethod
    def new_relative_extents_as_edges(cls, input: CompoundRegionOperationNode, fx1: float, fy1: float, fx2: float, fy2: float) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node returning edges at specified locations of the extent (e.g. at the center).
        """
    @classmethod
    def new_rounded_corners(cls, input: CompoundRegionOperationNode, rinner: float, router: float, n: int) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node generating rounded corners.
        @param rinner The inner corner radius.@param router The outer corner radius.@param n The number if points per full circle.
        """
    @classmethod
    def new_secondary(cls, region: Region) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node object representing the secondary input from the given region
        """
    @classmethod
    def new_separation_check(cls, other: CompoundRegionOperationNode, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Region.Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ..., shielded: Optional[bool] = ..., opposite_filter: Optional[Region.OppositeFilter] = ..., rect_filter: Optional[Region.RectFilter] = ..., negative: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a separation check.
        """
    @classmethod
    def new_sized(cls, input: CompoundRegionOperationNode, dx: int, dy: int, mode: int) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing sizing.
        """
    @classmethod
    def new_smoothed(cls, input: CompoundRegionOperationNode, d: int, keep_hv: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node smoothing the polygons.
        @param d The tolerance to be applied for the smoothing.
        @param keep_hv If true, horizontal and vertical edges are maintained.
        """
    @classmethod
    def new_space_check(cls, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Region.Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ..., shielded: Optional[bool] = ..., opposite_filter: Optional[Region.OppositeFilter] = ..., rect_filter: Optional[Region.RectFilter] = ..., negative: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a space check.
        """
    @classmethod
    def new_start_segments(cls, input: CompoundRegionOperationNode, length: int, fraction: float) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node delivering a part at the beginning of each input edge.
        """
    @classmethod
    def new_strange_polygons_filter(cls, input: CompoundRegionOperationNode) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node extracting strange polygons.
        'strange polygons' are ones which cannot be oriented - e.g. '8' shape polygons.
        """
    @classmethod
    def new_trapezoid_decomposition(cls, input: CompoundRegionOperationNode, mode: TrapezoidDecompositionMode) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a composition into trapezoids.
        """
    @classmethod
    def new_width_check(cls, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Region.Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ..., shielded: Optional[bool] = ..., negative: Optional[bool] = ...) -> CompoundRegionOperationNode:
        r"""
        @brief Creates a node providing a width check.
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def result_type(self) -> CompoundRegionOperationNode.ResultType:
        r"""
        @brief Gets the result type of this node
        """

class TrapezoidDecompositionMode:
    r"""
    @brief This class represents the TrapezoidDecompositionMode enum used within trapezoid decomposition

    This enum has been introduced in version 0.27.
    """
    TD_htrapezoids: ClassVar[TrapezoidDecompositionMode]
    r"""
    @brief Indicates horizontal trapezoid decomposition.
    """
    TD_simple: ClassVar[TrapezoidDecompositionMode]
    r"""
    @brief Indicates unspecific decomposition.
    """
    TD_vtrapezoids: ClassVar[TrapezoidDecompositionMode]
    r"""
    @brief Indicates vertical trapezoid decomposition.
    """
    @overload
    @classmethod
    def new(cls, i: int) -> TrapezoidDecompositionMode:
        r"""
        @brief Creates an enum from an integer value
        """
    @overload
    @classmethod
    def new(cls, s: str) -> TrapezoidDecompositionMode:
        r"""
        @brief Creates an enum from a string value
        """
    def __copy__(self) -> TrapezoidDecompositionMode:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> TrapezoidDecompositionMode:
        r"""
        @brief Creates a copy of self
        """
    @overload
    def __eq__(self, other: object) -> bool:
        r"""
        @brief Compares an enum with an integer value
        """
    @overload
    def __eq__(self, other: object) -> bool:
        r"""
        @brief Compares two enums
        """
    @overload
    def __init__(self, i: int) -> None:
        r"""
        @brief Creates an enum from an integer value
        """
    @overload
    def __init__(self, s: str) -> None:
        r"""
        @brief Creates an enum from a string value
        """
    @overload
    def __lt__(self, other: int) -> bool:
        r"""
        @brief Returns true if the enum is less (in the enum symbol order) than the integer value
        """
    @overload
    def __lt__(self, other: TrapezoidDecompositionMode) -> bool:
        r"""
        @brief Returns true if the first enum is less (in the enum symbol order) than the second
        """
    @overload
    def __ne__(self, other: object) -> bool:
        r"""
        @brief Compares an enum with an integer for inequality
        """
    @overload
    def __ne__(self, other: object) -> bool:
        r"""
        @brief Compares two enums for inequality
        """
    def __repr__(self) -> str:
        r"""
        @brief Converts an enum to a visual string
        """
    def __str__(self) -> str:
        r"""
        @brief Gets the symbolic string from an enum
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: TrapezoidDecompositionMode) -> None:
        r"""
        @brief Assigns another object to self
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> TrapezoidDecompositionMode:
        r"""
        @brief Creates a copy of self
        """
    def inspect(self) -> str:
        r"""
        @brief Converts an enum to a visual string
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def to_i(self) -> int:
        r"""
        @brief Gets the integer value from the enum
        """
    def to_s(self) -> str:
        r"""
        @brief Gets the symbolic string from an enum
        """

class PreferredOrientation:
    r"""
    @brief This class represents the PreferredOrientation enum used within polygon decomposition

    This enum has been introduced in version 0.27.
    """
    PO_any: ClassVar[PreferredOrientation]
    r"""
    @brief Indicates any orientation.
    """
    PO_horizontal: ClassVar[PreferredOrientation]
    r"""
    @brief Indicates horizontal orientation.
    """
    PO_htrapezoids: ClassVar[PreferredOrientation]
    r"""
    @brief Indicates horizontal trapezoid decomposition.
    """
    PO_vertical: ClassVar[PreferredOrientation]
    r"""
    @brief Indicates vertical orientation.
    """
    PO_vtrapezoids: ClassVar[PreferredOrientation]
    r"""
    @brief Indicates vertical trapezoid decomposition.
    """
    @overload
    @classmethod
    def new(cls, i: int) -> PreferredOrientation:
        r"""
        @brief Creates an enum from an integer value
        """
    @overload
    @classmethod
    def new(cls, s: str) -> PreferredOrientation:
        r"""
        @brief Creates an enum from a string value
        """
    def __copy__(self) -> PreferredOrientation:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> PreferredOrientation:
        r"""
        @brief Creates a copy of self
        """
    @overload
    def __eq__(self, other: object) -> bool:
        r"""
        @brief Compares an enum with an integer value
        """
    @overload
    def __eq__(self, other: object) -> bool:
        r"""
        @brief Compares two enums
        """
    @overload
    def __init__(self, i: int) -> None:
        r"""
        @brief Creates an enum from an integer value
        """
    @overload
    def __init__(self, s: str) -> None:
        r"""
        @brief Creates an enum from a string value
        """
    @overload
    def __lt__(self, other: int) -> bool:
        r"""
        @brief Returns true if the enum is less (in the enum symbol order) than the integer value
        """
    @overload
    def __lt__(self, other: PreferredOrientation) -> bool:
        r"""
        @brief Returns true if the first enum is less (in the enum symbol order) than the second
        """
    @overload
    def __ne__(self, other: object) -> bool:
        r"""
        @brief Compares an enum with an integer for inequality
        """
    @overload
    def __ne__(self, other: object) -> bool:
        r"""
        @brief Compares two enums for inequality
        """
    def __repr__(self) -> str:
        r"""
        @brief Converts an enum to a visual string
        """
    def __str__(self) -> str:
        r"""
        @brief Gets the symbolic string from an enum
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: PreferredOrientation) -> None:
        r"""
        @brief Assigns another object to self
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> PreferredOrientation:
        r"""
        @brief Creates a copy of self
        """
    def inspect(self) -> str:
        r"""
        @brief Converts an enum to a visual string
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def to_i(self) -> int:
        r"""
        @brief Gets the integer value from the enum
        """
    def to_s(self) -> str:
        r"""
        @brief Gets the symbolic string from an enum
        """

class Edge:
    r"""
    @brief An edge class

    An edge is a connection between points, usually participating in a larger context such as a polygon. An edge has a defined direction (from p1 to p2). Edges play a role in the database as parts of polygons and to describe a line through both points.
    Although supported, edges are rarely used as individual database objects.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects like the Edge class.
    """
    p1: Point
    r"""
    Getter:
    @brief The first point.

    Setter:
    @brief Sets the first point.
    This method has been added in version 0.23.
    """
    p2: Point
    r"""
    Getter:
    @brief The second point.

    Setter:
    @brief Sets the second point.
    This method has been added in version 0.23.
    """
    x1: int
    r"""
    Getter:
    @brief Shortcut for p1.x

    Setter:
    @brief Sets p1.x
    This method has been added in version 0.23.
    """
    x2: int
    r"""
    Getter:
    @brief Shortcut for p2.x

    Setter:
    @brief Sets p2.x
    This method has been added in version 0.23.
    """
    y1: int
    r"""
    Getter:
    @brief Shortcut for p1.y

    Setter:
    @brief Sets p1.y
    This method has been added in version 0.23.
    """
    y2: int
    r"""
    Getter:
    @brief Shortcut for p2.y

    Setter:
    @brief Sets p2.y
    This method has been added in version 0.23.
    """
    @classmethod
    def from_dedge(cls, dedge: DEdge) -> Edge:
        r"""
        @brief Creates an integer coordinate edge from a floating-point coordinate edge

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dedge'.
        """
    @classmethod
    def from_s(cls, s: str) -> Edge:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> Edge:
        r"""
        @brief Default constructor: creates a degenerated edge 0,0 to 0,0
        """
    @overload
    @classmethod
    def new(cls, dedge: DEdge) -> Edge:
        r"""
        @brief Creates an integer coordinate edge from a floating-point coordinate edge

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dedge'.
        """
    @overload
    @classmethod
    def new(cls, p1: Point, p2: Point) -> Edge:
        r"""
        @brief Constructor with two points

        Two points are given to create a new edge.
        """
    @overload
    @classmethod
    def new(cls, x1: int, y1: int, x2: int, y2: int) -> Edge:
        r"""
        @brief Constructor with two coordinates given as single values

        Two points are given to create a new edge.
        """
    @classmethod
    def new_pp(cls, p1: Point, p2: Point) -> Edge:
        r"""
        @brief Constructor with two points

        Two points are given to create a new edge.
        """
    @classmethod
    def new_xyxy(cls, x1: int, y1: int, x2: int, y2: int) -> Edge:
        r"""
        @brief Constructor with two coordinates given as single values

        Two points are given to create a new edge.
        """
    def __copy__(self) -> Edge:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> Edge:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, e: object) -> bool:
        r"""
        @brief Equality test
        @param e The object to compare against
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given edge. This method enables edges as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor: creates a degenerated edge 0,0 to 0,0
        """
    @overload
    def __init__(self, dedge: DEdge) -> None:
        r"""
        @brief Creates an integer coordinate edge from a floating-point coordinate edge

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dedge'.
        """
    @overload
    def __init__(self, p1: Point, p2: Point) -> None:
        r"""
        @brief Constructor with two points

        Two points are given to create a new edge.
        """
    @overload
    def __init__(self, x1: int, y1: int, x2: int, y2: int) -> None:
        r"""
        @brief Constructor with two coordinates given as single values

        Two points are given to create a new edge.
        """
    def __lt__(self, e: Edge) -> bool:
        r"""
        @brief Less operator
        @param e The object to compare against
        @return True, if the edge is 'less' as the other edge with respect to first and second point
        """
    def __mul__(self, scale_factor: float) -> Edge:
        r"""
        @brief Scale edge

        The * operator scales self with the given factor.

        This method has been introduced in version 0.22.

        @param scale_factor The scaling factor

        @return The scaled edge
        """
    def __ne__(self, e: object) -> bool:
        r"""
        @brief Inequality test
        @param e The object to compare against
        """
    def __rmul__(self, scale_factor: float) -> Edge:
        r"""
        @brief Scale edge

        The * operator scales self with the given factor.

        This method has been introduced in version 0.22.

        @param scale_factor The scaling factor

        @return The scaled edge
        """
    def __str__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: Edge) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> Box:
        r"""
        @brief Return the bounding box of the edge.
        """
    def clipped(self, box: Box) -> Any:
        r"""
        @brief Returns the edge clipped at the given box

        @param box The clip box.
        @return The clipped edge or nil if the edge does not intersect with the box.

        This method has been introduced in version 0.26.2.
        """
    def clipped_line(self, box: Box) -> Any:
        r"""
        @brief Returns the line through the edge clipped at the given box

        @param box The clip box.
        @return The part of the line through the box or nil if the line does not intersect with the box.

        In contrast to \clipped, this method will consider the edge extended infinitely (a "line"). The returned edge will be the part of this line going through the box.

        This method has been introduced in version 0.26.2.
        """
    def coincident(self, e: Edge) -> bool:
        r"""
        @brief Coincidence check.

        Checks whether a edge is coincident with another edge. 
        Coincidence is defined by being parallel and that 
        at least one point of one edge is on the other edge.

        @param e the edge to test with

        @return True if the edges are coincident.
        """
    def contains(self, p: Point) -> bool:
        r"""
        @brief Test whether a point is on an edge.

        A point is on a edge if it is on (or at least closer 
        than a grid point to) the edge.

        @param p The point to test with the edge.

        @return True if the point is on the edge.
        """
    def contains_excl(self, p: Point) -> bool:
        r"""
        @brief Test whether a point is on an edge excluding the endpoints.

        A point is on a edge if it is on (or at least closer 
        than a grid point to) the edge.

        @param p The point to test with the edge.

        @return True if the point is on the edge but not equal p1 or p2.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def crossed_by(self, e: Edge) -> bool:
        r"""
        @brief Check, if an edge is cut by a line (given by an edge)

        This method returns true if p1 is in one semispace 
        while p2 is in the other or one of them is on the line
        through the edge "e"

        @param e The edge representing the line that the edge must be crossing.
        """
    def crossing_point(self, e: Edge) -> Point:
        r"""
        @brief Returns the crossing point on two edges. 

        This method delivers the point where the given edge (self) crosses the line given by the edge in argument "e". If self does not cross this line, the result is undefined. See \crossed_by? for a description of the crossing predicate.

        @param e The edge representing the line that self must be crossing.
        @return The point where self crosses the line given by "e".

        This method has been introduced in version 0.19.
        """
    def cut_point(self, e: Edge) -> Any:
        r"""
        @brief Returns the intersection point of the lines through the two edges.

        This method delivers the intersection point between the lines through the two edges. If the lines are parallel and do not intersect, the result will be nil.
        In contrast to \intersection_point, this method will regard the edges as infinitely extended and intersection is not confined to the edge span.

        @param e The edge to test.
        @return The point where the lines intersect.

        This method has been introduced in version 0.27.1.
        """
    def d(self) -> Vector:
        r"""
        @brief Gets the edge extension as a vector.
        This method is equivalent to p2 - p1.
        This method has been introduced in version 0.26.2.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def distance(self, p: Point) -> int:
        r"""
        @brief Distance between the edge and a point.

        Returns the distance between the edge and the point. The 
        distance is signed which is negative if the point is to the
        "right" of the edge and positive if the point is to the "left".
        The distance is measured by projecting the point onto the
        line through the edge. If the edge is degenerated, the distance
        is not defined.

        @param p The point to test.

        @return The distance
        """
    def distance_abs(self, p: Point) -> int:
        r"""
        @brief Absolute distance between the edge and a point.

        Returns the distance between the edge and the point. 

        @param p The point to test.

        @return The distance
        """
    def dup(self) -> Edge:
        r"""
        @brief Creates a copy of self
        """
    def dx(self) -> int:
        r"""
        @brief The horizontal extend of the edge.
        """
    def dx_abs(self) -> int:
        r"""
        @brief The absolute value of the horizontal extend of the edge.
        """
    def dy(self) -> int:
        r"""
        @brief The vertical extend of the edge.
        """
    def dy_abs(self) -> int:
        r"""
        @brief The absolute value of the vertical extend of the edge.
        """
    def enlarge(self, p: Vector) -> Edge:
        r"""
        @brief Enlarges the edge.

        Enlarges the edge by the given distance and returns the 
        enlarged edge. The edge is overwritten.
        Enlargement means
        that the first point is shifted by -p, the second by p.

        @param p The distance to move the edge points.

        @return The enlarged edge.
        """
    def enlarged(self, p: Vector) -> Edge:
        r"""
        @brief Returns the enlarged edge (does not modify self)

        Enlarges the edge by the given offset and returns the 
        enlarged edge. The edge is not modified. Enlargement means
        that the first point is shifted by -p, the second by p.

        @param p The distance to move the edge points.

        @return The enlarged edge.
        """
    def extend(self, d: int) -> Edge:
        r"""
        @brief Extends the edge (modifies self)

        Extends the edge by the given distance and returns the 
        extended edge. The edge is not modified. Extending means
        that the first point is shifted by -d along the edge, the second by d.
        The length of the edge will increase by 2*d.

        \extended is a version that does not modify self but returns the extended edges.

        This method has been introduced in version 0.23.

        @param d The distance by which to shift the end points.

        @return The extended edge (self).
        """
    def extended(self, d: int) -> Edge:
        r"""
        @brief Returns the extended edge (does not modify self)

        Extends the edge by the given distance and returns the 
        extended edge. The edge is not modified. Extending means
        that the first point is shifted by -d along the edge, the second by d.
        The length of the edge will increase by 2*d.

        \extend is a version that modifies self (in-place).

        This method has been introduced in version 0.23.

        @param d The distance by which to shift the end points.

        @return The extended edge.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given edge. This method enables edges as hash keys.

        This method has been introduced in version 0.25.
        """
    def intersect(self, e: Edge) -> bool:
        r"""
        @brief Intersection test. 

        Returns true if the edges intersect. Two edges intersect if they share at least one point. 
        If the edges coincide, they also intersect.
        For degenerated edges, the intersection is mapped to
        point containment tests.

        @param e The edge to test.
        """
    def intersection_point(self, e: Edge) -> Any:
        r"""
        @brief Returns the intersection point of two edges. 

        This method delivers the intersection point. If the edges do not intersect, the result will be nil.

        @param e The edge to test.
        @return The point where the edges intersect.

        This method has been introduced in version 0.19.
        From version 0.26.2, this method will return nil in case of non-intersection.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_degenerate(self) -> bool:
        r"""
        @brief Test for degenerated edge

        An edge is degenerate, if both end and start point are identical.
        """
    def is_parallel(self, e: Edge) -> bool:
        r"""
        @brief Test for being parallel

        @param e The edge to test against

        @return True if both edges are parallel
        """
    def length(self) -> int:
        r"""
        @brief The length of the edge
        """
    @overload
    def move(self, p: Vector) -> Edge:
        r"""
        @brief Moves the edge.

        Moves the edge by the given offset and returns the 
        moved edge. The edge is overwritten.

        @param p The distance to move the edge.

        @return The moved edge.
        """
    @overload
    def move(self, dx: int, dy: int) -> Edge:
        r"""
        @brief Moves the edge.

        Moves the edge by the given offset and returns the 
        moved edge. The edge is overwritten.

        @param dx The x distance to move the edge.
        @param dy The y distance to move the edge.

        @return The moved edge.

        This version has been added in version 0.23.
        """
    @overload
    def moved(self, p: Vector) -> Edge:
        r"""
        @brief Returns the moved edge (does not modify self)

        Moves the edge by the given offset and returns the 
        moved edge. The edge is not modified.

        @param p The distance to move the edge.

        @return The moved edge.
        """
    @overload
    def moved(self, dx: int, dy: int) -> Edge:
        r"""
        @brief Returns the moved edge (does not modify self)

        Moves the edge by the given offset and returns the 
        moved edge. The edge is not modified.

        @param dx The x distance to move the edge.
        @param dy The y distance to move the edge.

        @return The moved edge.

        This version has been added in version 0.23.
        """
    def ortho_length(self) -> int:
        r"""
        @brief The orthogonal length of the edge ("manhattan-length")

        @return The orthogonal length (abs(dx)+abs(dy))
        """
    def shift(self, d: int) -> Edge:
        r"""
        @brief Shifts the edge (modifies self)

        Shifts the edge by the given distance and returns the 
        shifted edge. The edge is not modified. Shifting by a positive value will produce an edge which is shifted by d to the left. Shifting by a negative value will produce an edge which is shifted by d to the right.

        \shifted is a version that does not modify self but returns the extended edges.

        This method has been introduced in version 0.23.

        @param d The distance by which to shift the edge.

        @return The shifted edge (self).
        """
    def shifted(self, d: int) -> Edge:
        r"""
        @brief Returns the shifted edge (does not modify self)

        Shifts the edge by the given distance and returns the 
        shifted edge. The edge is not modified. Shifting by a positive value will produce an edge which is shifted by d to the left. Shifting by a negative value will produce an edge which is shifted by d to the right.

        \shift is a version that modifies self (in-place).

        This method has been introduced in version 0.23.

        @param d The distance by which to shift the edge.

        @return The shifted edge.
        """
    def side_of(self, p: Point) -> int:
        r"""
        @brief Indicates at which side the point is located relative to the edge.

        Returns 1 if the point is "left" of the edge, 0 if on
        and -1 if the point is "right" of the edge.

        @param p The point to test.

        @return The side value
        """
    def sq_length(self) -> int:
        r"""
        @brief The square of the length of the edge
        """
    def swap_points(self) -> Edge:
        r"""
        @brief Swap the points of the edge

        This version modifies self. A version that does not modify self is \swapped_points. Swapping the points basically reverses the direction of the edge.

        This method has been introduced in version 0.23.
        """
    def swapped_points(self) -> Edge:
        r"""
        @brief Returns an edge in which both points are swapped

        Swapping the points basically reverses the direction of the edge.

        This method has been introduced in version 0.23.
        """
    def to_dtype(self, dbu: Optional[float] = ...) -> DEdge:
        r"""
        @brief Converts the edge to a floating-point coordinate edge

        The database unit can be specified to translate the integer-coordinate edge into a floating-point coordinate edge in micron units. The database unit is basically a scaling factor.

        This method has been introduced in version 0.25.
        """
    def to_s(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    @overload
    def transformed(self, t: CplxTrans) -> DEdge:
        r"""
        @brief Transform the edge.

        Transforms the edge with the given complex transformation.
        Does not modify the edge but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge.
        """
    @overload
    def transformed(self, t: ICplxTrans) -> Edge:
        r"""
        @brief Transform the edge.

        Transforms the edge with the given complex transformation.
        Does not modify the edge but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge (in this case an integer coordinate edge).

        This method has been introduced in version 0.18.
        """
    @overload
    def transformed(self, t: Trans) -> Edge:
        r"""
        @brief Transform the edge.

        Transforms the edge with the given transformation.
        Does not modify the edge but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge.
        """
    def transformed_cplx(self, t: CplxTrans) -> DEdge:
        r"""
        @brief Transform the edge.

        Transforms the edge with the given complex transformation.
        Does not modify the edge but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge.
        """

class DEdge:
    r"""
    @brief An edge class

    An edge is a connection between points, usually participating in a larger context such as a polygon. An edge has a defined direction (from p1 to p2). Edges play a role in the database as parts of polygons and to describe a line through both points.
    The \Edge object is also used inside the boolean processor (\EdgeProcessor).
    Although supported, edges are rarely used as individual database objects.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects like the Edge class.
    """
    p1: DPoint
    r"""
    Getter:
    @brief The first point.

    Setter:
    @brief Sets the first point.
    This method has been added in version 0.23.
    """
    p2: DPoint
    r"""
    Getter:
    @brief The second point.

    Setter:
    @brief Sets the second point.
    This method has been added in version 0.23.
    """
    x1: float
    r"""
    Getter:
    @brief Shortcut for p1.x

    Setter:
    @brief Sets p1.x
    This method has been added in version 0.23.
    """
    x2: float
    r"""
    Getter:
    @brief Shortcut for p2.x

    Setter:
    @brief Sets p2.x
    This method has been added in version 0.23.
    """
    y1: float
    r"""
    Getter:
    @brief Shortcut for p1.y

    Setter:
    @brief Sets p1.y
    This method has been added in version 0.23.
    """
    y2: float
    r"""
    Getter:
    @brief Shortcut for p2.y

    Setter:
    @brief Sets p2.y
    This method has been added in version 0.23.
    """
    @classmethod
    def from_iedge(cls, edge: Edge) -> DEdge:
        r"""
        @brief Creates a floating-point coordinate edge from an integer coordinate edge

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_iedge'.
        """
    @classmethod
    def from_s(cls, s: str) -> DEdge:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> DEdge:
        r"""
        @brief Default constructor: creates a degenerated edge 0,0 to 0,0
        """
    @overload
    @classmethod
    def new(cls, edge: Edge) -> DEdge:
        r"""
        @brief Creates a floating-point coordinate edge from an integer coordinate edge

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_iedge'.
        """
    @overload
    @classmethod
    def new(cls, p1: DPoint, p2: DPoint) -> DEdge:
        r"""
        @brief Constructor with two points

        Two points are given to create a new edge.
        """
    @overload
    @classmethod
    def new(cls, x1: float, y1: float, x2: float, y2: float) -> DEdge:
        r"""
        @brief Constructor with two coordinates given as single values

        Two points are given to create a new edge.
        """
    @classmethod
    def new_pp(cls, p1: DPoint, p2: DPoint) -> DEdge:
        r"""
        @brief Constructor with two points

        Two points are given to create a new edge.
        """
    @classmethod
    def new_xyxy(cls, x1: float, y1: float, x2: float, y2: float) -> DEdge:
        r"""
        @brief Constructor with two coordinates given as single values

        Two points are given to create a new edge.
        """
    def __copy__(self) -> DEdge:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DEdge:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, e: object) -> bool:
        r"""
        @brief Equality test
        @param e The object to compare against
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given edge. This method enables edges as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor: creates a degenerated edge 0,0 to 0,0
        """
    @overload
    def __init__(self, edge: Edge) -> None:
        r"""
        @brief Creates a floating-point coordinate edge from an integer coordinate edge

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_iedge'.
        """
    @overload
    def __init__(self, p1: DPoint, p2: DPoint) -> None:
        r"""
        @brief Constructor with two points

        Two points are given to create a new edge.
        """
    @overload
    def __init__(self, x1: float, y1: float, x2: float, y2: float) -> None:
        r"""
        @brief Constructor with two coordinates given as single values

        Two points are given to create a new edge.
        """
    def __lt__(self, e: DEdge) -> bool:
        r"""
        @brief Less operator
        @param e The object to compare against
        @return True, if the edge is 'less' as the other edge with respect to first and second point
        """
    def __mul__(self, scale_factor: float) -> DEdge:
        r"""
        @brief Scale edge

        The * operator scales self with the given factor.

        This method has been introduced in version 0.22.

        @param scale_factor The scaling factor

        @return The scaled edge
        """
    def __ne__(self, e: object) -> bool:
        r"""
        @brief Inequality test
        @param e The object to compare against
        """
    def __rmul__(self, scale_factor: float) -> DEdge:
        r"""
        @brief Scale edge

        The * operator scales self with the given factor.

        This method has been introduced in version 0.22.

        @param scale_factor The scaling factor

        @return The scaled edge
        """
    def __str__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: DEdge) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> DBox:
        r"""
        @brief Return the bounding box of the edge.
        """
    def clipped(self, box: DBox) -> Any:
        r"""
        @brief Returns the edge clipped at the given box

        @param box The clip box.
        @return The clipped edge or nil if the edge does not intersect with the box.

        This method has been introduced in version 0.26.2.
        """
    def clipped_line(self, box: DBox) -> Any:
        r"""
        @brief Returns the line through the edge clipped at the given box

        @param box The clip box.
        @return The part of the line through the box or nil if the line does not intersect with the box.

        In contrast to \clipped, this method will consider the edge extended infinitely (a "line"). The returned edge will be the part of this line going through the box.

        This method has been introduced in version 0.26.2.
        """
    def coincident(self, e: DEdge) -> bool:
        r"""
        @brief Coincidence check.

        Checks whether a edge is coincident with another edge. 
        Coincidence is defined by being parallel and that 
        at least one point of one edge is on the other edge.

        @param e the edge to test with

        @return True if the edges are coincident.
        """
    def contains(self, p: DPoint) -> bool:
        r"""
        @brief Test whether a point is on an edge.

        A point is on a edge if it is on (or at least closer 
        than a grid point to) the edge.

        @param p The point to test with the edge.

        @return True if the point is on the edge.
        """
    def contains_excl(self, p: DPoint) -> bool:
        r"""
        @brief Test whether a point is on an edge excluding the endpoints.

        A point is on a edge if it is on (or at least closer 
        than a grid point to) the edge.

        @param p The point to test with the edge.

        @return True if the point is on the edge but not equal p1 or p2.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def crossed_by(self, e: DEdge) -> bool:
        r"""
        @brief Check, if an edge is cut by a line (given by an edge)

        This method returns true if p1 is in one semispace 
        while p2 is in the other or one of them is on the line
        through the edge "e"

        @param e The edge representing the line that the edge must be crossing.
        """
    def crossing_point(self, e: DEdge) -> DPoint:
        r"""
        @brief Returns the crossing point on two edges. 

        This method delivers the point where the given edge (self) crosses the line given by the edge in argument "e". If self does not cross this line, the result is undefined. See \crossed_by? for a description of the crossing predicate.

        @param e The edge representing the line that self must be crossing.
        @return The point where self crosses the line given by "e".

        This method has been introduced in version 0.19.
        """
    def cut_point(self, e: DEdge) -> Any:
        r"""
        @brief Returns the intersection point of the lines through the two edges.

        This method delivers the intersection point between the lines through the two edges. If the lines are parallel and do not intersect, the result will be nil.
        In contrast to \intersection_point, this method will regard the edges as infinitely extended and intersection is not confined to the edge span.

        @param e The edge to test.
        @return The point where the lines intersect.

        This method has been introduced in version 0.27.1.
        """
    def d(self) -> DVector:
        r"""
        @brief Gets the edge extension as a vector.
        This method is equivalent to p2 - p1.
        This method has been introduced in version 0.26.2.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def distance(self, p: DPoint) -> float:
        r"""
        @brief Distance between the edge and a point.

        Returns the distance between the edge and the point. The 
        distance is signed which is negative if the point is to the
        "right" of the edge and positive if the point is to the "left".
        The distance is measured by projecting the point onto the
        line through the edge. If the edge is degenerated, the distance
        is not defined.

        @param p The point to test.

        @return The distance
        """
    def distance_abs(self, p: DPoint) -> float:
        r"""
        @brief Absolute distance between the edge and a point.

        Returns the distance between the edge and the point. 

        @param p The point to test.

        @return The distance
        """
    def dup(self) -> DEdge:
        r"""
        @brief Creates a copy of self
        """
    def dx(self) -> float:
        r"""
        @brief The horizontal extend of the edge.
        """
    def dx_abs(self) -> float:
        r"""
        @brief The absolute value of the horizontal extend of the edge.
        """
    def dy(self) -> float:
        r"""
        @brief The vertical extend of the edge.
        """
    def dy_abs(self) -> float:
        r"""
        @brief The absolute value of the vertical extend of the edge.
        """
    def enlarge(self, p: DVector) -> DEdge:
        r"""
        @brief Enlarges the edge.

        Enlarges the edge by the given distance and returns the 
        enlarged edge. The edge is overwritten.
        Enlargement means
        that the first point is shifted by -p, the second by p.

        @param p The distance to move the edge points.

        @return The enlarged edge.
        """
    def enlarged(self, p: DVector) -> DEdge:
        r"""
        @brief Returns the enlarged edge (does not modify self)

        Enlarges the edge by the given offset and returns the 
        enlarged edge. The edge is not modified. Enlargement means
        that the first point is shifted by -p, the second by p.

        @param p The distance to move the edge points.

        @return The enlarged edge.
        """
    def extend(self, d: float) -> DEdge:
        r"""
        @brief Extends the edge (modifies self)

        Extends the edge by the given distance and returns the 
        extended edge. The edge is not modified. Extending means
        that the first point is shifted by -d along the edge, the second by d.
        The length of the edge will increase by 2*d.

        \extended is a version that does not modify self but returns the extended edges.

        This method has been introduced in version 0.23.

        @param d The distance by which to shift the end points.

        @return The extended edge (self).
        """
    def extended(self, d: float) -> DEdge:
        r"""
        @brief Returns the extended edge (does not modify self)

        Extends the edge by the given distance and returns the 
        extended edge. The edge is not modified. Extending means
        that the first point is shifted by -d along the edge, the second by d.
        The length of the edge will increase by 2*d.

        \extend is a version that modifies self (in-place).

        This method has been introduced in version 0.23.

        @param d The distance by which to shift the end points.

        @return The extended edge.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given edge. This method enables edges as hash keys.

        This method has been introduced in version 0.25.
        """
    def intersect(self, e: DEdge) -> bool:
        r"""
        @brief Intersection test. 

        Returns true if the edges intersect. Two edges intersect if they share at least one point. 
        If the edges coincide, they also intersect.
        For degenerated edges, the intersection is mapped to
        point containment tests.

        @param e The edge to test.
        """
    def intersection_point(self, e: DEdge) -> Any:
        r"""
        @brief Returns the intersection point of two edges. 

        This method delivers the intersection point. If the edges do not intersect, the result will be nil.

        @param e The edge to test.
        @return The point where the edges intersect.

        This method has been introduced in version 0.19.
        From version 0.26.2, this method will return nil in case of non-intersection.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_degenerate(self) -> bool:
        r"""
        @brief Test for degenerated edge

        An edge is degenerate, if both end and start point are identical.
        """
    def is_parallel(self, e: DEdge) -> bool:
        r"""
        @brief Test for being parallel

        @param e The edge to test against

        @return True if both edges are parallel
        """
    def length(self) -> float:
        r"""
        @brief The length of the edge
        """
    @overload
    def move(self, p: DVector) -> DEdge:
        r"""
        @brief Moves the edge.

        Moves the edge by the given offset and returns the 
        moved edge. The edge is overwritten.

        @param p The distance to move the edge.

        @return The moved edge.
        """
    @overload
    def move(self, dx: float, dy: float) -> DEdge:
        r"""
        @brief Moves the edge.

        Moves the edge by the given offset and returns the 
        moved edge. The edge is overwritten.

        @param dx The x distance to move the edge.
        @param dy The y distance to move the edge.

        @return The moved edge.

        This version has been added in version 0.23.
        """
    @overload
    def moved(self, p: DVector) -> DEdge:
        r"""
        @brief Returns the moved edge (does not modify self)

        Moves the edge by the given offset and returns the 
        moved edge. The edge is not modified.

        @param p The distance to move the edge.

        @return The moved edge.
        """
    @overload
    def moved(self, dx: float, dy: float) -> DEdge:
        r"""
        @brief Returns the moved edge (does not modify self)

        Moves the edge by the given offset and returns the 
        moved edge. The edge is not modified.

        @param dx The x distance to move the edge.
        @param dy The y distance to move the edge.

        @return The moved edge.

        This version has been added in version 0.23.
        """
    def ortho_length(self) -> float:
        r"""
        @brief The orthogonal length of the edge ("manhattan-length")

        @return The orthogonal length (abs(dx)+abs(dy))
        """
    def shift(self, d: float) -> DEdge:
        r"""
        @brief Shifts the edge (modifies self)

        Shifts the edge by the given distance and returns the 
        shifted edge. The edge is not modified. Shifting by a positive value will produce an edge which is shifted by d to the left. Shifting by a negative value will produce an edge which is shifted by d to the right.

        \shifted is a version that does not modify self but returns the extended edges.

        This method has been introduced in version 0.23.

        @param d The distance by which to shift the edge.

        @return The shifted edge (self).
        """
    def shifted(self, d: float) -> DEdge:
        r"""
        @brief Returns the shifted edge (does not modify self)

        Shifts the edge by the given distance and returns the 
        shifted edge. The edge is not modified. Shifting by a positive value will produce an edge which is shifted by d to the left. Shifting by a negative value will produce an edge which is shifted by d to the right.

        \shift is a version that modifies self (in-place).

        This method has been introduced in version 0.23.

        @param d The distance by which to shift the edge.

        @return The shifted edge.
        """
    def side_of(self, p: DPoint) -> int:
        r"""
        @brief Indicates at which side the point is located relative to the edge.

        Returns 1 if the point is "left" of the edge, 0 if on
        and -1 if the point is "right" of the edge.

        @param p The point to test.

        @return The side value
        """
    def sq_length(self) -> float:
        r"""
        @brief The square of the length of the edge
        """
    def swap_points(self) -> DEdge:
        r"""
        @brief Swap the points of the edge

        This version modifies self. A version that does not modify self is \swapped_points. Swapping the points basically reverses the direction of the edge.

        This method has been introduced in version 0.23.
        """
    def swapped_points(self) -> DEdge:
        r"""
        @brief Returns an edge in which both points are swapped

        Swapping the points basically reverses the direction of the edge.

        This method has been introduced in version 0.23.
        """
    def to_itype(self, dbu: Optional[float] = ...) -> Edge:
        r"""
        @brief Converts the edge to an integer coordinate edge

        The database unit can be specified to translate the floating-point coordinate edge in micron units to an integer-coordinate edge in database units. The edges coordinates will be divided by the database unit.

        This method has been introduced in version 0.25.
        """
    def to_s(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    @overload
    def transformed(self, t: DCplxTrans) -> DEdge:
        r"""
        @brief Transform the edge.

        Transforms the edge with the given complex transformation.
        Does not modify the edge but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge.
        """
    @overload
    def transformed(self, t: DTrans) -> DEdge:
        r"""
        @brief Transform the edge.

        Transforms the edge with the given transformation.
        Does not modify the edge but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge.
        """
    @overload
    def transformed(self, t: VCplxTrans) -> Edge:
        r"""
        @brief Transforms the edge with the given complex transformation

        @param t The magnifying transformation to apply
        @return The transformed edge (in this case an integer coordinate edge)

        This method has been introduced in version 0.25.
        """
    def transformed_cplx(self, t: DCplxTrans) -> DEdge:
        r"""
        @brief Transform the edge.

        Transforms the edge with the given complex transformation.
        Does not modify the edge but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge.
        """

class EdgePair:
    r"""
    @brief An edge pair (a pair of two edges)
    Edge pairs are objects representing two edges or parts of edges. They play a role mainly in the context of DRC functions, where they specify a DRC violation by connecting two edges which violate the condition checked. Within the framework of polygon and edge collections which provide DRC functionality, edges pairs are used in the form of edge pair collections (\EdgePairs).

    Edge pairs basically consist of two edges, called first and second. If created by a two-layer DRC function, the first edge will correspond to edges from the first layer and the second to edges from the second layer.

    This class has been introduced in version 0.23.
    """
    first: Edge
    r"""
    Getter:
    @brief Gets the first edge

    Setter:
    @brief Sets the first edge
    """
    second: Edge
    r"""
    Getter:
    @brief Gets the second edge

    Setter:
    @brief Sets the second edge
    """
    symmetric: bool
    r"""
    Getter:
    @brief Returns a value indicating whether the edge pair is symmetric
    For symmetric edge pairs, the edges are commutable. Specifically, a symmetric edge pair with (e1,e2) is identical to (e2,e1). Symmetric edge pairs are generated by some checks for which there is no directed error marker (width, space, notch, isolated).

    Symmetric edge pairs have been introduced in version 0.27.

    Setter:
    @brief Sets a value indicating whether the edge pair is symmetric
    See \symmetric? for a description of this attribute.

    Symmetric edge pairs have been introduced in version 0.27.
    """
    @classmethod
    def from_s(cls, s: str) -> EdgePair:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> EdgePair:
        r"""
        @brief Default constructor

        This constructor creates an default edge pair.
        """
    @overload
    @classmethod
    def new(cls, dedge_pair: DEdgePair) -> EdgePair:
        r"""
        @brief Creates an integer coordinate edge pair from a floating-point coordinate edge pair

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dedge_pair'.
        """
    @overload
    @classmethod
    def new(cls, first: Edge, second: Edge, symmetric: Optional[bool] = ...) -> EdgePair:
        r"""
        @brief Constructor from two edges

        This constructor creates an edge pair from the two edges given.
        See \symmetric? for a description of this attribute.
        """
    def __copy__(self) -> EdgePair:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> EdgePair:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, box: object) -> bool:
        r"""
        @brief Equality
        Returns true, if this edge pair and the given one are equal

        This method has been introduced in version 0.25.
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given edge pair. This method enables edge pairs as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor

        This constructor creates an default edge pair.
        """
    @overload
    def __init__(self, dedge_pair: DEdgePair) -> None:
        r"""
        @brief Creates an integer coordinate edge pair from a floating-point coordinate edge pair

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dedge_pair'.
        """
    @overload
    def __init__(self, first: Edge, second: Edge, symmetric: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor from two edges

        This constructor creates an edge pair from the two edges given.
        See \symmetric? for a description of this attribute.
        """
    def __lt__(self, box: EdgePair) -> bool:
        r"""
        @brief Less operator
        Returns true, if this edge pair is 'less' with respect to first and second edge

        This method has been introduced in version 0.25.
        """
    def __ne__(self, box: object) -> bool:
        r"""
        @brief Inequality
        Returns true, if this edge pair and the given one are not equal

        This method has been introduced in version 0.25.
        """
    def __str__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge pair
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def area(self) -> int:
        r"""
        @brief Gets the area between the edges of the edge pair

        This attribute has been introduced in version 0.28.
        """
    def assign(self, other: EdgePair) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> Box:
        r"""
        @brief Gets the bounding box of the edge pair
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> EdgePair:
        r"""
        @brief Creates a copy of self
        """
    def greater(self) -> Edge:
        r"""
        @brief Gets the 'greater' edge for symmetric edge pairs
        As first and second edges are commutable for symmetric edge pairs (see \symmetric?), this accessor allows retrieving a 'second' edge in a way independent on the actual assignment.

        This read-only attribute has been introduced in version 0.27.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given edge pair. This method enables edge pairs as hash keys.

        This method has been introduced in version 0.25.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def lesser(self) -> Edge:
        r"""
        @brief Gets the 'lesser' edge for symmetric edge pairs
        As first and second edges are commutable for symmetric edge pairs (see \symmetric?), this accessor allows retrieving a 'first' edge in a way independent on the actual assignment.

        This read-only attribute has been introduced in version 0.27.
        """
    def normalized(self) -> EdgePair:
        r"""
        @brief Normalizes the edge pair
        This method normalized the edge pair such that when connecting the edges at their 
        start and end points a closed loop is formed which is oriented clockwise. To achieve this, the points of the first and/or first and second edge are swapped. Normalization is a first step recommended before converting an edge pair to a polygon, because that way the polygons won't be self-overlapping and the enlargement parameter is applied properly.
        """
    def perimeter(self) -> int:
        r"""
        @brief Gets the perimeter of the edge pair

        The perimeter is defined as the sum of the lengths of both edges ('active perimeter').

        This attribute has been introduced in version 0.28.
        """
    def polygon(self, e: int) -> Polygon:
        r"""
        @brief Convert an edge pair to a polygon
        The polygon is formed by connecting the end and start points of the edges. It is recommended to use \normalized before converting the edge pair to a polygon.

        The enlargement parameter applies the specified enlargement parallel and perpendicular to the edges. Basically this introduces a bias which blows up edge pairs by the specified amount. That parameter is useful to convert degenerated edge pairs to valid polygons, i.e. edge pairs with coincident edges and edge pairs consisting of two point-like edges.

        Another version for converting edge pairs to simple polygons is \simple_polygon which renders a \SimplePolygon object.
        @param e The enlargement (set to zero for exact representation)
        """
    def simple_polygon(self, e: int) -> SimplePolygon:
        r"""
        @brief Convert an edge pair to a simple polygon
        The polygon is formed by connecting the end and start points of the edges. It is recommended to use \normalized before converting the edge pair to a polygon.

        The enlargement parameter applies the specified enlargement parallel and perpendicular to the edges. Basically this introduces a bias which blows up edge pairs by the specified amount. That parameter is useful to convert degenerated edge pairs to valid polygons, i.e. edge pairs with coincident edges and edge pairs consisting of two point-like edges.

        Another version for converting edge pairs to polygons is \polygon which renders a \Polygon object.
        @param e The enlargement (set to zero for exact representation)
        """
    def to_dtype(self, dbu: Optional[float] = ...) -> DEdgePair:
        r"""
        @brief Converts the edge pair to a floating-point coordinate edge pair

        The database unit can be specified to translate the integer-coordinate edge pair into a floating-point coordinate edge pair in micron units. The database unit is basically a scaling factor.

        This method has been introduced in version 0.25.
        """
    def to_s(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge pair
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    @overload
    def transformed(self, t: CplxTrans) -> DEdgePair:
        r"""
        @brief Returns the transformed edge pair

        Transforms the edge pair with the given complex transformation.
        Does not modify the edge pair but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge pair
        """
    @overload
    def transformed(self, t: ICplxTrans) -> EdgePair:
        r"""
        @brief Returns the transformed edge pair

        Transforms the edge pair with the given complex transformation.
        Does not modify the edge pair but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge pair (in this case an integer coordinate edge pair).
        """
    @overload
    def transformed(self, t: Trans) -> EdgePair:
        r"""
        @brief Returns the transformed pair

        Transforms the edge pair with the given transformation.
        Does not modify the edge pair but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge pair
        """

class DEdgePair:
    r"""
    @brief An edge pair (a pair of two edges)
    Edge pairs are objects representing two edges or parts of edges. They play a role mainly in the context of DRC functions, where they specify a DRC violation by connecting two edges which violate the condition checked. Within the framework of polygon and edge collections which provide DRC functionality, edges pairs with integer coordinates (\EdgePair type) are used in the form of edge pair collections (\EdgePairs).

    Edge pairs basically consist of two edges, called first and second. If created by a two-layer DRC function, the first edge will correspond to edges from the first layer and the second to edges from the second layer.

    This class has been introduced in version 0.23.
    """
    first: DEdge
    r"""
    Getter:
    @brief Gets the first edge

    Setter:
    @brief Sets the first edge
    """
    second: DEdge
    r"""
    Getter:
    @brief Gets the second edge

    Setter:
    @brief Sets the second edge
    """
    symmetric: bool
    r"""
    Getter:
    @brief Returns a value indicating whether the edge pair is symmetric
    For symmetric edge pairs, the edges are commutable. Specifically, a symmetric edge pair with (e1,e2) is identical to (e2,e1). Symmetric edge pairs are generated by some checks for which there is no directed error marker (width, space, notch, isolated).

    Symmetric edge pairs have been introduced in version 0.27.

    Setter:
    @brief Sets a value indicating whether the edge pair is symmetric
    See \symmetric? for a description of this attribute.

    Symmetric edge pairs have been introduced in version 0.27.
    """
    @classmethod
    def from_s(cls, s: str) -> DEdgePair:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> DEdgePair:
        r"""
        @brief Default constructor

        This constructor creates an default edge pair.
        """
    @overload
    @classmethod
    def new(cls, edge_pair: EdgePair) -> DEdgePair:
        r"""
        @brief Creates a floating-point coordinate edge pair from an integer coordinate edge

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_iedge_pair'.
        """
    @overload
    @classmethod
    def new(cls, first: DEdge, second: DEdge, symmetric: Optional[bool] = ...) -> DEdgePair:
        r"""
        @brief Constructor from two edges

        This constructor creates an edge pair from the two edges given.
        See \symmetric? for a description of this attribute.
        """
    def __copy__(self) -> DEdgePair:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DEdgePair:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, box: object) -> bool:
        r"""
        @brief Equality
        Returns true, if this edge pair and the given one are equal

        This method has been introduced in version 0.25.
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given edge pair. This method enables edge pairs as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor

        This constructor creates an default edge pair.
        """
    @overload
    def __init__(self, edge_pair: EdgePair) -> None:
        r"""
        @brief Creates a floating-point coordinate edge pair from an integer coordinate edge

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_iedge_pair'.
        """
    @overload
    def __init__(self, first: DEdge, second: DEdge, symmetric: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor from two edges

        This constructor creates an edge pair from the two edges given.
        See \symmetric? for a description of this attribute.
        """
    def __lt__(self, box: DEdgePair) -> bool:
        r"""
        @brief Less operator
        Returns true, if this edge pair is 'less' with respect to first and second edge

        This method has been introduced in version 0.25.
        """
    def __ne__(self, box: object) -> bool:
        r"""
        @brief Inequality
        Returns true, if this edge pair and the given one are not equal

        This method has been introduced in version 0.25.
        """
    def __str__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge pair
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def area(self) -> float:
        r"""
        @brief Gets the area between the edges of the edge pair

        This attribute has been introduced in version 0.28.
        """
    def assign(self, other: DEdgePair) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> DBox:
        r"""
        @brief Gets the bounding box of the edge pair
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DEdgePair:
        r"""
        @brief Creates a copy of self
        """
    def greater(self) -> DEdge:
        r"""
        @brief Gets the 'greater' edge for symmetric edge pairs
        As first and second edges are commutable for symmetric edge pairs (see \symmetric?), this accessor allows retrieving a 'second' edge in a way independent on the actual assignment.

        This read-only attribute has been introduced in version 0.27.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given edge pair. This method enables edge pairs as hash keys.

        This method has been introduced in version 0.25.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def lesser(self) -> DEdge:
        r"""
        @brief Gets the 'lesser' edge for symmetric edge pairs
        As first and second edges are commutable for symmetric edge pairs (see \symmetric?), this accessor allows retrieving a 'first' edge in a way independent on the actual assignment.

        This read-only attribute has been introduced in version 0.27.
        """
    def normalized(self) -> DEdgePair:
        r"""
        @brief Normalizes the edge pair
        This method normalized the edge pair such that when connecting the edges at their 
        start and end points a closed loop is formed which is oriented clockwise. To achieve this, the points of the first and/or first and second edge are swapped. Normalization is a first step recommended before converting an edge pair to a polygon, because that way the polygons won't be self-overlapping and the enlargement parameter is applied properly.
        """
    def perimeter(self) -> float:
        r"""
        @brief Gets the perimeter of the edge pair

        The perimeter is defined as the sum of the lengths of both edges ('active perimeter').

        This attribute has been introduced in version 0.28.
        """
    def polygon(self, e: float) -> DPolygon:
        r"""
        @brief Convert an edge pair to a polygon
        The polygon is formed by connecting the end and start points of the edges. It is recommended to use \normalized before converting the edge pair to a polygon.

        The enlargement parameter applies the specified enlargement parallel and perpendicular to the edges. Basically this introduces a bias which blows up edge pairs by the specified amount. That parameter is useful to convert degenerated edge pairs to valid polygons, i.e. edge pairs with coincident edges and edge pairs consisting of two point-like edges.

        Another version for converting edge pairs to simple polygons is \simple_polygon which renders a \SimplePolygon object.
        @param e The enlargement (set to zero for exact representation)
        """
    def simple_polygon(self, e: float) -> DSimplePolygon:
        r"""
        @brief Convert an edge pair to a simple polygon
        The polygon is formed by connecting the end and start points of the edges. It is recommended to use \normalized before converting the edge pair to a polygon.

        The enlargement parameter applies the specified enlargement parallel and perpendicular to the edges. Basically this introduces a bias which blows up edge pairs by the specified amount. That parameter is useful to convert degenerated edge pairs to valid polygons, i.e. edge pairs with coincident edges and edge pairs consisting of two point-like edges.

        Another version for converting edge pairs to polygons is \polygon which renders a \Polygon object.
        @param e The enlargement (set to zero for exact representation)
        """
    def to_itype(self, dbu: Optional[float] = ...) -> EdgePair:
        r"""
        @brief Converts the edge pair to an integer coordinate edge pair

        The database unit can be specified to translate the floating-point coordinate edge pair in micron units to an integer-coordinate edge pair in database units. The edge pair's' coordinates will be divided by the database unit.

        This method has been introduced in version 0.25.
        """
    def to_s(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief Returns a string representing the edge pair
         If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    @overload
    def transformed(self, t: DCplxTrans) -> DEdgePair:
        r"""
        @brief Returns the transformed edge pair

        Transforms the edge pair with the given complex transformation.
        Does not modify the edge pair but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge pair
        """
    @overload
    def transformed(self, t: DTrans) -> DEdgePair:
        r"""
        @brief Returns the transformed pair

        Transforms the edge pair with the given transformation.
        Does not modify the edge pair but returns the transformed edge.

        @param t The transformation to apply.

        @return The transformed edge pair
        """
    @overload
    def transformed(self, t: VCplxTrans) -> EdgePair:
        r"""
        @brief Transforms the edge pair with the given complex transformation


        @param t The magnifying transformation to apply
        @return The transformed edge pair (in this case an integer coordinate edge pair)

        This method has been introduced in version 0.25.
        """

class EdgePairs(ShapeCollection):
    r"""
    @brief EdgePairs (a collection of edge pairs)

    Edge pairs are used mainly in the context of the DRC functions (width_check, space_check etc.) of \Region and \Edges. A single edge pair represents two edges participating in a DRC violation. In the two-layer checks (inside, overlap) The first edge represents an edge from the first layer and the second edge an edge from the second layer. For single-layer checks (width, space) the order of the edges is arbitrary.

    This class has been introduced in version 0.23.
    """
    @overload
    @classmethod
    def new(cls) -> EdgePairs:
        r"""
        @brief Default constructor

        This constructor creates an empty edge pair collection.
        """
    @overload
    @classmethod
    def new(cls, array: Sequence[EdgePair]) -> EdgePairs:
        r"""
        @brief Constructor from an edge pair array

        This constructor creates an edge pair collection from an array of \EdgePair objects.

        This constructor has been introduced in version 0.26.
        """
    @overload
    @classmethod
    def new(cls, edge_pair: EdgePair) -> EdgePairs:
        r"""
        @brief Constructor from a single edge pair object

        This constructor creates an edge pair collection with a single edge pair.

        This constructor has been introduced in version 0.26.
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator) -> EdgePairs:
        r"""
        @brief Constructor from a hierarchical shape set

        This constructor creates an edge pair collection from the shapes delivered by the given recursive shape iterator.
        Only edge pairs are taken from the shape set and other shapes are ignored.
        This method allows feeding the edge pair collection from a hierarchy of cells.
        Edge pairs in layout objects are somewhat special as most formats don't support reading or writing of edge pairs. Still they are useful objects and can be created and manipulated inside layouts.

        @code
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        r = RBA::EdgePairs::new(layout.begin_shapes(cell, layer))
        @/code

        This constructor has been introduced in version 0.26.
        """
    @overload
    @classmethod
    def new(cls, shapes: Shapes) -> EdgePairs:
        r"""
        @brief Shapes constructor

        This constructor creates an edge pair collection from a \Shapes collection.

        This constructor has been introduced in version 0.26.
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore) -> EdgePairs:
        r"""
        @brief Creates a hierarchical edge pair collection from an original layer

        This constructor creates an edge pair collection from the shapes delivered by the given recursive shape iterator.
        This version will create a hierarchical edge pair collection which supports hierarchical operations.
        Edge pairs in layout objects are somewhat special as most formats don't support reading or writing of edge pairs. Still they are useful objects and can be created and manipulated inside layouts.

        @code
        dss    = RBA::DeepShapeStore::new
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        r = RBA::EdgePairs::new(layout.begin_shapes(cell, layer))
        @/code

        This constructor has been introduced in version 0.26.
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator, trans: ICplxTrans) -> EdgePairs:
        r"""
        @brief Constructor from a hierarchical shape set with a transformation

        This constructor creates an edge pair collection from the shapes delivered by the given recursive shape iterator.
        Only edge pairs are taken from the shape set and other shapes are ignored.
        The given transformation is applied to each edge pair taken.
        This method allows feeding the edge pair collection from a hierarchy of cells.
        The transformation is useful to scale to a specific database unit for example.
        Edge pairs in layout objects are somewhat special as most formats don't support reading or writing of edge pairs. Still they are useful objects and can be created and manipulated inside layouts.

        @code
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        dbu    = 0.1 # the target database unit
        r = RBA::EdgePairs::new(layout.begin_shapes(cell, layer), RBA::ICplxTrans::new(layout.dbu / dbu))
        @/code

        This constructor has been introduced in version 0.26.
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore, trans: ICplxTrans) -> EdgePairs:
        r"""
        @brief Creates a hierarchical edge pair collection from an original layer with a transformation

        This constructor creates an edge pair collection from the shapes delivered by the given recursive shape iterator.
        This version will create a hierarchical edge pair collection which supports hierarchical operations.
        The transformation is useful to scale to a specific database unit for example.
        Edge pairs in layout objects are somewhat special as most formats don't support reading or writing of edge pairs. Still they are useful objects and can be created and manipulated inside layouts.

        @code
        dss    = RBA::DeepShapeStore::new
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        dbu    = 0.1 # the target database unit
        r = RBA::EdgePairs::new(layout.begin_shapes(cell, layer), RBA::ICplxTrans::new(layout.dbu / dbu))
        @/code

        This constructor has been introduced in version 0.26.
        """
    def __add__(self, other: EdgePairs) -> EdgePairs:
        r"""
        @brief Returns the combined edge pair collection of self and the other one

        @return The resulting edge pair collection

        This operator adds the edge pairs of the other collection to self and returns a new combined set.

        This method has been introduced in version 0.24.
        """
    def __copy__(self) -> EdgePairs:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> EdgePairs:
        r"""
        @brief Creates a copy of self
        """
    def __getitem__(self, n: int) -> EdgePair:
        r"""
        @brief Returns the nth edge pair

        This method returns nil if the index is out of range. It is available for flat edge pairs only - i.e. those for which \has_valid_edge_pairs? is true. Use \flatten to explicitly flatten an edge pair collection.

        The \each iterator is the more general approach to access the edge pairs.
        """
    def __iadd__(self, other: EdgePairs) -> EdgePairs:
        r"""
        @brief Adds the edge pairs of the other edge pair collection to self

        @return The edge pair collection after modification (self)

        This operator adds the edge pairs of the other collection to self.

        This method has been introduced in version 0.24.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor

        This constructor creates an empty edge pair collection.
        """
    @overload
    def __init__(self, array: Sequence[EdgePair]) -> None:
        r"""
        @brief Constructor from an edge pair array

        This constructor creates an edge pair collection from an array of \EdgePair objects.

        This constructor has been introduced in version 0.26.
        """
    @overload
    def __init__(self, edge_pair: EdgePair) -> None:
        r"""
        @brief Constructor from a single edge pair object

        This constructor creates an edge pair collection with a single edge pair.

        This constructor has been introduced in version 0.26.
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator) -> None:
        r"""
        @brief Constructor from a hierarchical shape set

        This constructor creates an edge pair collection from the shapes delivered by the given recursive shape iterator.
        Only edge pairs are taken from the shape set and other shapes are ignored.
        This method allows feeding the edge pair collection from a hierarchy of cells.
        Edge pairs in layout objects are somewhat special as most formats don't support reading or writing of edge pairs. Still they are useful objects and can be created and manipulated inside layouts.

        @code
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        r = RBA::EdgePairs::new(layout.begin_shapes(cell, layer))
        @/code

        This constructor has been introduced in version 0.26.
        """
    @overload
    def __init__(self, shapes: Shapes) -> None:
        r"""
        @brief Shapes constructor

        This constructor creates an edge pair collection from a \Shapes collection.

        This constructor has been introduced in version 0.26.
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore) -> None:
        r"""
        @brief Creates a hierarchical edge pair collection from an original layer

        This constructor creates an edge pair collection from the shapes delivered by the given recursive shape iterator.
        This version will create a hierarchical edge pair collection which supports hierarchical operations.
        Edge pairs in layout objects are somewhat special as most formats don't support reading or writing of edge pairs. Still they are useful objects and can be created and manipulated inside layouts.

        @code
        dss    = RBA::DeepShapeStore::new
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        r = RBA::EdgePairs::new(layout.begin_shapes(cell, layer))
        @/code

        This constructor has been introduced in version 0.26.
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator, trans: ICplxTrans) -> None:
        r"""
        @brief Constructor from a hierarchical shape set with a transformation

        This constructor creates an edge pair collection from the shapes delivered by the given recursive shape iterator.
        Only edge pairs are taken from the shape set and other shapes are ignored.
        The given transformation is applied to each edge pair taken.
        This method allows feeding the edge pair collection from a hierarchy of cells.
        The transformation is useful to scale to a specific database unit for example.
        Edge pairs in layout objects are somewhat special as most formats don't support reading or writing of edge pairs. Still they are useful objects and can be created and manipulated inside layouts.

        @code
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        dbu    = 0.1 # the target database unit
        r = RBA::EdgePairs::new(layout.begin_shapes(cell, layer), RBA::ICplxTrans::new(layout.dbu / dbu))
        @/code

        This constructor has been introduced in version 0.26.
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore, trans: ICplxTrans) -> None:
        r"""
        @brief Creates a hierarchical edge pair collection from an original layer with a transformation

        This constructor creates an edge pair collection from the shapes delivered by the given recursive shape iterator.
        This version will create a hierarchical edge pair collection which supports hierarchical operations.
        The transformation is useful to scale to a specific database unit for example.
        Edge pairs in layout objects are somewhat special as most formats don't support reading or writing of edge pairs. Still they are useful objects and can be created and manipulated inside layouts.

        @code
        dss    = RBA::DeepShapeStore::new
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        dbu    = 0.1 # the target database unit
        r = RBA::EdgePairs::new(layout.begin_shapes(cell, layer), RBA::ICplxTrans::new(layout.dbu / dbu))
        @/code

        This constructor has been introduced in version 0.26.
        """
    def __iter__(self) -> Iterator[EdgePair]:
        r"""
        @brief Returns each edge pair of the edge pair collection
        """
    def __len__(self) -> int:
        r"""
        @brief Returns the (flat) number of edge pairs in the edge pair collection

        The count is computed 'as if flat', i.e. edge pairs inside a cell are multiplied by the number of times a cell is instantiated.

        Starting with version 0.27, the method is called 'count' for consistency with \Region. 'size' is still provided as an alias.
        """
    def __str__(self) -> str:
        r"""
        @brief Converts the edge pair collection to a string
        The length of the output is limited to 20 edge pairs to avoid giant strings on large regions. For full output use "to_s" with a maximum count parameter.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: ShapeCollection) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> Box:
        r"""
        @brief Return the bounding box of the edge pair collection
        The bounding box is the box enclosing all points of all edge pairs.
        """
    def clear(self) -> None:
        r"""
        @brief Clears the edge pair collection
        """
    def count(self) -> int:
        r"""
        @brief Returns the (flat) number of edge pairs in the edge pair collection

        The count is computed 'as if flat', i.e. edge pairs inside a cell are multiplied by the number of times a cell is instantiated.

        Starting with version 0.27, the method is called 'count' for consistency with \Region. 'size' is still provided as an alias.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def data_id(self) -> int:
        r"""
        @brief Returns the data ID (a unique identifier for the underlying data storage)

        This method has been added in version 0.26.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def disable_progress(self) -> None:
        r"""
        @brief Disable progress reporting
        Calling this method will disable progress reporting. See \enable_progress.
        """
    def dup(self) -> EdgePairs:
        r"""
        @brief Creates a copy of self
        """
    def each(self) -> Iterator[EdgePair]:
        r"""
        @brief Returns each edge pair of the edge pair collection
        """
    def edges(self) -> Edges:
        r"""
        @brief Decomposes the edge pairs into single edges
        @return An edge collection containing the individual edges
        """
    def enable_progress(self, label: str) -> None:
        r"""
        @brief Enable progress reporting
        After calling this method, the edge pair collection will report the progress through a progress bar while expensive operations are running.
        The label is a text which is put in front of the progress bar.
        Using a progress bar will imply a performance penalty of a few percent typically.
        """
    @overload
    def extents(self) -> Region:
        r"""
        @brief Returns a region with the bounding boxes of the edge pairs
        This method will return a region consisting of the bounding boxes of the edge pairs.
        The boxes will not be merged, so it is possible to determine overlaps of these boxes for example.
        """
    @overload
    def extents(self, d: int) -> Region:
        r"""
        @brief Returns a region with the enlarged bounding boxes of the edge pairs
        This method will return a region consisting of the bounding boxes of the edge pairs enlarged by the given distance d.
        The enlargement is specified per edge, i.e the width and height will be increased by 2*d.
        The boxes will not be merged, so it is possible to determine overlaps of these boxes for example.
        """
    @overload
    def extents(self, dx: int, dy: int) -> Region:
        r"""
        @brief Returns a region with the enlarged bounding boxes of the edge pairs
        This method will return a region consisting of the bounding boxes of the edge pairs enlarged by the given distance dx in x direction and dy in y direction.
        The enlargement is specified per edge, i.e the width will be increased by 2*dx.
        The boxes will not be merged, so it is possible to determine overlaps of these boxes for example.
        """
    def first_edges(self) -> Edges:
        r"""
        @brief Returns the first one of all edges
        @return An edge collection containing the first edges
        """
    def flatten(self) -> None:
        r"""
        @brief Explicitly flattens an edge pair collection

        If the collection is already flat (i.e. \has_valid_edge_pairs? returns true), this method will not change the collection.

        This method has been introduced in version 0.26.
        """
    def has_valid_edge_pairs(self) -> bool:
        r"""
        @brief Returns true if the edge pair collection is flat and individual edge pairs can be accessed randomly

        This method has been introduced in version 0.26.
        """
    def hier_count(self) -> int:
        r"""
        @brief Returns the (hierarchical) number of edge pairs in the edge pair collection

        The count is computed 'hierarchical', i.e. edge pairs inside a cell are counted once even if the cell is instantiated multiple times.

        This method has been introduced in version 0.27.
        """
    @overload
    def insert(self, edge_pair: EdgePair) -> None:
        r"""
        @brief Inserts an edge pair into the collection
        """
    @overload
    def insert(self, edge_pairs: EdgePairs) -> None:
        r"""
        @brief Inserts all edge pairs from the other edge pair collection into this edge pair collection
        This method has been introduced in version 0.25.
        """
    @overload
    def insert(self, first: Edge, second: Edge) -> None:
        r"""
        @brief Inserts an edge pair into the collection
        """
    def insert_into(self, layout: Layout, cell_index: int, layer: int) -> None:
        r"""
        @brief Inserts this edge pairs into the given layout, below the given cell and into the given layer.
        If the edge pair collection is a hierarchical one, a suitable hierarchy will be built below the top cell or and existing hierarchy will be reused.

        This method has been introduced in version 0.26.
        """
    def insert_into_as_polygons(self, layout: Layout, cell_index: int, layer: int, e: int) -> None:
        r"""
        @brief Inserts this edge pairs into the given layout, below the given cell and into the given layer.
        If the edge pair collection is a hierarchical one, a suitable hierarchy will be built below the top cell or and existing hierarchy will be reused.

        The edge pairs will be converted to polygons with the enlargement value given be 'e'.

        This method has been introduced in version 0.26.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_deep(self) -> bool:
        r"""
        @brief Returns true if the edge pair collection is a deep (hierarchical) one

        This method has been added in version 0.26.
        """
    def is_empty(self) -> bool:
        r"""
        @brief Returns true if the collection is empty
        """
    @overload
    def move(self, p: Vector) -> EdgePairs:
        r"""
        @brief Moves the edge pair collection

        Moves the edge pairs by the given offset and returns the 
        moved edge pair collection. The edge pair collection is overwritten.

        @param p The distance to move the edge pairs.

        @return The moved edge pairs (self).

        Starting with version 0.25 the displacement is of vector type.
        """
    @overload
    def move(self, x: int, y: int) -> EdgePairs:
        r"""
        @brief Moves the edge pair collection

        Moves the edge pairs by the given offset and returns the 
        moved edge pairs. The edge pair collection is overwritten.

        @param x The x distance to move the edge pairs.
        @param y The y distance to move the edge pairs.

        @return The moved edge pairs (self).
        """
    @overload
    def moved(self, p: Vector) -> EdgePairs:
        r"""
        @brief Returns the moved edge pair collection (does not modify self)

        Moves the edge pairs by the given offset and returns the 
        moved edge pairs. The edge pair collection is not modified.

        @param p The distance to move the edge pairs.

        @return The moved edge pairs.

        Starting with version 0.25 the displacement is of vector type.
        """
    @overload
    def moved(self, x: int, y: int) -> EdgePairs:
        r"""
        @brief Returns the moved edge pair collection (does not modify self)

        Moves the edge pairs by the given offset and returns the 
        moved edge pairs. The edge pair collection is not modified.

        @param x The x distance to move the edge pairs.
        @param y The y distance to move the edge pairs.

        @return The moved edge pairs.
        """
    @overload
    def polygons(self) -> Region:
        r"""
        @brief Converts the edge pairs to polygons
        This method creates polygons from the edge pairs. Each polygon will be a triangle or quadrangle which connects the start and end points of the edges forming the edge pair.
        """
    @overload
    def polygons(self, e: int) -> Region:
        r"""
        @brief Converts the edge pairs to polygons
        This method creates polygons from the edge pairs. Each polygon will be a triangle or quadrangle which connects the start and end points of the edges forming the edge pair. This version allows one to specify an enlargement which is applied to the edges. The length of the edges is modified by applying the enlargement and the edges are shifted by the enlargement. By specifying an enlargement it is possible to give edge pairs an area which otherwise would not have one (coincident edges, two point-like edges).
        """
    def second_edges(self) -> Edges:
        r"""
        @brief Returns the second one of all edges
        @return An edge collection containing the second edges
        """
    def size(self) -> int:
        r"""
        @brief Returns the (flat) number of edge pairs in the edge pair collection

        The count is computed 'as if flat', i.e. edge pairs inside a cell are multiplied by the number of times a cell is instantiated.

        Starting with version 0.27, the method is called 'count' for consistency with \Region. 'size' is still provided as an alias.
        """
    def swap(self, other: EdgePairs) -> None:
        r"""
        @brief Swap the contents of this collection with the contents of another collection
        This method is useful to avoid excessive memory allocation in some cases. For managed memory languages such as Ruby, those cases will be rare. 
        """
    @overload
    def to_s(self) -> str:
        r"""
        @brief Converts the edge pair collection to a string
        The length of the output is limited to 20 edge pairs to avoid giant strings on large regions. For full output use "to_s" with a maximum count parameter.
        """
    @overload
    def to_s(self, max_count: int) -> str:
        r"""
        @brief Converts the edge pair collection to a string
        This version allows specification of the maximum number of edge pairs contained in the string.
        """
    @overload
    def transform(self, t: ICplxTrans) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection with a complex transformation (modifies self)

        Transforms the edge pair collection with the given transformation.
        This version modifies the edge pair collection and returns a reference to self.

        @param t The transformation to apply.

        @return The transformed edge pair collection.
        """
    @overload
    def transform(self, t: IMatrix2d) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection (modifies self)

        Transforms the edge pair collection with the given 2d matrix transformation.
        This version modifies the edge pair collection and returns a reference to self.

        @param t The transformation to apply.

        @return The transformed edge pair collection.

        This variant has been introduced in version 0.27.
        """
    @overload
    def transform(self, t: IMatrix3d) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection (modifies self)

        Transforms the edge pair collection with the given 3d matrix transformation.
        This version modifies the edge pair collection and returns a reference to self.

        @param t The transformation to apply.

        @return The transformed edge pair collection.

        This variant has been introduced in version 0.27.
        """
    @overload
    def transform(self, t: Trans) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection (modifies self)

        Transforms the edge pair collection with the given transformation.
        This version modifies the edge pair collection and returns a reference to self.

        @param t The transformation to apply.

        @return The transformed edge pair collection.
        """
    def transform_icplx(self, t: ICplxTrans) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection with a complex transformation (modifies self)

        Transforms the edge pair collection with the given transformation.
        This version modifies the edge pair collection and returns a reference to self.

        @param t The transformation to apply.

        @return The transformed edge pair collection.
        """
    @overload
    def transformed(self, t: ICplxTrans) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection with a complex transformation

        Transforms the edge pairs with the given complex transformation.
        Does not modify the edge pair collection but returns the transformed edge pairs.

        @param t The transformation to apply.

        @return The transformed edge pairs.
        """
    @overload
    def transformed(self, t: IMatrix2d) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection

        Transforms the edge pairs with the given 2d matrix transformation.
        Does not modify the edge pair collection but returns the transformed edge pairs.

        @param t The transformation to apply.

        @return The transformed edge pairs.

        This variant has been introduced in version 0.27.
        """
    @overload
    def transformed(self, t: IMatrix3d) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection

        Transforms the edge pairs with the given 3d matrix transformation.
        Does not modify the edge pair collection but returns the transformed edge pairs.

        @param t The transformation to apply.

        @return The transformed edge pairs.

        This variant has been introduced in version 0.27.
        """
    @overload
    def transformed(self, t: Trans) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection

        Transforms the edge pairs with the given transformation.
        Does not modify the edge pair collection but returns the transformed edge pairs.

        @param t The transformation to apply.

        @return The transformed edge pairs.
        """
    def transformed_icplx(self, t: ICplxTrans) -> EdgePairs:
        r"""
        @brief Transform the edge pair collection with a complex transformation

        Transforms the edge pairs with the given complex transformation.
        Does not modify the edge pair collection but returns the transformed edge pairs.

        @param t The transformation to apply.

        @return The transformed edge pairs.
        """
    @overload
    def with_angle(self, angle: float, inverse: bool) -> EdgePairs:
        r"""
        @brief Filter the edge pairs by orientation of their edges
        Filters the edge pairs in the edge pair collection by orientation. If "inverse" is false, only edge pairs with at least one edge having the given angle to the x-axis are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        This will filter edge pairs with at least one horizontal edge:

        @code
        horizontal = edge_pairs.with_angle(0, false)
        @/code

        Note that the inverse @b result @/b of \with_angle is delivered by \with_angle_both with the inverse flag set as edge pairs are unselected when both edges fail to meet the criterion.
        I.e

        @code
        result = edge_pairs.with_angle(0, false)
        others = edge_pairs.with_angle_both(0, true)
        @/code

        This method has been added in version 0.27.1.
        """
    @overload
    def with_angle(self, type: Edges.EdgeType, inverse: bool) -> EdgePairs:
        r"""
        @brief Filter the edge pairs by orientation of their edges
        Filters the edge pairs in the edge pair collection by orientation. If "inverse" is false, only edge pairs with at least one edge having an angle of the given type are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        This version allows specifying an edge type instead of an angle. Edge types include multiple distinct orientations and are specified using one of the \Edges#OrthoEdges, \Edges#DiagonalEdges or \Edges#OrthoDiagonalEdges types.

        Note that the inverse @b result @/b of \with_angle is delivered by \with_angle_both with the inverse flag set as edge pairs are unselected when both edges fail to meet the criterion.
        I.e

        @code
        result = edge_pairs.with_angle(RBA::Edges::Ortho, false)
        others = edge_pairs.with_angle_both(RBA::Edges::Ortho, true)
        @/code

        This method has been added in version 0.28.
        """
    @overload
    def with_angle(self, min_angle: float, max_angle: float, inverse: bool, include_min_angle: Optional[bool] = ..., include_max_angle: Optional[bool] = ...) -> EdgePairs:
        r"""
        @brief Filter the edge pairs by orientation of their edges
        Filters the edge pairs in the edge pair collection by orientation. If "inverse" is false, only edge pairs with at least one edge having an angle between min_angle and max_angle are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        With "include_min_angle" set to true (the default), the minimum angle is included in the criterion while with false, the minimum angle itself is not included. Same for "include_max_angle" where the default is false, meaning the maximum angle is not included in the range.

        Note that the inverse @b result @/b of \with_angle is delivered by \with_angle_both with the inverse flag set as edge pairs are unselected when both edges fail to meet the criterion.
        I.e

        @code
        result = edge_pairs.with_angle(0, 45, false)
        others = edge_pairs.with_angle_both(0, 45, true)
        @/code

        This method has been added in version 0.27.1.
        """
    @overload
    def with_angle_both(self, angle: float, inverse: bool) -> EdgePairs:
        r"""
        @brief Filter the edge pairs by orientation of both of their edges
        Filters the edge pairs in the edge pair collection by orientation. If "inverse" is false, only edge pairs with both edges having the given angle to the x-axis are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        This will filter edge pairs with at least one horizontal edge:

        @code
        horizontal = edge_pairs.with_angle_both(0, false)
        @/code

        Note that the inverse @b result @/b of \with_angle_both is delivered by \with_angle with the inverse flag set as edge pairs are unselected when one edge fails to meet the criterion.
        I.e

        @code
        result = edge_pairs.with_angle_both(0, false)
        others = edge_pairs.with_angle(0, true)
        @/code

        This method has been added in version 0.27.1.
        """
    @overload
    def with_angle_both(self, type: Edges.EdgeType, inverse: bool) -> EdgePairs:
        r"""
        @brief Filter the edge pairs by orientation of their edges
        Filters the edge pairs in the edge pair collection by orientation. If "inverse" is false, only edge pairs with both edges having an angle of the given type are returned. If "inverse" is true, edge pairs not fulfilling this criterion for both edges are returned.

        This version allows specifying an edge type instead of an angle. Edge types include multiple distinct orientations and are specified using one of the \Edges#OrthoEdges, \Edges#DiagonalEdges or \Edges#OrthoDiagonalEdges types.

        Note that the inverse @b result @/b of \with_angle_both is delivered by \with_angle with the inverse flag set as edge pairs are unselected when one edge fails to meet the criterion.
        I.e

        @code
        result = edge_pairs.with_angle_both(RBA::Edges::Ortho, false)
        others = edge_pairs.with_angle(RBA::Edges::Ortho, true)
        @/code

        This method has been added in version 0.28.
        """
    @overload
    def with_angle_both(self, min_angle: float, max_angle: float, inverse: bool, include_min_angle: Optional[bool] = ..., include_max_angle: Optional[bool] = ...) -> EdgePairs:
        r"""
        @brief Filter the edge pairs by orientation of both of their edges
        Filters the edge pairs in the edge pair collection by orientation. If "inverse" is false, only edge pairs with both edges having an angle between min_angle and max_angle are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        With "include_min_angle" set to true (the default), the minimum angle is included in the criterion while with false, the minimum angle itself is not included. Same for "include_max_angle" where the default is false, meaning the maximum angle is not included in the range.

        Note that the inverse @b result @/b of \with_angle_both is delivered by \with_angle with the inverse flag set as edge pairs are unselected when one edge fails to meet the criterion.
        I.e

        @code
        result = edge_pairs.with_angle_both(0, 45, false)
        others = edge_pairs.with_angle(0, 45, true)
        @/code

        This method has been added in version 0.27.1.
        """
    @overload
    def with_area(self, area: int, inverse: bool) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by the enclosed area
        Filters the edge pairs in the edge pair collection by enclosed area. If "inverse" is false, only edge pairs with the given area are returned. If "inverse" is true, edge pairs not with the given area are returned.

        This method has been added in version 0.27.2.
        """
    @overload
    def with_area(self, min_area: int, max_area: int, inverse: bool) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by the enclosed area
        Filters the edge pairs in the edge pair collection by enclosed area. If "inverse" is false, only edge pairs with an area between min_area and max_area (max_area itself is excluded) are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        This method has been added in version 0.27.2.
        """
    @overload
    def with_distance(self, distance: int, inverse: bool) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by the distance of the edges
        Filters the edge pairs in the edge pair collection by distance of the edges. If "inverse" is false, only edge pairs where both edges have the given distance are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        Distance is measured as the shortest distance between any of the points on the edges.

        This method has been added in version 0.27.1.
        """
    @overload
    def with_distance(self, min_distance: Any, max_distance: Any, inverse: bool) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by the distance of the edges
        Filters the edge pairs in the edge pair collection by distance of the edges. If "inverse" is false, only edge pairs where both edges have a distance between min_distance and max_distance (max_distance itself is excluded) are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        Distance is measured as the shortest distance between any of the points on the edges.

        This method has been added in version 0.27.1.
        """
    @overload
    def with_internal_angle(self, angle: float, inverse: bool) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by the angle between their edges
        Filters the edge pairs in the edge pair collection by the angle between their edges. If "inverse" is false, only edge pairs with the given angle are returned. If "inverse" is true, edge pairs not with the given angle are returned.

        The angle is measured between the two edges. It is between 0 (parallel or anti-parallel edges) and 90 degree (perpendicular edges).

        This method has been added in version 0.27.2.
        """
    @overload
    def with_internal_angle(self, min_angle: float, max_angle: float, inverse: bool, include_min_angle: Optional[bool] = ..., include_max_angle: Optional[bool] = ...) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by the angle between their edges
        Filters the edge pairs in the edge pair collection by the angle between their edges. If "inverse" is false, only edge pairs with an angle between min_angle and max_angle (max_angle itself is excluded) are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        The angle is measured between the two edges. It is between 0 (parallel or anti-parallel edges) and 90 degree (perpendicular edges).

        With "include_min_angle" set to true (the default), the minimum angle is included in the criterion while with false, the minimum angle itself is not included. Same for "include_max_angle" where the default is false, meaning the maximum angle is not included in the range.

        This method has been added in version 0.27.2.
        """
    @overload
    def with_length(self, length: int, inverse: bool) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by length of one of their edges
        Filters the edge pairs in the edge pair collection by length of at least one of their edges. If "inverse" is false, only edge pairs with at least one edge having the given length are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        This method has been added in version 0.27.1.
        """
    @overload
    def with_length(self, min_length: Any, max_length: Any, inverse: bool) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by length of one of their edges
        Filters the edge pairs in the edge pair collection by length of at least one of their edges. If "inverse" is false, only edge pairs with at least one edge having a length between min_length and max_length (excluding max_length itself) are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        If you don't want to specify a lower or upper limit, pass nil to that parameter.

        This method has been added in version 0.27.1.
        """
    @overload
    def with_length_both(self, length: int, inverse: bool) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by length of both of their edges
        Filters the edge pairs in the edge pair collection by length of both of their edges. If "inverse" is false, only edge pairs where both edges have the given length are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        This method has been added in version 0.27.1.
        """
    @overload
    def with_length_both(self, min_length: Any, max_length: Any, inverse: bool) -> EdgePairs:
        r"""
        @brief Filters the edge pairs by length of both of their edges
        Filters the edge pairs in the edge pair collection by length of both of their edges. If "inverse" is false, only edge pairs with both edges having a length between min_length and max_length (excluding max_length itself) are returned. If "inverse" is true, edge pairs not fulfilling this criterion are returned.

        If you don't want to specify a lower or upper limit, pass nil to that parameter.

        This method has been added in version 0.27.1.
        """

class EdgeProcessor:
    r"""
    @brief The edge processor (boolean, sizing, merge)

    The edge processor implements the boolean and edge set operations (size, merge). Because the edge processor might allocate resources which can be reused in later operations, it is implemented as an object that can be used several times.

    Here is a simple example of how to use the edge processor:

    @code
    ep = RBA::EdgeProcessor::new
    # Prepare two boxes
    a = [ RBA::Polygon::new(RBA::Box::new(0, 0, 300, 300)) ]
    b = [ RBA::Polygon::new(RBA::Box::new(100, 100, 200, 200)) ]
    # Run an XOR -> creates a polygon with a hole, since the 'resolve_holes' parameter
    # is false:
    out = ep.boolean_p2p(a, b, RBA::EdgeProcessor::ModeXor, false, false)
    out.to_s    # -> [(0,0;0,300;300,300;300,0/100,100;200,100;200,200;100,200)]
    @/code
    """
    ModeANotB: ClassVar[int]
    r"""
    @brief boolean method's mode value for A NOT B operation
    """
    ModeAnd: ClassVar[int]
    r"""
    @brief boolean method's mode value for AND operation
    """
    ModeBNotA: ClassVar[int]
    r"""
    @brief boolean method's mode value for B NOT A operation
    """
    ModeOr: ClassVar[int]
    r"""
    @brief boolean method's mode value for OR operation
    """
    ModeXor: ClassVar[int]
    r"""
    @brief boolean method's mode value for XOR operation
    """
    @classmethod
    def mode_and(cls) -> int:
        r"""
        @brief boolean method's mode value for AND operation
        """
    @classmethod
    def mode_anotb(cls) -> int:
        r"""
        @brief boolean method's mode value for A NOT B operation
        """
    @classmethod
    def mode_bnota(cls) -> int:
        r"""
        @brief boolean method's mode value for B NOT A operation
        """
    @classmethod
    def mode_or(cls) -> int:
        r"""
        @brief boolean method's mode value for OR operation
        """
    @classmethod
    def mode_xor(cls) -> int:
        r"""
        @brief boolean method's mode value for XOR operation
        """
    @classmethod
    def new(cls) -> EdgeProcessor:
        r"""
        @brief Creates a new object of this class
        """
    def __copy__(self) -> EdgeProcessor:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> EdgeProcessor:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: EdgeProcessor) -> None:
        r"""
        @brief Assigns another object to self
        """
    @overload
    def boolean(self, a: Sequence[Edge], b: Sequence[Edge], mode: int) -> List[Edge]:
        r"""
        @brief Boolean operation for a set of given edges, creating edges

        This method computes the result for the given boolean operation on two sets of edges.
        The input edges must form closed contours where holes and hulls must be oriented differently. 
        The input edges are processed with a simple non-zero wrap count rule as a whole.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        Prior to version 0.21 this method was called 'boolean'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param a The input edges (first operand)
        @param b The input edges (second operand)
        @param mode The boolean mode (one of the Mode.. values)
        @return The output edges
        """
    @overload
    def boolean(self, a: Sequence[Polygon], b: Sequence[Polygon], mode: int) -> List[Edge]:
        r"""
        @brief Boolean operation for a set of given polygons, creating edges

        This method computes the result for the given boolean operation on two sets of polygons.
        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a Boolean operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'boolean'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param a The input polygons (first operand)
        @param b The input polygons (second operand)
        @param mode The boolean mode
        @return The output edges
        """
    def boolean_e2e(self, a: Sequence[Edge], b: Sequence[Edge], mode: int) -> List[Edge]:
        r"""
        @brief Boolean operation for a set of given edges, creating edges

        This method computes the result for the given boolean operation on two sets of edges.
        The input edges must form closed contours where holes and hulls must be oriented differently. 
        The input edges are processed with a simple non-zero wrap count rule as a whole.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        Prior to version 0.21 this method was called 'boolean'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param a The input edges (first operand)
        @param b The input edges (second operand)
        @param mode The boolean mode (one of the Mode.. values)
        @return The output edges
        """
    def boolean_e2p(self, a: Sequence[Edge], b: Sequence[Edge], mode: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Boolean operation for a set of given edges, creating polygons

        This method computes the result for the given boolean operation on two sets of edges.
        The input edges must form closed contours where holes and hulls must be oriented differently. 
        The input edges are processed with a simple non-zero wrap count rule as a whole.

        This method produces polygons on output and allows fine-tuning of the parameters for that purpose.

        Prior to version 0.21 this method was called 'boolean_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param a The input polygons (first operand)
        @param b The input polygons (second operand)
        @param mode The boolean mode (one of the Mode.. values)
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    def boolean_p2e(self, a: Sequence[Polygon], b: Sequence[Polygon], mode: int) -> List[Edge]:
        r"""
        @brief Boolean operation for a set of given polygons, creating edges

        This method computes the result for the given boolean operation on two sets of polygons.
        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a Boolean operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'boolean'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param a The input polygons (first operand)
        @param b The input polygons (second operand)
        @param mode The boolean mode
        @return The output edges
        """
    def boolean_p2p(self, a: Sequence[Polygon], b: Sequence[Polygon], mode: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Boolean operation for a set of given polygons, creating polygons

        This method computes the result for the given boolean operation on two sets of polygons.
        This method produces polygons on output and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a Boolean operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'boolean_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param a The input polygons (first operand)
        @param b The input polygons (second operand)
        @param mode The boolean mode (one of the Mode.. values)
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def boolean_to_polygon(self, a: Sequence[Edge], b: Sequence[Edge], mode: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Boolean operation for a set of given edges, creating polygons

        This method computes the result for the given boolean operation on two sets of edges.
        The input edges must form closed contours where holes and hulls must be oriented differently. 
        The input edges are processed with a simple non-zero wrap count rule as a whole.

        This method produces polygons on output and allows fine-tuning of the parameters for that purpose.

        Prior to version 0.21 this method was called 'boolean_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param a The input polygons (first operand)
        @param b The input polygons (second operand)
        @param mode The boolean mode (one of the Mode.. values)
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def boolean_to_polygon(self, a: Sequence[Polygon], b: Sequence[Polygon], mode: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Boolean operation for a set of given polygons, creating polygons

        This method computes the result for the given boolean operation on two sets of polygons.
        This method produces polygons on output and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a Boolean operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'boolean_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param a The input polygons (first operand)
        @param b The input polygons (second operand)
        @param mode The boolean mode (one of the Mode.. values)
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def disable_progress(self) -> None:
        r"""
        @brief Disable progress reporting
        Calling this method will stop the edge processor from showing a progress bar. See \enable_progress.

        This method has been introduced in version 0.23.
        """
    def dup(self) -> EdgeProcessor:
        r"""
        @brief Creates a copy of self
        """
    def enable_progress(self, label: str) -> None:
        r"""
        @brief Enable progress reporting
        After calling this method, the edge processor will report the progress through a progress bar.
        The label is a text which is put in front of the progress bar.
        Using a progress bar will imply a performance penalty of a few percent typically.

        This method has been introduced in version 0.23.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def merge(self, in_: Sequence[Polygon], min_wc: int) -> List[Edge]:
        r"""
        @brief Merge the given polygons 

        In contrast to "simple_merge", this merge implementation considers each polygon individually before merging them.
        Thus self-overlaps are effectively removed before the output is computed and holes are correctly merged with the
        hull. In addition, this method allows selecting areas with a higher wrap count which in turn allows computing overlaps
        of polygons on the same layer. Because this method merges the polygons before the overlap is computed, self-overlapping
        polygons do not contribute to higher wrap count areas.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        Prior to version 0.21 this method was called 'merge'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param min_wc The minimum wrap count for output (0: all polygons, 1: at least two overlapping)
        @return The output edges
        """
    def merge_p2e(self, in_: Sequence[Polygon], min_wc: int) -> List[Edge]:
        r"""
        @brief Merge the given polygons 

        In contrast to "simple_merge", this merge implementation considers each polygon individually before merging them.
        Thus self-overlaps are effectively removed before the output is computed and holes are correctly merged with the
        hull. In addition, this method allows selecting areas with a higher wrap count which in turn allows computing overlaps
        of polygons on the same layer. Because this method merges the polygons before the overlap is computed, self-overlapping
        polygons do not contribute to higher wrap count areas.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        Prior to version 0.21 this method was called 'merge'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param min_wc The minimum wrap count for output (0: all polygons, 1: at least two overlapping)
        @return The output edges
        """
    def merge_p2p(self, in_: Sequence[Polygon], min_wc: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Merge the given polygons 

        In contrast to "simple_merge", this merge implementation considers each polygon individually before merging them.
        Thus self-overlaps are effectively removed before the output is computed and holes are correctly merged with the
        hull. In addition, this method allows selecting areas with a higher wrap count which in turn allows computing overlaps
        of polygons on the same layer. Because this method merges the polygons before the overlap is computed, self-overlapping
        polygons do not contribute to higher wrap count areas.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        Prior to version 0.21 this method was called 'merge_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param min_wc The minimum wrap count for output (0: all polygons, 1: at least two overlapping)
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    def merge_to_polygon(self, in_: Sequence[Polygon], min_wc: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Merge the given polygons 

        In contrast to "simple_merge", this merge implementation considers each polygon individually before merging them.
        Thus self-overlaps are effectively removed before the output is computed and holes are correctly merged with the
        hull. In addition, this method allows selecting areas with a higher wrap count which in turn allows computing overlaps
        of polygons on the same layer. Because this method merges the polygons before the overlap is computed, self-overlapping
        polygons do not contribute to higher wrap count areas.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        Prior to version 0.21 this method was called 'merge_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param min_wc The minimum wrap count for output (0: all polygons, 1: at least two overlapping)
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def simple_merge(self, in_: Sequence[Edge]) -> List[Edge]:
        r"""
        @brief Merge the given edges in a simple "non-zero wrapcount" fashion

        The edges provided must form valid closed contours. Contours oriented differently "cancel" each other. 
        Overlapping contours are merged when the orientation is the same.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'simple_merge'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input edges
        @return The output edges
        """
    @overload
    def simple_merge(self, in_: Sequence[Polygon]) -> List[Edge]:
        r"""
        @brief Merge the given polygons in a simple "non-zero wrapcount" fashion

        The wrapcount is computed over all polygons, i.e. overlapping polygons may "cancel" if they
        have different orientation (since a polygon is oriented by construction that is not easy to achieve).
        The other merge operation provided for this purpose is "merge" which normalizes each polygon individually before
        merging them. "simple_merge" is somewhat faster and consumes less memory.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'simple_merge'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @return The output edges
        """
    @overload
    def simple_merge(self, in_: Sequence[Edge], mode: int) -> List[Edge]:
        r"""
        @brief Merge the given polygons and specify the merge mode

        The edges provided must form valid closed contours. Contours oriented differently "cancel" each other. 
        Overlapping contours are merged when the orientation is the same.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        This method has been added in version 0.22.

        The mode specifies the rule to use when producing output. A value of 0 specifies the even-odd rule. A positive value specifies the wrap count threshold (positive only). A negative value specifies the threshold of the absolute value of the wrap count (i.e. -1 is non-zero rule).

        @param mode See description
        @param in The input edges
        @return The output edges
        """
    @overload
    def simple_merge(self, in_: Sequence[Polygon], mode: int) -> List[Edge]:
        r"""
        @brief Merge the given polygons and specify the merge mode

        The wrapcount is computed over all polygons, i.e. overlapping polygons may "cancel" if they
        have different orientation (since a polygon is oriented by construction that is not easy to achieve).
        The other merge operation provided for this purpose is "merge" which normalizes each polygon individually before
        merging them. "simple_merge" is somewhat faster and consumes less memory.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        This method has been added in version 0.22.

        The mode specifies the rule to use when producing output. A value of 0 specifies the even-odd rule. A positive value specifies the wrap count threshold (positive only). A negative value specifies the threshold of the absolute value of the wrap count (i.e. -1 is non-zero rule).

        @param mode See description
        @param in The input polygons
        @return The output edges
        """
    @overload
    def simple_merge_e2e(self, in_: Sequence[Edge]) -> List[Edge]:
        r"""
        @brief Merge the given edges in a simple "non-zero wrapcount" fashion

        The edges provided must form valid closed contours. Contours oriented differently "cancel" each other. 
        Overlapping contours are merged when the orientation is the same.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'simple_merge'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input edges
        @return The output edges
        """
    @overload
    def simple_merge_e2e(self, in_: Sequence[Edge], mode: int) -> List[Edge]:
        r"""
        @brief Merge the given polygons and specify the merge mode

        The edges provided must form valid closed contours. Contours oriented differently "cancel" each other. 
        Overlapping contours are merged when the orientation is the same.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        This method has been added in version 0.22.

        The mode specifies the rule to use when producing output. A value of 0 specifies the even-odd rule. A positive value specifies the wrap count threshold (positive only). A negative value specifies the threshold of the absolute value of the wrap count (i.e. -1 is non-zero rule).

        @param mode See description
        @param in The input edges
        @return The output edges
        """
    @overload
    def simple_merge_e2p(self, in_: Sequence[Edge], resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Merge the given edges in a simple "non-zero wrapcount" fashion into polygons

        The edges provided must form valid closed contours. Contours oriented differently "cancel" each other. 
        Overlapping contours are merged when the orientation is the same.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'simple_merge_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input edges
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def simple_merge_e2p(self, in_: Sequence[Edge], resolve_holes: bool, min_coherence: bool, mode: int) -> List[Polygon]:
        r"""
        @brief Merge the given polygons and specify the merge mode

        The edges provided must form valid closed contours. Contours oriented differently "cancel" each other. 
        Overlapping contours are merged when the orientation is the same.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        This method has been added in version 0.22.

        The mode specifies the rule to use when producing output. A value of 0 specifies the even-odd rule. A positive value specifies the wrap count threshold (positive only). A negative value specifies the threshold of the absolute value of the wrap count (i.e. -1 is non-zero rule).

        @param mode See description
        @param in The input edges
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def simple_merge_p2e(self, in_: Sequence[Polygon]) -> List[Edge]:
        r"""
        @brief Merge the given polygons in a simple "non-zero wrapcount" fashion

        The wrapcount is computed over all polygons, i.e. overlapping polygons may "cancel" if they
        have different orientation (since a polygon is oriented by construction that is not easy to achieve).
        The other merge operation provided for this purpose is "merge" which normalizes each polygon individually before
        merging them. "simple_merge" is somewhat faster and consumes less memory.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'simple_merge'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @return The output edges
        """
    @overload
    def simple_merge_p2e(self, in_: Sequence[Polygon], mode: int) -> List[Edge]:
        r"""
        @brief Merge the given polygons and specify the merge mode

        The wrapcount is computed over all polygons, i.e. overlapping polygons may "cancel" if they
        have different orientation (since a polygon is oriented by construction that is not easy to achieve).
        The other merge operation provided for this purpose is "merge" which normalizes each polygon individually before
        merging them. "simple_merge" is somewhat faster and consumes less memory.

        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        This method has been added in version 0.22.

        The mode specifies the rule to use when producing output. A value of 0 specifies the even-odd rule. A positive value specifies the wrap count threshold (positive only). A negative value specifies the threshold of the absolute value of the wrap count (i.e. -1 is non-zero rule).

        @param mode See description
        @param in The input polygons
        @return The output edges
        """
    @overload
    def simple_merge_p2p(self, in_: Sequence[Polygon], resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Merge the given polygons in a simple "non-zero wrapcount" fashion into polygons

        The wrapcount is computed over all polygons, i.e. overlapping polygons may "cancel" if they
        have different orientation (since a polygon is oriented by construction that is not easy to achieve).
        The other merge operation provided for this purpose is "merge" which normalizes each polygon individually before
        merging them. "simple_merge" is somewhat faster and consumes less memory.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'simple_merge_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def simple_merge_p2p(self, in_: Sequence[Polygon], resolve_holes: bool, min_coherence: bool, mode: int) -> List[Polygon]:
        r"""
        @brief Merge the given polygons and specify the merge mode

        The wrapcount is computed over all polygons, i.e. overlapping polygons may "cancel" if they
        have different orientation (since a polygon is oriented by construction that is not easy to achieve).
        The other merge operation provided for this purpose is "merge" which normalizes each polygon individually before
        merging them. "simple_merge" is somewhat faster and consumes less memory.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        This method has been added in version 0.22.

        The mode specifies the rule to use when producing output. A value of 0 specifies the even-odd rule. A positive value specifies the wrap count threshold (positive only). A negative value specifies the threshold of the absolute value of the wrap count (i.e. -1 is non-zero rule).

        @param mode See description
        @param in The input polygons
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def simple_merge_to_polygon(self, in_: Sequence[Edge], resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Merge the given edges in a simple "non-zero wrapcount" fashion into polygons

        The edges provided must form valid closed contours. Contours oriented differently "cancel" each other. 
        Overlapping contours are merged when the orientation is the same.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'simple_merge_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input edges
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def simple_merge_to_polygon(self, in_: Sequence[Polygon], resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Merge the given polygons in a simple "non-zero wrapcount" fashion into polygons

        The wrapcount is computed over all polygons, i.e. overlapping polygons may "cancel" if they
        have different orientation (since a polygon is oriented by construction that is not easy to achieve).
        The other merge operation provided for this purpose is "merge" which normalizes each polygon individually before
        merging them. "simple_merge" is somewhat faster and consumes less memory.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        Prior to version 0.21 this method was called 'simple_merge_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def simple_merge_to_polygon(self, in_: Sequence[Edge], resolve_holes: bool, min_coherence: bool, mode: int) -> List[Polygon]:
        r"""
        @brief Merge the given polygons and specify the merge mode

        The edges provided must form valid closed contours. Contours oriented differently "cancel" each other. 
        Overlapping contours are merged when the orientation is the same.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        This method has been added in version 0.22.

        The mode specifies the rule to use when producing output. A value of 0 specifies the even-odd rule. A positive value specifies the wrap count threshold (positive only). A negative value specifies the threshold of the absolute value of the wrap count (i.e. -1 is non-zero rule).

        @param mode See description
        @param in The input edges
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def simple_merge_to_polygon(self, in_: Sequence[Polygon], resolve_holes: bool, min_coherence: bool, mode: int) -> List[Polygon]:
        r"""
        @brief Merge the given polygons and specify the merge mode

        The wrapcount is computed over all polygons, i.e. overlapping polygons may "cancel" if they
        have different orientation (since a polygon is oriented by construction that is not easy to achieve).
        The other merge operation provided for this purpose is "merge" which normalizes each polygon individually before
        merging them. "simple_merge" is somewhat faster and consumes less memory.

        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        This is a convenience method that bundles filling of the edges, processing with
        a SimpleMerge operator and puts the result into an output vector.

        This method has been added in version 0.22.

        The mode specifies the rule to use when producing output. A value of 0 specifies the even-odd rule. A positive value specifies the wrap count threshold (positive only). A negative value specifies the threshold of the absolute value of the wrap count (i.e. -1 is non-zero rule).

        @param mode See description
        @param in The input polygons
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def size(self, in_: Sequence[Polygon], d: int, mode: int) -> List[Edge]:
        r"""
        @brief Size the given polygons (isotropic)

        This method is equivalent to calling the anisotropic version with identical dx and dy.

        Prior to version 0.21 this method was called 'size'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param d The sizing value in x direction
        @param mode The sizing mode
        @return The output edges
        """
    @overload
    def size(self, in_: Sequence[Polygon], dx: int, dy: int, mode: int) -> List[Edge]:
        r"""
        @brief Size the given polygons 

        This method sizes a set of polygons. Before the sizing is applied, the polygons are merged. After that, sizing is applied 
        on the individual result polygons of the merge step. The result may contain overlapping contours, but no self-overlaps. 

        dx and dy describe the sizing. A positive value indicates oversize (outwards) while a negative one describes undersize (inwards).
        The sizing applied can be chosen differently in x and y direction. In this case, the sign must be identical for both
        dx and dy.

        The 'mode' parameter describes the corner fill strategy. Mode 0 connects all corner segments directly. Mode 1 is the 'octagon' strategy in which square corners are interpolated with a partial octagon. Mode 2 is the standard mode in which corners are filled by expanding edges unless these edges form a sharp bend with an angle of more than 90 degree. In that case, the corners are cut off. In Mode 3, no cutoff occurs up to a bending angle of 135 degree. Mode 4 and 5 are even more aggressive and allow very sharp bends without cutoff. This strategy may produce long spikes on sharply bending corners. 
        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        Prior to version 0.21 this method was called 'size'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param dx The sizing value in x direction
        @param dy The sizing value in y direction
        @param mode The sizing mode (standard is 2)
        @return The output edges
        """
    @overload
    def size_p2e(self, in_: Sequence[Polygon], d: int, mode: int) -> List[Edge]:
        r"""
        @brief Size the given polygons (isotropic)

        This method is equivalent to calling the anisotropic version with identical dx and dy.

        Prior to version 0.21 this method was called 'size'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param d The sizing value in x direction
        @param mode The sizing mode
        @return The output edges
        """
    @overload
    def size_p2e(self, in_: Sequence[Polygon], dx: int, dy: int, mode: int) -> List[Edge]:
        r"""
        @brief Size the given polygons 

        This method sizes a set of polygons. Before the sizing is applied, the polygons are merged. After that, sizing is applied 
        on the individual result polygons of the merge step. The result may contain overlapping contours, but no self-overlaps. 

        dx and dy describe the sizing. A positive value indicates oversize (outwards) while a negative one describes undersize (inwards).
        The sizing applied can be chosen differently in x and y direction. In this case, the sign must be identical for both
        dx and dy.

        The 'mode' parameter describes the corner fill strategy. Mode 0 connects all corner segments directly. Mode 1 is the 'octagon' strategy in which square corners are interpolated with a partial octagon. Mode 2 is the standard mode in which corners are filled by expanding edges unless these edges form a sharp bend with an angle of more than 90 degree. In that case, the corners are cut off. In Mode 3, no cutoff occurs up to a bending angle of 135 degree. Mode 4 and 5 are even more aggressive and allow very sharp bends without cutoff. This strategy may produce long spikes on sharply bending corners. 
        The result is presented as a set of edges forming closed contours. Hulls are oriented clockwise while
        holes are oriented counter-clockwise.

        Prior to version 0.21 this method was called 'size'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param dx The sizing value in x direction
        @param dy The sizing value in y direction
        @param mode The sizing mode (standard is 2)
        @return The output edges
        """
    @overload
    def size_p2p(self, in_: Sequence[Polygon], d: int, mode: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Size the given polygons into polygons (isotropic)

        This method is equivalent to calling the anisotropic version with identical dx and dy.

        Prior to version 0.21 this method was called 'size_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param d The sizing value in x direction
        @param mode The sizing mode
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def size_p2p(self, in_: Sequence[Polygon], dx: int, dy: int, mode: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Size the given polygons into polygons

        This method sizes a set of polygons. Before the sizing is applied, the polygons are merged. After that, sizing is applied 
        on the individual result polygons of the merge step. The result may contain overlapping polygons, but no self-overlapping ones. 
        Polygon overlap occurs if the polygons are close enough, so a positive sizing makes polygons overlap.

        dx and dy describe the sizing. A positive value indicates oversize (outwards) while a negative one describes undersize (inwards).
        The sizing applied can be chosen differently in x and y direction. In this case, the sign must be identical for both
        dx and dy.

        The 'mode' parameter describes the corner fill strategy. Mode 0 connects all corner segments directly. Mode 1 is the 'octagon' strategy in which square corners are interpolated with a partial octagon. Mode 2 is the standard mode in which corners are filled by expanding edges unless these edges form a sharp bend with an angle of more than 90 degree. In that case, the corners are cut off. In Mode 3, no cutoff occurs up to a bending angle of 135 degree. Mode 4 and 5 are even more aggressive and allow very sharp bends without cutoff. This strategy may produce long spikes on sharply bending corners. 
        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        Prior to version 0.21 this method was called 'size_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param dx The sizing value in x direction
        @param dy The sizing value in y direction
        @param mode The sizing mode (standard is 2)
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def size_to_polygon(self, in_: Sequence[Polygon], d: int, mode: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Size the given polygons into polygons (isotropic)

        This method is equivalent to calling the anisotropic version with identical dx and dy.

        Prior to version 0.21 this method was called 'size_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param d The sizing value in x direction
        @param mode The sizing mode
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """
    @overload
    def size_to_polygon(self, in_: Sequence[Polygon], dx: int, dy: int, mode: int, resolve_holes: bool, min_coherence: bool) -> List[Polygon]:
        r"""
        @brief Size the given polygons into polygons

        This method sizes a set of polygons. Before the sizing is applied, the polygons are merged. After that, sizing is applied 
        on the individual result polygons of the merge step. The result may contain overlapping polygons, but no self-overlapping ones. 
        Polygon overlap occurs if the polygons are close enough, so a positive sizing makes polygons overlap.

        dx and dy describe the sizing. A positive value indicates oversize (outwards) while a negative one describes undersize (inwards).
        The sizing applied can be chosen differently in x and y direction. In this case, the sign must be identical for both
        dx and dy.

        The 'mode' parameter describes the corner fill strategy. Mode 0 connects all corner segments directly. Mode 1 is the 'octagon' strategy in which square corners are interpolated with a partial octagon. Mode 2 is the standard mode in which corners are filled by expanding edges unless these edges form a sharp bend with an angle of more than 90 degree. In that case, the corners are cut off. In Mode 3, no cutoff occurs up to a bending angle of 135 degree. Mode 4 and 5 are even more aggressive and allow very sharp bends without cutoff. This strategy may produce long spikes on sharply bending corners. 
        This method produces polygons and allows fine-tuning of the parameters for that purpose.

        Prior to version 0.21 this method was called 'size_to_polygon'. Is was renamed to avoid ambiguities for empty input arrays. The old version is still available but deprecated.

        @param in The input polygons
        @param dx The sizing value in x direction
        @param dy The sizing value in y direction
        @param mode The sizing mode (standard is 2)
        @param resolve_holes true, if holes should be resolved into the hull
        @param min_coherence true, if touching corners should be resolved into less connected contours
        @return The output polygons
        """

class Edges(ShapeCollection):
    r"""
    @brief A collection of edges (Not necessarily describing closed contours)


    This class was introduced to simplify operations on edges sets. See \Edge for a description of the individual edge object. The edge collection contains an arbitrary number of edges and supports operations to select edges by various criteria, produce polygons from the edges by applying an extension, filtering edges against other edges collections and checking geometrical relations to other edges (DRC functionality).

    The edge collection is supposed to work closely with the \Region polygon set. Both are related, although the edge collection has a lower rank since it potentially represents a disconnected collection of edges. Edge collections may form closed contours, for example immediately after they have been derived from a polygon set using \Region#edges. But this state is volatile and can easily be destroyed by filtering edges. Hence the connected state does not play an important role in the edge collection's API.

    Edge collections may also contain points (degenerated edges with identical start and end points). Such point-like objects participate in some although not all methods of the edge collection class. 
    Edge collections can be used in two different flavors: in raw mode or merged semantics. With merged semantics (the default), connected edges are considered to belong together and are effectively merged.
    Overlapping parts are counted once in that mode. Dot-like edges are not considered in merged semantics.
    In raw mode (without merged semantics), each edge is considered as it is. Overlaps between edges
    may exists and merging has to be done explicitly using the \merge method. The semantics can be
    selected using \merged_semantics=.


    This class has been introduced in version 0.23.
    """
    class EdgeType:
        r"""
        @brief This enum specifies the the edge type for edge angle filters.

        This enum was introduced in version 0.28.
        """
        DiagonalEdges: ClassVar[Edges.EdgeType]
        r"""
        @brief Diagonal edges are selected (-45 and 45 degree)
        """
        OrthoDiagonalEdges: ClassVar[Edges.EdgeType]
        r"""
        @brief Diagonal or orthogonal edges are selected (0, 90, -45 and 45 degree)
        """
        OrthoEdges: ClassVar[Edges.EdgeType]
        r"""
        @brief Horizontal and vertical edges are selected
        """
        @overload
        @classmethod
        def new(cls, i: int) -> Edges.EdgeType:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        @classmethod
        def new(cls, s: str) -> Edges.EdgeType:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares two enums
            """
        @overload
        def __init__(self, i: int) -> None:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        def __init__(self, s: str) -> None:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __lt__(self, other: int) -> bool:
            r"""
            @brief Returns true if the enum is less (in the enum symbol order) than the integer value
            """
        @overload
        def __lt__(self, other: Edges.EdgeType) -> bool:
            r"""
            @brief Returns true if the first enum is less (in the enum symbol order) than the second
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer for inequality
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares two enums for inequality
            """
        def __repr__(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def __str__(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
        def inspect(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def to_i(self) -> int:
            r"""
            @brief Gets the integer value from the enum
            """
        def to_s(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
    DiagonalEdges: ClassVar[Edges.EdgeType]
    r"""
    @brief Diagonal edges are selected (-45 and 45 degree)
    """
    Euclidian: ClassVar[Region.Metrics]
    r"""
    @brief Specifies Euclidian metrics for the check functions
    This value can be used for the metrics parameter in the check functions, i.e. \width_check. This value specifies Euclidian metrics, i.e. the distance between two points is measured by:

    @code
    d = sqrt(dx^2 + dy^2)
    @/code

    All points within a circle with radius d around one point are considered to have a smaller distance than d.
    """
    OrthoDiagonalEdges: ClassVar[Edges.EdgeType]
    r"""
    @brief Diagonal or orthogonal edges are selected (0, 90, -45 and 45 degree)
    """
    OrthoEdges: ClassVar[Edges.EdgeType]
    r"""
    @brief Horizontal and vertical edges are selected
    """
    Projection: ClassVar[Region.Metrics]
    r"""
    @brief Specifies projected distance metrics for the check functions
    This value can be used for the metrics parameter in the check functions, i.e. \width_check. This value specifies projected metrics, i.e. the distance is defined as the minimum distance measured perpendicular to one edge. That implies that the distance is defined only where two edges have a non-vanishing projection onto each other.
    """
    Square: ClassVar[Region.Metrics]
    r"""
    @brief Specifies square metrics for the check functions
    This value can be used for the metrics parameter in the check functions, i.e. \width_check. This value specifies square metrics, i.e. the distance between two points is measured by:

    @code
    d = max(abs(dx), abs(dy))
    @/code

    All points within a square with length 2*d around one point are considered to have a smaller distance than d in this metrics.
    """
    merged_semantics: bool
    r"""
    Getter:
    @brief Gets a flag indicating whether merged semantics is enabled
    See \merged_semantics= for a description of this attribute.

    Setter:
    @brief Enable or disable merged semantics
    If merged semantics is enabled (the default), colinear, connected or overlapping edges will be considered
    as single edges.
    """
    @overload
    @classmethod
    def new(cls) -> Edges:
        r"""
        @brief Default constructor

        This constructor creates an empty edge collection.
        """
    @overload
    @classmethod
    def new(cls, array: Sequence[Edge]) -> Edges:
        r"""
        @brief Constructor from an edge array

        This constructor creates an edge collection from an array of edges.
        """
    @overload
    @classmethod
    def new(cls, array: Sequence[Polygon]) -> Edges:
        r"""
        @brief Constructor from a polygon array

        This constructor creates an edge collection from an array of polygons.
        The edges form the contours of the polygons.
        """
    @overload
    @classmethod
    def new(cls, box: Box) -> Edges:
        r"""
        @brief Box constructor

        This constructor creates an edge collection from a box.
        The edges form the contour of the box.
        """
    @overload
    @classmethod
    def new(cls, edge: Edge) -> Edges:
        r"""
        @brief Constructor from a single edge

        This constructor creates an edge collection with a single edge.
        """
    @overload
    @classmethod
    def new(cls, path: Path) -> Edges:
        r"""
        @brief Path constructor

        This constructor creates an edge collection from a path.
        The edges form the contour of the path.
        """
    @overload
    @classmethod
    def new(cls, polygon: Polygon) -> Edges:
        r"""
        @brief Polygon constructor

        This constructor creates an edge collection from a polygon.
        The edges form the contour of the polygon.
        """
    @overload
    @classmethod
    def new(cls, polygon: SimplePolygon) -> Edges:
        r"""
        @brief Simple polygon constructor

        This constructor creates an edge collection from a simple polygon.
        The edges form the contour of the polygon.
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator, as_edges: Optional[bool] = ...) -> Edges:
        r"""
        @brief Constructor of a flat edge collection from a hierarchical shape set

        This constructor creates an edge collection from the shapes delivered by the given recursive shape iterator.
        It feeds the shapes from a hierarchy of cells into a flat edge set.

        Text objects are not inserted, because they cannot be converted to edges.
        Edge objects are inserted as such. If "as_edges" is true, "solid" objects (boxes, polygons, paths) are converted to edges which form the hull of these objects. If "as_edges" is false, solid objects are ignored.

        @code
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        r = RBA::Edges::new(layout.begin_shapes(cell, layer), false)
        @/code
        """
    @overload
    @classmethod
    def new(cls, shapes: Shapes, as_edges: Optional[bool] = ...) -> Edges:
        r"""
        @brief Constructor of a flat edge collection from a \Shapes container

        If 'as_edges' is true, the shapes from the container will be converted to edges (i.e. polygon contours to edges). Otherwise, only edges will be taken from the container.

        This method has been introduced in version 0.26.
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore, as_edges: Optional[bool] = ...) -> Edges:
        r"""
        @brief Constructor of a hierarchical edge collection

        This constructor creates an edge collection from the shapes delivered by the given recursive shape iterator.
        It feeds the shapes from a hierarchy of cells into the hierarchical edge set.
        The edges remain within their original hierarchy unless other operations require the edges to be moved in the hierarchy.

        Text objects are not inserted, because they cannot be converted to edges.
        Edge objects are inserted as such. If "as_edges" is true, "solid" objects (boxes, polygons, paths) are converted to edges which form the hull of these objects. If "as_edges" is false, solid objects are ignored.

        @code
        dss    = RBA::DeepShapeStore::new
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        r = RBA::Edges::new(layout.begin_shapes(cell, layer), dss, false)
        @/code
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator, expr: str, as_pattern: Optional[bool] = ...) -> Edges:
        r"""
        @brief Constructor from a text set

        @param shape_iterator The iterator from which to derive the texts
        @param expr The selection string
        @param as_pattern If true, the selection string is treated as a glob pattern. Otherwise the match is exact.

        This special constructor will create dot-like edges from the text objects delivered by the shape iterator. Each text object will give a degenerated edge (a dot) that represents the text origin.
        Texts can be selected by their strings - either through a glob pattern or by exact comparison with the given string. The following options are available:

        @code
        dots = RBA::Edges::new(iter, "*")           # all texts
        dots = RBA::Edges::new(iter, "A*")          # all texts starting with an 'A'
        dots = RBA::Edges::new(iter, "A*", false)   # all texts exactly matching 'A*'
        @/code

        This method has been introduced in version 0.26.
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator, trans: ICplxTrans, as_edges: Optional[bool] = ...) -> Edges:
        r"""
        @brief Constructor of a flat edge collection from a hierarchical shape set with a transformation

        This constructor creates an edge collection from the shapes delivered by the given recursive shape iterator.
        It feeds the shapes from a hierarchy of cells into a flat edge set.
        The transformation is useful to scale to a specific database unit for example.

        Text objects are not inserted, because they cannot be converted to edges.
        Edge objects are inserted as such. If "as_edges" is true, "solid" objects (boxes, polygons, paths) are converted to edges which form the hull of these objects. If "as_edges" is false, solid objects are ignored.

        @code
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        dbu    = 0.1 # the target database unit
        r = RBA::Edges::new(layout.begin_shapes(cell, layer), RBA::ICplxTrans::new(layout.dbu / dbu))
        @/code
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore, expr: str, as_pattern: Optional[bool] = ...) -> Edges:
        r"""
        @brief Constructor from a text set

        @param shape_iterator The iterator from which to derive the texts
        @param dss The \DeepShapeStore object that acts as a heap for hierarchical operations.
        @param expr The selection string
        @param as_pattern If true, the selection string is treated as a glob pattern. Otherwise the match is exact.

        This special constructor will create a deep edge set from the text objects delivered by the shape iterator. Each text object will give a degenerated edge (a dot) that represents the text origin.
        Texts can be selected by their strings - either through a glob pattern or by exact comparison with the given string. The following options are available:

        @code
        region = RBA::Region::new(iter, dss, "*")           # all texts
        region = RBA::Region::new(iter, dss, "A*")          # all texts starting with an 'A'
        region = RBA::Region::new(iter, dss, "A*", false)   # all texts exactly matching 'A*'
        @/code

        This method has been introduced in version 0.26.
        """
    @overload
    @classmethod
    def new(cls, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore, trans: ICplxTrans, as_edges: Optional[bool] = ...) -> Edges:
        r"""
        @brief Constructor of a hierarchical edge collection with a transformation

        This constructor creates an edge collection from the shapes delivered by the given recursive shape iterator.
        It feeds the shapes from a hierarchy of cells into the hierarchical edge set.
        The edges remain within their original hierarchy unless other operations require the edges to be moved in the hierarchy.
        The transformation is useful to scale to a specific database unit for example.

        Text objects are not inserted, because they cannot be converted to edges.
        Edge objects are inserted as such. If "as_edges" is true, "solid" objects (boxes, polygons, paths) are converted to edges which form the hull of these objects. If "as_edges" is false, solid objects are ignored.

        @code
        dss    = RBA::DeepShapeStore::new
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        dbu    = 0.1 # the target database unit
        r = RBA::Edges::new(layout.begin_shapes(cell, layer), dss, RBA::ICplxTrans::new(layout.dbu / dbu), false)
        @/code
        """
    def __add__(self, other: Edges) -> Edges:
        r"""
        @brief Returns the combined edge set of self and the other one

        @return The resulting edge set

        This operator adds the edges of the other edge set to self and returns a new combined edge set. This usually creates unmerged edge sets and edges may overlap. Use \merge if you want to ensure the result edge set is merged.
        """
    @overload
    def __and__(self, other: Edges) -> Edges:
        r"""
        @brief Returns the boolean AND between self and the other edge collection

        @return The result of the boolean AND operation

        The boolean AND operation will return all parts of the edges in this collection which are coincident with parts of the edges in the other collection.The result will be a merged edge collection.
        """
    @overload
    def __and__(self, other: Region) -> Edges:
        r"""
        @brief Returns the parts of the edges inside the given region

        @return The edges inside the given region

        This operation returns the parts of the edges which are inside the given region.
        Edges on the borders of the polygons are included in the edge set.
        As a side effect, the edges are made non-intersecting by introducing cut points where
        edges intersect.

        This method has been introduced in version 0.24.
        """
    def __copy__(self) -> Edges:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> Edges:
        r"""
        @brief Creates a copy of self
        """
    def __getitem__(self, n: int) -> Edge:
        r"""
        @brief Returns the nth edge of the collection

        This method returns nil if the index is out of range. It is available for flat edge collections only - i.e. those for which \has_valid_edges? is true. Use \flatten to explicitly flatten an edge collection.
        This method returns the raw edge (not merged edges, even if merged semantics is enabled).

        The \each iterator is the more general approach to access the edges.
        """
    def __iadd__(self, other: Edges) -> Edges:
        r"""
        @brief Adds the edges of the other edge collection to self

        @return The edge set after modification (self)

        This operator adds the edges of the other edge set to self. This usually creates unmerged edge sets and edges may overlap. Use \merge if you want to ensure the result edge set is merged.
        """
    @overload
    def __iand__(self, other: Edges) -> Edges:
        r"""
        @brief Performs the boolean AND between self and the other edge collection

        @return The edge collection after modification (self)

        The boolean AND operation will return all parts of the edges in this collection which are coincident with parts of the edges in the other collection.The result will be a merged edge collection.
        """
    @overload
    def __iand__(self, other: Region) -> Edges:
        r"""
        @brief Selects the parts of the edges inside the given region

        @return The edge collection after modification (self)

        This operation selects the parts of the edges which are inside the given region.
        Edges on the borders of the polygons are included in the edge set.
        As a side effect, the edges are made non-intersecting by introducing cut points where
        edges intersect.

        This method has been introduced in version 0.24.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor

        This constructor creates an empty edge collection.
        """
    @overload
    def __init__(self, array: Sequence[Edge]) -> None:
        r"""
        @brief Constructor from an edge array

        This constructor creates an edge collection from an array of edges.
        """
    @overload
    def __init__(self, array: Sequence[Polygon]) -> None:
        r"""
        @brief Constructor from a polygon array

        This constructor creates an edge collection from an array of polygons.
        The edges form the contours of the polygons.
        """
    @overload
    def __init__(self, box: Box) -> None:
        r"""
        @brief Box constructor

        This constructor creates an edge collection from a box.
        The edges form the contour of the box.
        """
    @overload
    def __init__(self, edge: Edge) -> None:
        r"""
        @brief Constructor from a single edge

        This constructor creates an edge collection with a single edge.
        """
    @overload
    def __init__(self, path: Path) -> None:
        r"""
        @brief Path constructor

        This constructor creates an edge collection from a path.
        The edges form the contour of the path.
        """
    @overload
    def __init__(self, polygon: Polygon) -> None:
        r"""
        @brief Polygon constructor

        This constructor creates an edge collection from a polygon.
        The edges form the contour of the polygon.
        """
    @overload
    def __init__(self, polygon: SimplePolygon) -> None:
        r"""
        @brief Simple polygon constructor

        This constructor creates an edge collection from a simple polygon.
        The edges form the contour of the polygon.
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator, as_edges: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor of a flat edge collection from a hierarchical shape set

        This constructor creates an edge collection from the shapes delivered by the given recursive shape iterator.
        It feeds the shapes from a hierarchy of cells into a flat edge set.

        Text objects are not inserted, because they cannot be converted to edges.
        Edge objects are inserted as such. If "as_edges" is true, "solid" objects (boxes, polygons, paths) are converted to edges which form the hull of these objects. If "as_edges" is false, solid objects are ignored.

        @code
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        r = RBA::Edges::new(layout.begin_shapes(cell, layer), false)
        @/code
        """
    @overload
    def __init__(self, shapes: Shapes, as_edges: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor of a flat edge collection from a \Shapes container

        If 'as_edges' is true, the shapes from the container will be converted to edges (i.e. polygon contours to edges). Otherwise, only edges will be taken from the container.

        This method has been introduced in version 0.26.
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore, as_edges: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor of a hierarchical edge collection

        This constructor creates an edge collection from the shapes delivered by the given recursive shape iterator.
        It feeds the shapes from a hierarchy of cells into the hierarchical edge set.
        The edges remain within their original hierarchy unless other operations require the edges to be moved in the hierarchy.

        Text objects are not inserted, because they cannot be converted to edges.
        Edge objects are inserted as such. If "as_edges" is true, "solid" objects (boxes, polygons, paths) are converted to edges which form the hull of these objects. If "as_edges" is false, solid objects are ignored.

        @code
        dss    = RBA::DeepShapeStore::new
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        r = RBA::Edges::new(layout.begin_shapes(cell, layer), dss, false)
        @/code
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator, expr: str, as_pattern: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor from a text set

        @param shape_iterator The iterator from which to derive the texts
        @param expr The selection string
        @param as_pattern If true, the selection string is treated as a glob pattern. Otherwise the match is exact.

        This special constructor will create dot-like edges from the text objects delivered by the shape iterator. Each text object will give a degenerated edge (a dot) that represents the text origin.
        Texts can be selected by their strings - either through a glob pattern or by exact comparison with the given string. The following options are available:

        @code
        dots = RBA::Edges::new(iter, "*")           # all texts
        dots = RBA::Edges::new(iter, "A*")          # all texts starting with an 'A'
        dots = RBA::Edges::new(iter, "A*", false)   # all texts exactly matching 'A*'
        @/code

        This method has been introduced in version 0.26.
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator, trans: ICplxTrans, as_edges: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor of a flat edge collection from a hierarchical shape set with a transformation

        This constructor creates an edge collection from the shapes delivered by the given recursive shape iterator.
        It feeds the shapes from a hierarchy of cells into a flat edge set.
        The transformation is useful to scale to a specific database unit for example.

        Text objects are not inserted, because they cannot be converted to edges.
        Edge objects are inserted as such. If "as_edges" is true, "solid" objects (boxes, polygons, paths) are converted to edges which form the hull of these objects. If "as_edges" is false, solid objects are ignored.

        @code
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        dbu    = 0.1 # the target database unit
        r = RBA::Edges::new(layout.begin_shapes(cell, layer), RBA::ICplxTrans::new(layout.dbu / dbu))
        @/code
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore, expr: str, as_pattern: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor from a text set

        @param shape_iterator The iterator from which to derive the texts
        @param dss The \DeepShapeStore object that acts as a heap for hierarchical operations.
        @param expr The selection string
        @param as_pattern If true, the selection string is treated as a glob pattern. Otherwise the match is exact.

        This special constructor will create a deep edge set from the text objects delivered by the shape iterator. Each text object will give a degenerated edge (a dot) that represents the text origin.
        Texts can be selected by their strings - either through a glob pattern or by exact comparison with the given string. The following options are available:

        @code
        region = RBA::Region::new(iter, dss, "*")           # all texts
        region = RBA::Region::new(iter, dss, "A*")          # all texts starting with an 'A'
        region = RBA::Region::new(iter, dss, "A*", false)   # all texts exactly matching 'A*'
        @/code

        This method has been introduced in version 0.26.
        """
    @overload
    def __init__(self, shape_iterator: RecursiveShapeIterator, dss: DeepShapeStore, trans: ICplxTrans, as_edges: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor of a hierarchical edge collection with a transformation

        This constructor creates an edge collection from the shapes delivered by the given recursive shape iterator.
        It feeds the shapes from a hierarchy of cells into the hierarchical edge set.
        The edges remain within their original hierarchy unless other operations require the edges to be moved in the hierarchy.
        The transformation is useful to scale to a specific database unit for example.

        Text objects are not inserted, because they cannot be converted to edges.
        Edge objects are inserted as such. If "as_edges" is true, "solid" objects (boxes, polygons, paths) are converted to edges which form the hull of these objects. If "as_edges" is false, solid objects are ignored.

        @code
        dss    = RBA::DeepShapeStore::new
        layout = ... # a layout
        cell   = ... # the index of the initial cell
        layer  = ... # the index of the layer from where to take the shapes from
        dbu    = 0.1 # the target database unit
        r = RBA::Edges::new(layout.begin_shapes(cell, layer), dss, RBA::ICplxTrans::new(layout.dbu / dbu), false)
        @/code
        """
    def __ior__(self, other: Edges) -> Edges:
        r"""
        @brief Performs the boolean OR between self and the other edge set

        @return The edge collection after modification (self)

        The boolean OR is implemented by merging the edges of both edge sets. To simply join the edge collections without merging, the + operator is more efficient.
        """
    @overload
    def __isub__(self, other: Edges) -> Edges:
        r"""
        @brief Performs the boolean NOT between self and the other edge collection

        @return The edge collection after modification (self)

        The boolean NOT operation will return all parts of the edges in this collection which are not coincident with parts of the edges in the other collection.The result will be a merged edge collection.
        """
    @overload
    def __isub__(self, other: Region) -> Edges:
        r"""
        @brief Selects the parts of the edges outside the given region

        @return The edge collection after modification (self)

        This operation selects the parts of the edges which are outside the given region.
        Edges on the borders of the polygons are not included in the edge set.
        As a side effect, the edges are made non-intersecting by introducing cut points where
        edges intersect.

        This method has been introduced in version 0.24.
        """
    def __iter__(self) -> Iterator[Edge]:
        r"""
        @brief Returns each edge of the region
        """
    def __ixor__(self, other: Edges) -> Edges:
        r"""
        @brief Performs the boolean XOR between self and the other edge collection

        @return The edge collection after modification (self)

        The boolean XOR operation will return all parts of the edges in this and the other collection except the parts where both are coincident.
        The result will be a merged edge collection.
        """
    def __len__(self) -> int:
        r"""
        @brief Returns the (flat) number of edges in the edge collection

        This returns the number of raw edges (not merged edges if merged semantics is enabled).
        The count is computed 'as if flat', i.e. edges inside a cell are multiplied by the number of times a cell is instantiated.

        Starting with version 0.27, the method is called 'count' for consistency with \Region. 'size' is still provided as an alias.
        """
    def __or__(self, other: Edges) -> Edges:
        r"""
        @brief Returns the boolean OR between self and the other edge set

        @return The resulting edge collection

        The boolean OR is implemented by merging the edges of both edge sets. To simply join the edge collections without merging, the + operator is more efficient.
        """
    def __str__(self) -> str:
        r"""
        @brief Converts the edge collection to a string
        The length of the output is limited to 20 edges to avoid giant strings on large regions. For full output use "to_s" with a maximum count parameter.
        """
    @overload
    def __sub__(self, other: Edges) -> Edges:
        r"""
        @brief Returns the boolean NOT between self and the other edge collection

        @return The result of the boolean NOT operation

        The boolean NOT operation will return all parts of the edges in this collection which are not coincident with parts of the edges in the other collection.The result will be a merged edge collection.
        """
    @overload
    def __sub__(self, other: Region) -> Edges:
        r"""
        @brief Returns the parts of the edges outside the given region

        @return The edges outside the given region

        This operation returns the parts of the edges which are outside the given region.
        Edges on the borders of the polygons are not included in the edge set.
        As a side effect, the edges are made non-intersecting by introducing cut points where
        edges intersect.

        This method has been introduced in version 0.24.
        """
    def __xor__(self, other: Edges) -> Edges:
        r"""
        @brief Returns the boolean XOR between self and the other edge collection

        @return The result of the boolean XOR operation

        The boolean XOR operation will return all parts of the edges in this and the other collection except the parts where both are coincident.
        The result will be a merged edge collection.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    @overload
    def andnot(self, other: Edges) -> List[Edges]:
        r"""
        @brief Returns the boolean AND and NOT between self and the other edge set

        @return A two-element array of edge collections with the first one being the AND result and the second one being the NOT result

        This method will compute the boolean AND and NOT between two edge sets simultaneously. Because this requires a single sweep only, using this method is faster than doing AND and NOT separately.

        This method has been added in version 0.28.
        """
    @overload
    def andnot(self, other: Region) -> List[Edges]:
        r"""
        @brief Returns the boolean AND and NOT between self and the region

        @return A two-element array of edge collections with the first one being the AND result and the second one being the NOT result

        This method will compute the boolean AND and NOT simultaneously. Because this requires a single sweep only, using this method is faster than doing AND and NOT separately.

        This method has been added in version 0.28.
        """
    def assign(self, other: ShapeCollection) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> Box:
        r"""
        @brief Returns the bounding box of the edge collection
        The bounding box is the box enclosing all points of all edges.
        """
    def centers(self, length: int, fraction: float) -> Edges:
        r"""
        @brief Returns edges representing the center part of the edges
        @return A new collection of edges representing the part around the center
        This method allows one to specify the length of these segments in a twofold way: either as a fixed length or by specifying a fraction of the original length:

        @code
        edges = ...  # An edge collection
        edges.centers(100, 0.0)     # All segments have a length of 100 DBU
        edges.centers(0, 50.0)      # All segments have a length of half the original length
        edges.centers(100, 50.0)    # All segments have a length of half the original length
                                    # or 100 DBU, whichever is larger
        @/code

        It is possible to specify 0 for both values. In this case, degenerated edges (points) are delivered which specify the centers of the edges but can't participate in some functions.
        """
    def clear(self) -> None:
        r"""
        @brief Clears the edge collection
        """
    def count(self) -> int:
        r"""
        @brief Returns the (flat) number of edges in the edge collection

        This returns the number of raw edges (not merged edges if merged semantics is enabled).
        The count is computed 'as if flat', i.e. edges inside a cell are multiplied by the number of times a cell is instantiated.

        Starting with version 0.27, the method is called 'count' for consistency with \Region. 'size' is still provided as an alias.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def data_id(self) -> int:
        r"""
        @brief Returns the data ID (a unique identifier for the underlying data storage)

        This method has been added in version 0.26.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def disable_progress(self) -> None:
        r"""
        @brief Disable progress reporting
        Calling this method will disable progress reporting. See \enable_progress.
        """
    def dup(self) -> Edges:
        r"""
        @brief Creates a copy of self
        """
    def each(self) -> Iterator[Edge]:
        r"""
        @brief Returns each edge of the region
        """
    def each_merged(self) -> Iterator[Edge]:
        r"""
        @brief Returns each edge of the region

        In contrast to \each, this method delivers merged edges if merge semantics applies while \each delivers the original edges only.

        This method has been introduced in version 0.25.
        """
    def enable_progress(self, label: str) -> None:
        r"""
        @brief Enable progress reporting
        After calling this method, the edge collection will report the progress through a progress bar while expensive operations are running.
        The label is a text which is put in front of the progress bar.
        Using a progress bar will imply a performance penalty of a few percent typically.
        """
    def enclosed_check(self, other: Edges, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Region.Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ...) -> EdgePairs:
        r"""
        @brief Performs an inside check with options
        @param d The minimum distance for which the edges are checked
        @param other The other edge collection against which to check
        @param whole_edges If true, deliver the whole edges
        @param metrics Specify the metrics type
        @param ignore_angle The threshold angle above which no check is performed
        @param min_projection The lower threshold of the projected length of one edge onto another
        @param max_projection The upper threshold of the projected length of one edge onto another

        If "whole_edges" is true, the resulting \EdgePairs collection will receive the whole edges which contribute in the width check.

        "metrics" can be one of the constants \Euclidian, \Square or \Projection. See there for a description of these constants.
        Use nil for this value to select the default (Euclidian metrics).

        "ignore_angle" specifies the angle threshold of two edges. If two edges form an angle equal or above the given value, they will not contribute in the check. Setting this value to 90 (the default) will exclude edges with an angle of 90 degree or more from the check.
        Use nil for this value to select the default.

        "min_projection" and "max_projection" allow selecting edges by their projected value upon each other. It is sufficient if the projection of one edge on the other matches the specified condition. The projected length must be larger or equal to "min_projection" and less than "max_projection". If you don't want to specify one threshold, pass nil to the respective value.

        The 'enclosed_check' alias was introduced in version 0.27.5.
        """
    def enclosing_check(self, other: Edges, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Region.Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ...) -> EdgePairs:
        r"""
        @brief Performs an enclosing check with options
        @param d The minimum distance for which the edges are checked
        @param other The other edge collection against which to check
        @param whole_edges If true, deliver the whole edges
        @param metrics Specify the metrics type
        @param ignore_angle The threshold angle above which no check is performed
        @param min_projection The lower threshold of the projected length of one edge onto another
        @param max_projection The upper threshold of the projected length of one edge onto another

        If "whole_edges" is true, the resulting \EdgePairs collection will receive the whole edges which contribute in the width check.

        "metrics" can be one of the constants \Euclidian, \Square or \Projection. See there for a description of these constants.
        Use nil for this value to select the default (Euclidian metrics).

        "ignore_angle" specifies the angle threshold of two edges. If two edges form an angle equal or above the given value, they will not contribute in the check. Setting this value to 90 (the default) will exclude edges with an angle of 90 degree or more from the check.
        Use nil for this value to select the default.

        "min_projection" and "max_projection" allow selecting edges by their projected value upon each other. It is sufficient if the projection of one edge on the other matches the specified condition. The projected length must be larger or equal to "min_projection" and less than "max_projection". If you don't want to specify one threshold, pass nil to the respective value.
        """
    def end_segments(self, length: int, fraction: float) -> Edges:
        r"""
        @brief Returns edges representing a part of the edge before the end point
        @return A new collection of edges representing the end part
        This method allows one to specify the length of these segments in a twofold way: either as a fixed length or by specifying a fraction of the original length:

        @code
        edges = ...  # An edge collection
        edges.end_segments(100, 0.0)     # All segments have a length of 100 DBU
        edges.end_segments(0, 50.0)      # All segments have a length of half the original length
        edges.end_segments(100, 50.0)    # All segments have a length of half the original length
                                          # or 100 DBU, whichever is larger
        @/code

        It is possible to specify 0 for both values. In this case, degenerated edges (points) are delivered which specify the end positions of the edges but can't participate in some functions.
        """
    def extended(self, b: int, e: int, o: int, i: int, join: bool) -> Region:
        r"""
        @brief Returns a region with shapes representing the edges with the specified extensions
        @param b the parallel extension at the start point of the edge
        @param e the parallel extension at the end point of the edge
        @param o the perpendicular extension to the "outside" (left side as seen in the direction of the edge)
        @param i the perpendicular extension to the "inside" (right side as seen in the direction of the edge)
        @param join If true, connected edges are joined before the extension is applied
        @return A region containing the polygons representing these extended edges
        This is a generic version of \extended_in and \extended_out. It allows one to specify extensions for all four directions of an edge and to join the edges before the extension is applied.

        For degenerated edges forming a point, a rectangle with the b, e, o and i used as left, right, top and bottom distance to the center point of this edge is created.

        If join is true and edges form a closed loop, the b and e parameters are ignored and a rim polygon is created that forms the loop with the outside and inside extension given by o and i.
        """
    def extended_in(self, e: int) -> Region:
        r"""
        @brief Returns a region with shapes representing the edges with the given width
        @param e The extension width
        @return A region containing the polygons representing these extended edges
        The edges are extended to the "inside" by the given distance "e". The distance will be applied to the right side as seen in the direction of the edge. By definition, this is the side pointing to the inside of the polygon if the edge was derived from a polygon.

        Other versions of this feature are \extended_out and \extended.
        """
    def extended_out(self, e: int) -> Region:
        r"""
        @brief Returns a region with shapes representing the edges with the given width
        @param e The extension width
        @return A region containing the polygons representing these extended edges
        The edges are extended to the "outside" by the given distance "e". The distance will be applied to the left side as seen in the direction of the edge. By definition, this is the side pointing to the outside of the polygon if the edge was derived from a polygon.

        Other versions of this feature are \extended_in and \extended.
        """
    @overload
    def extents(self) -> Region:
        r"""
        @brief Returns a region with the bounding boxes of the edges
        This method will return a region consisting of the bounding boxes of the edges.
        The boxes will not be merged, so it is possible to determine overlaps of these boxes for example.
        """
    @overload
    def extents(self, d: int) -> Region:
        r"""
        @brief Returns a region with the enlarged bounding boxes of the edges
        This method will return a region consisting of the bounding boxes of the edges enlarged by the given distance d.
        The enlargement is specified per edge, i.e the width and height will be increased by 2*d.
        The boxes will not be merged, so it is possible to determine overlaps of these boxes for example.
        """
    @overload
    def extents(self, dx: int, dy: int) -> Region:
        r"""
        @brief Returns a region with the enlarged bounding boxes of the edges
        This method will return a region consisting of the bounding boxes of the edges enlarged by the given distance dx in x direction and dy in y direction.
        The enlargement is specified per edge, i.e the width will be increased by 2*dx.
        The boxes will not be merged, so it is possible to determine overlaps of these boxes for example.
        """
    def flatten(self) -> None:
        r"""
        @brief Explicitly flattens an edge collection

        If the collection is already flat (i.e. \has_valid_edges? returns true), this method will not change it.

        This method has been introduced in version 0.26.
        """
    def has_valid_edges(self) -> bool:
        r"""
        @brief Returns true if the edge collection is flat and individual edges can be accessed randomly

        This method has been introduced in version 0.26.
        """
    def hier_count(self) -> int:
        r"""
        @brief Returns the (hierarchical) number of edges in the edge collection

        This returns the number of raw edges (not merged edges if merged semantics is enabled).
        The count is computed 'hierarchical', i.e. edges inside a cell are counted once even if the cell is instantiated multiple times.

        This method has been introduced in version 0.27.
        """
    def in_(self, other: Edges) -> Edges:
        r"""
        @brief Returns all edges which are members of the other edge collection
        This method returns all edges in self which can be found in the other edge collection as well with exactly the same geometry.
        """
    def in_and_out(self, other: Edges) -> List[Edges]:
        r"""
        @brief Returns all polygons which are members and not members of the other region
        This method is equivalent to calling \members_of and \not_members_of, but delivers both results at the same time and is more efficient than two separate calls. The first element returned is the \members_of part, the second is the \not_members_of part.

        This method has been introduced in version 0.28.
        """
    @overload
    def insert(self, box: Box) -> None:
        r"""
        @brief Inserts a box

        Inserts the edges that form the contour of the box into the edge collection.
        """
    @overload
    def insert(self, edge: Edge) -> None:
        r"""
        @brief Inserts an edge

        Inserts the edge into the edge collection.
        """
    @overload
    def insert(self, edges: Edges) -> None:
        r"""
        @brief Inserts all edges from the other edge collection into this one
        This method has been introduced in version 0.25.
        """
    @overload
    def insert(self, edges: Sequence[Edge]) -> None:
        r"""
        @brief Inserts all edges from the array into this edge collection
        """
    @overload
    def insert(self, path: Path) -> None:
        r"""
        @brief Inserts a path

        Inserts the edges that form the contour of the path into the edge collection.
        """
    @overload
    def insert(self, polygon: Polygon) -> None:
        r"""
        @brief Inserts a polygon

        Inserts the edges that form the contour of the polygon into the edge collection.
        """
    @overload
    def insert(self, polygon: SimplePolygon) -> None:
        r"""
        @brief Inserts a simple polygon

        Inserts the edges that form the contour of the simple polygon into the edge collection.
        """
    @overload
    def insert(self, polygons: Sequence[Polygon]) -> None:
        r"""
        @brief Inserts all polygons from the array into this edge collection
        """
    @overload
    def insert(self, region: Region) -> None:
        r"""
        @brief Inserts a region
        Inserts the edges that form the contours of the polygons from the region into the edge collection.

        This method has been introduced in version 0.25.
        """
    @overload
    def insert(self, shape_iterator: RecursiveShapeIterator) -> None:
        r"""
        @brief Inserts all shapes delivered by the recursive shape iterator into this edge collection

        For "solid" shapes (boxes, polygons, paths), this method inserts the edges that form the contour of the shape into the edge collection.
        Edge shapes are inserted as such.
        Text objects are not inserted, because they cannot be converted to polygons.
        """
    @overload
    def insert(self, shapes: Shapes) -> None:
        r"""
        @brief Inserts all edges from the shape collection into this edge collection
        This method takes each edge from the shape collection and inserts it into the region. "Polygon-like" objects are inserted as edges forming the contours of the polygons.
        Text objects are ignored.

        This method has been introduced in version 0.25.
        """
    @overload
    def insert(self, shape_iterator: RecursiveShapeIterator, trans: ICplxTrans) -> None:
        r"""
        @brief Inserts all shapes delivered by the recursive shape iterator into this edge collection with a transformation

        For "solid" shapes (boxes, polygons, paths), this method inserts the edges that form the contour of the shape into the edge collection.
        Edge shapes are inserted as such.
        Text objects are not inserted, because they cannot be converted to polygons.
        This variant will apply the given transformation to the shapes. This is useful to scale the shapes to a specific database unit for example.
        """
    @overload
    def insert(self, shapes: Shapes, trans: ICplxTrans) -> None:
        r"""
        @brief Inserts all edges from the shape collection into this edge collection with complex transformation
        This method acts as the version without transformation, but will apply the given complex transformation before inserting the edges.

        This method has been introduced in version 0.25.
        """
    @overload
    def insert(self, shapes: Shapes, trans: Trans) -> None:
        r"""
        @brief Inserts all edges from the shape collection into this edge collection (with transformation)
        This method acts as the version without transformation, but will apply the given transformation before inserting the edges.

        This method has been introduced in version 0.25.
        """
    def insert_into(self, layout: Layout, cell_index: int, layer: int) -> None:
        r"""
        @brief Inserts this edge collection into the given layout, below the given cell and into the given layer.
        If the edge collection is a hierarchical one, a suitable hierarchy will be built below the top cell or and existing hierarchy will be reused.

        This method has been introduced in version 0.26.
        """
    @overload
    def inside(self, other: Edges) -> Edges:
        r"""
        @brief Returns the edges of this edge collection which are inside (completely covered by) edges from the other edge collection

        @return A new edge collection containing the edges overlapping or touching edges from the other edge collection

        This method has been introduced in version 0.28.
        """
    @overload
    def inside(self, other: Region) -> Edges:
        r"""
        @brief Returns the edges from this edge collection which are inside (completely covered by) polygons from the region

        @return A new edge collection containing the edges overlapping or touching polygons from the region

        This method has been introduced in version 0.28.
        """
    def inside_check(self, other: Edges, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Region.Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ...) -> EdgePairs:
        r"""
        @brief Performs an inside check with options
        @param d The minimum distance for which the edges are checked
        @param other The other edge collection against which to check
        @param whole_edges If true, deliver the whole edges
        @param metrics Specify the metrics type
        @param ignore_angle The threshold angle above which no check is performed
        @param min_projection The lower threshold of the projected length of one edge onto another
        @param max_projection The upper threshold of the projected length of one edge onto another

        If "whole_edges" is true, the resulting \EdgePairs collection will receive the whole edges which contribute in the width check.

        "metrics" can be one of the constants \Euclidian, \Square or \Projection. See there for a description of these constants.
        Use nil for this value to select the default (Euclidian metrics).

        "ignore_angle" specifies the angle threshold of two edges. If two edges form an angle equal or above the given value, they will not contribute in the check. Setting this value to 90 (the default) will exclude edges with an angle of 90 degree or more from the check.
        Use nil for this value to select the default.

        "min_projection" and "max_projection" allow selecting edges by their projected value upon each other. It is sufficient if the projection of one edge on the other matches the specified condition. The projected length must be larger or equal to "min_projection" and less than "max_projection". If you don't want to specify one threshold, pass nil to the respective value.

        The 'enclosed_check' alias was introduced in version 0.27.5.
        """
    def inside_outside_part(self, other: Region) -> List[Edges]:
        r"""
        @brief Returns the partial edges inside and outside the given region

        @return A two-element array of edge collections with the first one being the \inside_part result and the second one being the \outside_part result

        This method will compute the results simultaneously. Because this requires a single sweep only, using this method is faster than doing \inside_part and \outside_part separately.

        This method has been added in version 0.28.
        """
    def inside_part(self, other: Region) -> Edges:
        r"""
        @brief Returns the parts of the edges of this edge collection which are inside the polygons of the region

        @return A new edge collection containing the edge parts inside the region

        This operation returns the parts of the edges which are inside the given region.
        This functionality is similar to the '&' operator, but edges on the borders of the polygons are not included in the edge set.
        As a side effect, the edges are made non-intersecting by introducing cut points where
        edges intersect.

        This method has been introduced in version 0.24.
        """
    @overload
    def interacting(self, other: Edges) -> Edges:
        r"""
        @brief Returns the edges of this edge collection which overlap or touch edges from the other edge collection

        @return A new edge collection containing the edges overlapping or touching edges from the other edge collection
        """
    @overload
    def interacting(self, other: Region) -> Edges:
        r"""
        @brief Returns the edges from this edge collection which overlap or touch polygons from the region

        @return A new edge collection containing the edges overlapping or touching polygons from the region
        """
    def intersections(self, other: Edges) -> Edges:
        r"""
        @brief Computes the intersections between this edges and other edges
        This computation is like an AND operation, but also including crossing points between non-coincident edges as degenerated (point-like) edges.

        This method has been introduced in version 0.26.2
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_deep(self) -> bool:
        r"""
        @brief Returns true if the edge collection is a deep (hierarchical) one

        This method has been added in version 0.26.
        """
    def is_empty(self) -> bool:
        r"""
        @brief Returns true if the edge collection is empty
        """
    def is_merged(self) -> bool:
        r"""
        @brief Returns true if the edge collection is merged
        If the region is merged, coincident edges have been merged into single edges. You can ensure merged state by calling \merge.
        """
    @overload
    def length(self) -> int:
        r"""
        @brief Returns the total length of all edges in the edge collection

        Merged semantics applies for this method (see \merged_semantics= of merged semantics)
        """
    @overload
    def length(self, rect: Box) -> int:
        r"""
        @brief Returns the total length of all edges in the edge collection (restricted to a rectangle)
        This version will compute the total length of all edges in the collection, restricting the computation to the given rectangle.
        Edges along the border are handled in a special way: they are counted when they are oriented with their inside side toward the rectangle (in other words: outside edges must coincide with the rectangle's border in order to be counted).

        Merged semantics applies for this method (see \merged_semantics= of merged semantics)
        """
    def members_of(self, other: Edges) -> Edges:
        r"""
        @brief Returns all edges which are members of the other edge collection
        This method returns all edges in self which can be found in the other edge collection as well with exactly the same geometry.
        """
    def merge(self) -> Edges:
        r"""
        @brief Merge the edges

        @return The edge collection after the edges have been merged (self).

        Merging joins parallel edges which overlap or touch.
        Crossing edges are not merged.
        If the edge collection is already merged, this method does nothing
        """
    def merged(self) -> Edges:
        r"""
        @brief Returns the merged edge collection

        @return The edge collection after the edges have been merged.

        Merging joins parallel edges which overlap or touch.
        Crossing edges are not merged.
        In contrast to \merge, this method does not modify the edge collection but returns a merged copy.
        """
    @overload
    def move(self, v: Vector) -> Edges:
        r"""
        @brief Moves the edge collection

        Moves the polygon by the given offset and returns the 
        moved edge collection. The edge collection is overwritten.

        @param v The distance to move the edge collection.

        @return The moved edge collection (self).

        Starting with version 0.25 the displacement type is a vector.
        """
    @overload
    def move(self, x: int, y: int) -> Edges:
        r"""
        @brief Moves the edge collection

        Moves the edge collection by the given offset and returns the 
        moved edge collection. The edge collection is overwritten.

        @param x The x distance to move the edge collection.
        @param y The y distance to move the edge collection.

        @return The moved edge collection (self).
        """
    @overload
    def moved(self, v: Vector) -> Edges:
        r"""
        @brief Returns the moved edge collection (does not modify self)

        Moves the edge collection by the given offset and returns the 
        moved edge collection. The edge collection is not modified.

        @param v The distance to move the edge collection.

        @return The moved edge collection.

        Starting with version 0.25 the displacement type is a vector.
        """
    @overload
    def moved(self, x: int, v: int) -> Edges:
        r"""
        @brief Returns the moved edge collection (does not modify self)

        Moves the edge collection by the given offset and returns the 
        moved edge collection. The edge collection is not modified.

        @param x The x distance to move the edge collection.
        @param y The y distance to move the edge collection.

        @return The moved edge collection.
        """
    def not_in(self, other: Edges) -> Edges:
        r"""
        @brief Returns all edges which are not members of the other edge collection
        This method returns all edges in self which can not be found in the other edge collection with exactly the same geometry.
        """
    @overload
    def not_inside(self, other: Edges) -> Edges:
        r"""
        @brief Returns the edges of this edge collection which are not inside (completely covered by) edges from the other edge collection

        @return A new edge collection containing the edges not overlapping or touching edges from the other edge collection

        This method has been introduced in version 0.28.
        """
    @overload
    def not_inside(self, other: Region) -> Edges:
        r"""
        @brief Returns the edges from this edge collection which are not inside (completely covered by) polygons from the region

        @return A new edge collection containing the edges not overlapping or touching polygons from the region

        This method has been introduced in version 0.28.
        """
    @overload
    def not_interacting(self, other: Edges) -> Edges:
        r"""
        @brief Returns the edges of this edge collection which do not overlap or touch edges from the other edge collection

        @return A new edge collection containing the edges not overlapping or touching edges from the other edge collection
        """
    @overload
    def not_interacting(self, other: Region) -> Edges:
        r"""
        @brief Returns the edges from this edge collection which do not overlap or touch polygons from the region

        @return A new edge collection containing the edges not overlapping or touching polygons from the region
        """
    def not_members_of(self, other: Edges) -> Edges:
        r"""
        @brief Returns all edges which are not members of the other edge collection
        This method returns all edges in self which can not be found in the other edge collection with exactly the same geometry.
        """
    @overload
    def not_outside(self, other: Edges) -> Edges:
        r"""
        @brief Returns the edges of this edge collection which are not outside (completely covered by) edges from the other edge collection

        @return A new edge collection containing the edges not overlapping or touching edges from the other edge collection

        This method has been introduced in version 0.28.
        """
    @overload
    def not_outside(self, other: Region) -> Edges:
        r"""
        @brief Returns the edges from this edge collection which are not outside (completely covered by) polygons from the region

        @return A new edge collection containing the edges not overlapping or touching polygons from the region

        This method has been introduced in version 0.28.
        """
    @overload
    def outside(self, other: Edges) -> Edges:
        r"""
        @brief Returns the edges of this edge collection which are outside (completely covered by) edges from the other edge collection

        @return A new edge collection containing the edges overlapping or touching edges from the other edge collection

        This method has been introduced in version 0.28.
        """
    @overload
    def outside(self, other: Region) -> Edges:
        r"""
        @brief Returns the edges from this edge collection which are outside (completely covered by) polygons from the region

        @return A new edge collection containing the edges overlapping or touching polygons from the region

        This method has been introduced in version 0.28.
        """
    def outside_part(self, other: Region) -> Edges:
        r"""
        @brief Returns the parts of the edges of this edge collection which are outside the polygons of the region

        @return A new edge collection containing the edge parts outside the region

        This operation returns the parts of the edges which are not inside the given region.
        This functionality is similar to the '-' operator, but edges on the borders of the polygons are included in the edge set.
        As a side effect, the edges are made non-intersecting by introducing cut points where
        edges intersect.

        This method has been introduced in version 0.24.
        """
    def overlap_check(self, other: Edges, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Region.Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ...) -> EdgePairs:
        r"""
        @brief Performs an overlap check with options
        @param d The minimum distance for which the edges are checked
        @param other The other edge collection against which to check
        @param whole_edges If true, deliver the whole edges
        @param metrics Specify the metrics type
        @param ignore_angle The threshold angle above which no check is performed
        @param min_projection The lower threshold of the projected length of one edge onto another
        @param max_projection The upper threshold of the projected length of one edge onto another

        If "whole_edges" is true, the resulting \EdgePairs collection will receive the whole edges which contribute in the width check.

        "metrics" can be one of the constants \Euclidian, \Square or \Projection. See there for a description of these constants.
        Use nil for this value to select the default (Euclidian metrics).

        "ignore_angle" specifies the angle threshold of two edges. If two edges form an angle equal or above the given value, they will not contribute in the check. Setting this value to 90 (the default) will exclude edges with an angle of 90 degree or more from the check.
        Use nil for this value to select the default.

        "min_projection" and "max_projection" allow selecting edges by their projected value upon each other. It is sufficient if the projection of one edge on the other matches the specified condition. The projected length must be larger or equal to "min_projection" and less than "max_projection". If you don't want to specify one threshold, pass nil to the respective value.
        """
    @overload
    def pull_interacting(self, other: Edges) -> Edges:
        r"""
        @brief Returns all edges of "other" which are interacting with polygons of this edge set
        See the other \pull_interacting version for more details.

        @return The edge collection after the edges have been selected (from other)

        Merged semantics applies for this method (see \merged_semantics= of merged semantics)

        This method has been introduced in version 0.26.1
        """
    @overload
    def pull_interacting(self, other: Region) -> Region:
        r"""
        @brief Returns all polygons of "other" which are interacting with (overlapping, touching) edges of this edge set
        The "pull_..." methods are similar to "select_..." but work the opposite way: they select shapes from the argument region rather than self. In a deep (hierarchical) context the output region will be hierarchically aligned with self, so the "pull_..." methods provide a way for re-hierarchization.

        @return The region after the polygons have been selected (from other)

        Merged semantics applies for this method (see \merged_semantics= of merged semantics)

        This method has been introduced in version 0.26.1
        """
    @overload
    def select_inside(self, other: Edges) -> Edges:
        r"""
        @brief Selects the edges from this edge collection which are inside (completely covered by) edges from the other edge collection

        @return The edge collection after the edges have been selected (self)

        This method has been introduced in version 0.28.
        """
    @overload
    def select_inside(self, other: Region) -> Edges:
        r"""
        @brief Selects the edges from this edge collection which are inside (completely covered by) polygons from the region

        @return The edge collection after the edges have been selected (self)

        This method has been introduced in version 0.28.
        """
    def select_inside_part(self, other: Region) -> Edges:
        r"""
        @brief Selects the parts of the edges from this edge collection which are inside the polygons of the given region

        @return The edge collection after the edges have been selected (self)

        This operation selects the parts of the edges which are inside the given region.
        This functionality is similar to the '&=' operator, but edges on the borders of the polygons are not included in the edge set.
        As a side effect, the edges are made non-intersecting by introducing cut points where
        edges intersect.

        This method has been introduced in version 0.24.
        """
    @overload
    def select_interacting(self, other: Edges) -> Edges:
        r"""
        @brief Selects the edges from this edge collection which overlap or touch edges from the other edge collection

        @return The edge collection after the edges have been selected (self)
        """
    @overload
    def select_interacting(self, other: Region) -> Edges:
        r"""
        @brief Selects the edges from this edge collection which overlap or touch polygons from the region

        @return The edge collection after the edges have been selected (self)
        """
    @overload
    def select_not_inside(self, other: Edges) -> Edges:
        r"""
        @brief Selects the edges from this edge collection which are not inside (completely covered by) edges from the other edge collection

        @return The edge collection after the edges have been selected (self)

        This method has been introduced in version 0.28.
        """
    @overload
    def select_not_inside(self, other: Region) -> Edges:
        r"""
        @brief Selects the edges from this edge collection which are not inside (completely covered by) polygons from the region

        @return The edge collection after the edges have been selected (self)

        This method has been introduced in version 0.28.
        """
    @overload
    def select_not_interacting(self, other: Edges) -> Edges:
        r"""
        @brief Selects the edges from this edge collection which do not overlap or touch edges from the other edge collection

        @return The edge collection after the edges have been selected (self)
        """
    @overload
    def select_not_interacting(self, other: Region) -> Edges:
        r"""
        @brief Selects the edges from this edge collection which do not overlap or touch polygons from the region

        @return The edge collection after the edges have been selected (self)
        """
    @overload
    def select_not_outside(self, other: Edges) -> Edges:
        r"""
        @brief Selects the edges from this edge collection which are not outside (completely covered by) edges from the other edge collection

        @return The edge collection after the edges have been selected (self)

        This method has been introduced in version 0.28.
        """
    @overload
    def select_not_outside(self, other: Region) -> Edges:
        r"""
        @brief Selects the edges from this edge collection which are not outside (completely covered by) polygons from the region

        @return The edge collection after the edges have been selected (self)

        This method has been introduced in version 0.28.
        """
    @overload
    def select_outside(self, other: Edges) -> Edges:
        r"""
        @brief Selects the edges from this edge collection which are outside (completely covered by) edges from the other edge collection

        @return The edge collection after the edges have been selected (self)

        This method has been introduced in version 0.28.
        """
    @overload
    def select_outside(self, other: Region) -> Edges:
        r"""
        @brief Selects the edges from this edge collection which are outside (completely covered by) polygons from the region

        @return The edge collection after the edges have been selected (self)

        This method has been introduced in version 0.28.
        """
    def select_outside_part(self, other: Region) -> Edges:
        r"""
        @brief Selects the parts of the edges from this edge collection which are outside the polygons of the given region

        @return The edge collection after the edges have been selected (self)

        This operation selects the parts of the edges which are not inside the given region.
        This functionality is similar to the '-=' operator, but edges on the borders of the polygons are included in the edge set.
        As a side effect, the edges are made non-intersecting by introducing cut points where
        edges intersect.

        This method has been introduced in version 0.24.
        """
    def separation_check(self, other: Edges, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Region.Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ...) -> EdgePairs:
        r"""
        @brief Performs an overlap check with options
        @param d The minimum distance for which the edges are checked
        @param other The other edge collection against which to check
        @param whole_edges If true, deliver the whole edges
        @param metrics Specify the metrics type
        @param ignore_angle The threshold angle above which no check is performed
        @param min_projection The lower threshold of the projected length of one edge onto another
        @param max_projection The upper threshold of the projected length of one edge onto another

        If "whole_edges" is true, the resulting \EdgePairs collection will receive the whole edges which contribute in the width check.

        "metrics" can be one of the constants \Euclidian, \Square or \Projection. See there for a description of these constants.
        Use nil for this value to select the default (Euclidian metrics).

        "ignore_angle" specifies the angle threshold of two edges. If two edges form an angle equal or above the given value, they will not contribute in the check. Setting this value to 90 (the default) will exclude edges with an angle of 90 degree or more from the check.
        Use nil for this value to select the default.

        "min_projection" and "max_projection" allow selecting edges by their projected value upon each other. It is sufficient if the projection of one edge on the other matches the specified condition. The projected length must be larger or equal to "min_projection" and less than "max_projection". If you don't want to specify one threshold, pass nil to the respective value.
        """
    def size(self) -> int:
        r"""
        @brief Returns the (flat) number of edges in the edge collection

        This returns the number of raw edges (not merged edges if merged semantics is enabled).
        The count is computed 'as if flat', i.e. edges inside a cell are multiplied by the number of times a cell is instantiated.

        Starting with version 0.27, the method is called 'count' for consistency with \Region. 'size' is still provided as an alias.
        """
    def space_check(self, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Region.Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ...) -> EdgePairs:
        r"""
        @brief Performs a space check with options
        @param d The minimum distance for which the edges are checked
        @param whole_edges If true, deliver the whole edges
        @param metrics Specify the metrics type
        @param ignore_angle The threshold angle above which no check is performed
        @param min_projection The lower threshold of the projected length of one edge onto another
        @param max_projection The upper threshold of the projected length of one edge onto another

        If "whole_edges" is true, the resulting \EdgePairs collection will receive the whole edges which contribute in the space check.

        "metrics" can be one of the constants \Euclidian, \Square or \Projection. See there for a description of these constants.
        Use nil for this value to select the default (Euclidian metrics).

        "ignore_angle" specifies the angle threshold of two edges. If two edges form an angle equal or above the given value, they will not contribute in the check. Setting this value to 90 (the default) will exclude edges with an angle of 90 degree or more from the check.
        Use nil for this value to select the default.

        "min_projection" and "max_projection" allow selecting edges by their projected value upon each other. It is sufficient if the projection of one edge on the other matches the specified condition. The projected length must be larger or equal to "min_projection" and less than "max_projection". If you don't want to specify one threshold, pass nil to the respective value.
        """
    @overload
    def split_inside(self, other: Edges) -> List[Edges]:
        r"""
        @brief Selects the edges from this edge collection which are and are not inside (completely covered by) edges from the other collection

        @return A two-element list of edge collections (first: inside, second: non-inside)

        This method provides a faster way to compute both inside and non-inside edges compared to using separate methods. It has been introduced in version 0.28.
        """
    @overload
    def split_inside(self, other: Region) -> List[Edges]:
        r"""
        @brief Selects the edges from this edge collection which are and are not inside (completely covered by) polygons from the other region

        @return A two-element list of edge collections (first: inside, second: non-inside)

        This method provides a faster way to compute both inside and non-inside edges compared to using separate methods. It has been introduced in version 0.28.
        """
    @overload
    def split_interacting(self, other: Edges) -> List[Edges]:
        r"""
        @brief Selects the edges from this edge collection which do and do not interact with edges from the other collection

        @return A two-element list of edge collections (first: interacting, second: non-interacting)

        This method provides a faster way to compute both interacting and non-interacting edges compared to using separate methods. It has been introduced in version 0.28.
        """
    @overload
    def split_interacting(self, other: Region) -> List[Edges]:
        r"""
        @brief Selects the edges from this edge collection which do and do not interact with polygons from the other region

        @return A two-element list of edge collections (first: interacting, second: non-interacting)

        This method provides a faster way to compute both interacting and non-interacting edges compared to using separate methods. It has been introduced in version 0.28.
        """
    @overload
    def split_outside(self, other: Edges) -> List[Edges]:
        r"""
        @brief Selects the edges from this edge collection which are and are not outside (completely covered by) edges from the other collection

        @return A two-element list of edge collections (first: outside, second: non-outside)

        This method provides a faster way to compute both outside and non-outside edges compared to using separate methods. It has been introduced in version 0.28.
        """
    @overload
    def split_outside(self, other: Region) -> List[Edges]:
        r"""
        @brief Selects the edges from this edge collection which are and are not outside (completely covered by) polygons from the other region

        @return A two-element list of edge collections (first: outside, second: non-outside)

        This method provides a faster way to compute both outside and non-outside edges compared to using separate methods. It has been introduced in version 0.28.
        """
    def start_segments(self, length: int, fraction: float) -> Edges:
        r"""
        @brief Returns edges representing a part of the edge after the start point
        @return A new collection of edges representing the start part
        This method allows one to specify the length of these segments in a twofold way: either as a fixed length or by specifying a fraction of the original length:

        @code
        edges = ...  # An edge collection
        edges.start_segments(100, 0.0)    # All segments have a length of 100 DBU
        edges.start_segments(0, 50.0)     # All segments have a length of half the original length
        edges.start_segments(100, 50.0)   # All segments have a length of half the original length
                                          # or 100 DBU, whichever is larger
        @/code

        It is possible to specify 0 for both values. In this case, degenerated edges (points) are delivered which specify the start positions of the edges but can't participate in some functions.
        """
    def swap(self, other: Edges) -> None:
        r"""
        @brief Swap the contents of this edge collection with the contents of another one
        This method is useful to avoid excessive memory allocation in some cases. For managed memory languages such as Ruby, those cases will be rare. 
        """
    @overload
    def to_s(self) -> str:
        r"""
        @brief Converts the edge collection to a string
        The length of the output is limited to 20 edges to avoid giant strings on large regions. For full output use "to_s" with a maximum count parameter.
        """
    @overload
    def to_s(self, max_count: int) -> str:
        r"""
        @brief Converts the edge collection to a string
        This version allows specification of the maximum number of edges contained in the string.
        """
    @overload
    def transform(self, t: ICplxTrans) -> Edges:
        r"""
        @brief Transform the edge collection with a complex transformation (modifies self)

        Transforms the edge collection with the given transformation.
        This version modifies the edge collection and returns a reference to self.

        @param t The transformation to apply.

        @return The transformed edge collection.
        """
    @overload
    def transform(self, t: IMatrix2d) -> Edges:
        r"""
        @brief Transform the edge collection (modifies self)

        Transforms the edge collection with the given 2d matrix transformation.
        This version modifies the edge collection and returns a reference to self.

        @param t The transformation to apply.

        @return The transformed edge collection.

        This variant has been introduced in version 0.27.
        """
    @overload
    def transform(self, t: IMatrix3d) -> Edges:
        r"""
        @brief Transform the edge collection (modifies self)

        Transforms the edge collection with the given 3d matrix transformation.
        This version modifies the edge collection and returns a reference to self.

        @param t The transformation to apply.

        @return The transformed edge collection.

        This variant has been introduced in version 0.27.
        """
    @overload
    def transform(self, t: Trans) -> Edges:
        r"""
        @brief Transform the edge collection (modifies self)

        Transforms the edge collection with the given transformation.
        This version modifies the edge collection and returns a reference to self.

        @param t The transformation to apply.

        @return The transformed edge collection.
        """
    def transform_icplx(self, t: ICplxTrans) -> Edges:
        r"""
        @brief Transform the edge collection with a complex transformation (modifies self)

        Transforms the edge collection with the given transformation.
        This version modifies the edge collection and returns a reference to self.

        @param t The transformation to apply.

        @return The transformed edge collection.
        """
    @overload
    def transformed(self, t: ICplxTrans) -> Edges:
        r"""
        @brief Transform the edge collection with a complex transformation

        Transforms the edge collection with the given complex transformation.
        Does not modify the edge collection but returns the transformed edge collection.

        @param t The transformation to apply.

        @return The transformed edge collection.
        """
    @overload
    def transformed(self, t: IMatrix2d) -> Edges:
        r"""
        @brief Transform the edge collection

        Transforms the edge collection with the given 2d matrix transformation.
        Does not modify the edge collection but returns the transformed edge collection.

        @param t The transformation to apply.

        @return The transformed edge collection.

        This variant has been introduced in version 0.27.
        """
    @overload
    def transformed(self, t: IMatrix3d) -> Edges:
        r"""
        @brief Transform the edge collection

        Transforms the edge collection with the given 3d matrix transformation.
        Does not modify the edge collection but returns the transformed edge collection.

        @param t The transformation to apply.

        @return The transformed edge collection.

        This variant has been introduced in version 0.27.
        """
    @overload
    def transformed(self, t: Trans) -> Edges:
        r"""
        @brief Transform the edge collection

        Transforms the edge collection with the given transformation.
        Does not modify the edge collection but returns the transformed edge collection.

        @param t The transformation to apply.

        @return The transformed edge collection.
        """
    def transformed_icplx(self, t: ICplxTrans) -> Edges:
        r"""
        @brief Transform the edge collection with a complex transformation

        Transforms the edge collection with the given complex transformation.
        Does not modify the edge collection but returns the transformed edge collection.

        @param t The transformation to apply.

        @return The transformed edge collection.
        """
    def width_check(self, d: int, whole_edges: Optional[bool] = ..., metrics: Optional[Region.Metrics] = ..., ignore_angle: Optional[Any] = ..., min_projection: Optional[Any] = ..., max_projection: Optional[Any] = ...) -> EdgePairs:
        r"""
        @brief Performs a width check with options
        @param d The minimum width for which the edges are checked
        @param whole_edges If true, deliver the whole edges
        @param metrics Specify the metrics type
        @param ignore_angle The threshold angle above which no check is performed
        @param min_projection The lower threshold of the projected length of one edge onto another
        @param max_projection The upper threshold of the projected length of one edge onto another

        If "whole_edges" is true, the resulting \EdgePairs collection will receive the whole edges which contribute in the width check.

        "metrics" can be one of the constants \Euclidian, \Square or \Projection. See there for a description of these constants.
        Use nil for this value to select the default (Euclidian metrics).

        "ignore_angle" specifies the angle threshold of two edges. If two edges form an angle equal or above the given value, they will not contribute in the check. Setting this value to 90 (the default) will exclude edges with an angle of 90 degree or more from the check.
        Use nil for this value to select the default.

        "min_projection" and "max_projection" allow selecting edges by their projected value upon each other. It is sufficient if the projection of one edge on the other matches the specified condition. The projected length must be larger or equal to "min_projection" and less than "max_projection". If you don't want to specify one threshold, pass nil to the respective value.
        """
    @overload
    def with_angle(self, angle: float, inverse: bool) -> Edges:
        r"""
        @brief Filters the edges by orientation
        Filters the edges in the edge collection by orientation. If "inverse" is false, only edges which have the given angle to the x-axis are returned. If "inverse" is true, edges not having the given angle are returned.

        This will select horizontal edges:

        @code
        horizontal = edges.with_angle(0, false)
        @/code
        """
    @overload
    def with_angle(self, type: Edges.EdgeType, inverse: bool) -> Edges:
        r"""
        @brief Filters the edges by orientation type
        Filters the edges in the edge collection by orientation. If "inverse" is false, only edges which have an angle of the given type are returned. If "inverse" is true, edges which do not conform to this criterion are returned.

        This version allows specifying an edge type instead of an angle. Edge types include multiple distinct orientations and are specified using one of the \OrthoEdges, \DiagonalEdges or \OrthoDiagonalEdges types.

        This method has been added in version 0.28.
        """
    @overload
    def with_angle(self, min_angle: float, max_angle: float, inverse: bool, include_min_angle: Optional[bool] = ..., include_max_angle: Optional[bool] = ...) -> Edges:
        r"""
        @brief Filters the edges by orientation
        Filters the edges in the edge collection by orientation. If "inverse" is false, only edges which have an angle to the x-axis larger or equal to "min_angle" (depending on "include_min_angle") and equal or less than "max_angle" (depending on "include_max_angle") are returned. If "inverse" is true, edges which do not conform to this criterion are returned.

        With "include_min_angle" set to true (the default), the minimum angle is included in the criterion while with false, the minimum angle itself is not included. Same for "include_max_angle" where the default is false, meaning the maximum angle is not included in the range.

        The two "include.." arguments have been added in version 0.27.
        """
    @overload
    def with_length(self, length: int, inverse: bool) -> Edges:
        r"""
        @brief Filters the edges by length
        Filters the edges in the edge collection by length. If "inverse" is false, only edges which have the given length are returned. If "inverse" is true, edges not having the given length are returned.
        """
    @overload
    def with_length(self, min_length: Any, max_length: Any, inverse: bool) -> Edges:
        r"""
        @brief Filters the edges by length
        Filters the edges in the edge collection by length. If "inverse" is false, only edges which have a length larger or equal to "min_length" and less than "max_length" are returned. If "inverse" is true, edges not having a length less than "min_length" or larger or equal than "max_length" are returned.

        If you don't want to specify a lower or upper limit, pass nil to that parameter.
        """

class InstElement:
    r"""
    @brief An element in an instantiation path

    This objects are used to reference a single instance in a instantiation path. The object is composed of a \CellInstArray object (accessible through the \cell_inst accessor) that describes the basic instance, which may be an array. The particular instance within the array can be further retrieved using the \array_member_trans, \specific_trans or \specific_cplx_trans methods.
    """
    @overload
    @classmethod
    def new(cls) -> InstElement:
        r"""
        @brief Default constructor
        """
    @overload
    @classmethod
    def new(cls, inst: Instance) -> InstElement:
        r"""
        @brief Create an instance element from a single instance alone
        Starting with version 0.15, this method takes an \Instance object (an instance reference) as the argument.
        """
    @overload
    @classmethod
    def new(cls, inst: Instance, a_index: int, b_index: int) -> InstElement:
        r"""
        @brief Create an instance element from an array instance pointing into a certain array member
        Starting with version 0.15, this method takes an \Instance object (an instance reference) as the first argument.
        """
    @classmethod
    def new_i(cls, inst: Instance) -> InstElement:
        r"""
        @brief Create an instance element from a single instance alone
        Starting with version 0.15, this method takes an \Instance object (an instance reference) as the argument.
        """
    @classmethod
    def new_iab(cls, inst: Instance, a_index: int, b_index: int) -> InstElement:
        r"""
        @brief Create an instance element from an array instance pointing into a certain array member
        Starting with version 0.15, this method takes an \Instance object (an instance reference) as the first argument.
        """
    def __copy__(self) -> InstElement:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> InstElement:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, b: object) -> bool:
        r"""
        @brief Equality of two InstElement objects
        Note: this operator returns true if both instance elements refer to the same instance, not just identical ones.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor
        """
    @overload
    def __init__(self, inst: Instance) -> None:
        r"""
        @brief Create an instance element from a single instance alone
        Starting with version 0.15, this method takes an \Instance object (an instance reference) as the argument.
        """
    @overload
    def __init__(self, inst: Instance, a_index: int, b_index: int) -> None:
        r"""
        @brief Create an instance element from an array instance pointing into a certain array member
        Starting with version 0.15, this method takes an \Instance object (an instance reference) as the first argument.
        """
    def __lt__(self, b: InstElement) -> bool:
        r"""
        @brief Provides an order criterion for two InstElement objects
        Note: this operator is just provided to establish any order, not a particular one.
        """
    def __ne__(self, b: object) -> bool:
        r"""
        @brief Inequality of two InstElement objects
        See the comments on the == operator.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def array_member_trans(self) -> Trans:
        r"""
        @brief Returns the transformation for this array member

        The array member transformation is the one applicable in addition to the global transformation for the member selected from an array.
        If this instance is not an array instance, the specific transformation is a unit transformation without displacement.
        """
    def assign(self, other: InstElement) -> None:
        r"""
        @brief Assigns another object to self
        """
    def cell_inst(self) -> CellInstArray:
        r"""
        @brief Accessor to the cell instance (array).

        This method is equivalent to "self.inst.cell_inst" and provided for convenience.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> InstElement:
        r"""
        @brief Creates a copy of self
        """
    def ia(self) -> int:
        r"""
        @brief Returns the 'a' axis index for array instances
        For instance elements describing one member of an array, this attribute will deliver the a axis index addressed by this element. See \ib and \array_member_trans for further attributes applicable to array members.
        If the element is a plain instance and not an array member, this attribute is a negative value.

        This method has been introduced in version 0.25.
        """
    def ib(self) -> int:
        r"""
        @brief Returns the 'b' axis index for array instances
        For instance elements describing one member of an array, this attribute will deliver the a axis index addressed by this element. See \ia and \array_member_trans for further attributes applicable to array members.
        If the element is a plain instance and not an array member, this attribute is a negative value.

        This method has been introduced in version 0.25.
        """
    def inst(self) -> Instance:
        r"""
        @brief Gets the \Instance object held in this instance path element.

        This method has been added in version 0.24.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def prop_id(self) -> int:
        r"""
        @brief Accessor to the property attached to this instance.

        This method is equivalent to "self.inst.prop_id" and provided for convenience.
        """
    def specific_cplx_trans(self) -> ICplxTrans:
        r"""
        @brief Returns the specific complex transformation for this instance

        The specific transformation is the one applicable for the member selected from an array.
        This is the effective transformation applied for this array member. \array_member_trans gives the transformation applied additionally to the instances' global transformation (in other words, specific_cplx_trans = array_member_trans * cell_inst.cplx_trans).
        """
    def specific_trans(self) -> Trans:
        r"""
        @brief Returns the specific transformation for this instance

        The specific transformation is the one applicable for the member selected from an array.
        This is the effective transformation applied for this array member. \array_member_trans gives the transformation applied additionally to the instances' global transformation (in other words, specific_trans = array_member_trans * cell_inst.trans).
        This method delivers a simple transformation that does not include magnification components. To get these as well, use \specific_cplx_trans.
        """

class LayerMapping:
    r"""
    @brief A layer mapping (source to target layout)

    A layer mapping is an association of layers in two layouts forming pairs of layers, i.e. one layer corresponds to another layer in the other layout. The LayerMapping object describes the mapping of layers of a source layout A to a target layout B.

    A layer mapping can be set up manually or using the methods \create or \create_full.

    @code
    lm = RBA::LayerMapping::new
    # explicit:
    lm.map(2, 1)  # map layer index 2 of source to 1 of target
    lm.map(7, 3)  # map layer index 7 of source to 3 of target
    ...
    # or employing the specification identity:
    lm.create(target_layout, source_layout)
    # plus creating layers which don't exist in the target layout yet:
    new_layers = lm.create_full(target_layout, source_layout)
    @/code

    A layer might not be mapped to another layer which basically means that there is no corresponding layer.
    Such layers will be ignored in operations using the layer mapping. Use \create_full to ensure all layers
    of the source layout are mapped.

    LayerMapping objects play a role mainly in the hierarchical copy or move operations of \Layout. However, use is not restricted to these applications.

    This class has been introduced in version 0.23.
    """
    @classmethod
    def new(cls) -> LayerMapping:
        r"""
        @brief Creates a new object of this class
        """
    def __copy__(self) -> LayerMapping:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> LayerMapping:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: LayerMapping) -> None:
        r"""
        @brief Assigns another object to self
        """
    def clear(self) -> None:
        r"""
        @brief Clears the mapping.
        """
    def create(self, layout_a: Layout, layout_b: Layout) -> None:
        r"""
        @brief Initialize the layer mapping from two layouts

        @param layout_a The target layout
        @param layout_b The source layout

        The layer mapping is created by looking up each layer of layout_b in layout_a. All layers with matching specifications (\LayerInfo) are mapped. Layouts without a layer/datatype/name specification will not be mapped.
        \create_full is a version of this method which creates new layers in layout_a if no corresponding layer is found.
        """
    def create_full(self, layout_a: Layout, layout_b: Layout) -> List[int]:
        r"""
        @brief Initialize the layer mapping from two layouts

        @param layout_a The target layout
        @param layout_b The source layout
        @return A list of layers created

        The layer mapping is created by looking up each layer of layout_b in layout_a. All layers with matching specifications (\LayerInfo) are mapped. Layouts without a layer/datatype/name specification will not be mapped.
        Layers with a valid specification which are not found in layout_a are created there.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> LayerMapping:
        r"""
        @brief Creates a copy of self
        """
    def has_mapping(self, layer_index_b: int) -> bool:
        r"""
        @brief Determine if a layer in layout_b has a mapping to a layout_a layer.


        @param layer_index_b The index of the layer in layout_b whose mapping is requested.
        @return true, if the layer has a mapping
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def layer_mapping(self, layer_index_b: int) -> int:
        r"""
        @brief Determine layer mapping of a layout_b layer to the corresponding layout_a layer.


        @param layer_index_b The index of the layer in layout_b whose mapping is requested.
        @return The corresponding layer in layout_a.
        """
    def map(self, layer_index_b: int, layer_index_a: int) -> None:
        r"""
        @brief Explicitly specify a mapping.


        @param layer_index_b The index of the layer in layout B (the "source")
        @param layer_index_a The index of the layer in layout A (the "target")

        Beside using the mapping generator algorithms provided through \create and \create_full, it is possible to explicitly specify layer mappings using this method.
        """
    def table(self) -> Dict[int, int]:
        r"""
        @brief Returns the mapping table.

        The mapping table is a dictionary where the keys are source layout layer indexes and the values are the target layout layer indexes.

        This method has been introduced in version 0.25.
        """

class LayerInfo:
    r"""
    @brief A structure encapsulating the layer properties

    The layer properties describe how a layer is stored in a GDS2 or OASIS file for example. The \LayerInfo object represents the storage properties that are attached to a layer in the database.

    In general, a layer has either a layer and a datatype number (in GDS2), a name (for example in DXF or CIF) or both (in OASIS). In the latter case, the primary identification is through layer and datatype number and the name is some annotation attached to it. A \LayerInfo object which specifies just a name returns true on \is_named?.
    The \LayerInfo object can also specify an anonymous layer (use \LayerInfo#new without arguments). Such a layer will not be stored when saving the layout. They can be employed for temporary layers for example. Use \LayerInfo#anonymous? to test whether a layer does not have a specification.

    The \LayerInfo is used for example in \Layout#insert_layer to specify the properties of the new layer that will be created. The \is_equivalent? method compares two \LayerInfo objects using the layer and datatype numbers with a higher priority over the name.
    """
    datatype: int
    r"""
    Getter:
    @brief Gets the datatype

    Setter:
    @brief Set the datatype
    """
    layer: int
    r"""
    Getter:
    @brief Gets the layer number

    Setter:
    @brief Sets the layer number
    """
    name: str
    r"""
    Getter:
    @brief Gets the layer name

    Setter:
    @brief Set the layer name
    The name is set on OASIS input for example, if the layer has a name.
    """
    @classmethod
    def from_string(cls, s: str, as_target: Optional[bool] = ...) -> LayerInfo:
        r"""
        @brief Create a layer info object from a string
        @param The string
        @return The LayerInfo object

        If 'as_target' is true, relative specifications such as '*+1' for layer or datatype are permitted.

        This method will take strings as produced by \to_s and create a \LayerInfo object from them. The format is either "layer", "layer/datatype", "name" or "name (layer/datatype)".

        This method was added in version 0.23.
        The 'as_target' argument has been added in version 0.26.5.
        """
    @overload
    @classmethod
    def new(cls) -> LayerInfo:
        r"""
        @brief The default constructor.
        Creates a default \LayerInfo object.

        This method was added in version 0.18.
        """
    @overload
    @classmethod
    def new(cls, name: str) -> LayerInfo:
        r"""
        @brief The constructor for a named layer.
        Creates a \LayerInfo object representing a named layer.
        @param name The name

        This method was added in version 0.18.
        """
    @overload
    @classmethod
    def new(cls, layer: int, datatype: int) -> LayerInfo:
        r"""
        @brief The constructor for a layer/datatype pair.
        Creates a \LayerInfo object representing a layer and datatype.
        @param layer The layer number
        @param datatype The datatype number

        This method was added in version 0.18.
        """
    @overload
    @classmethod
    def new(cls, layer: int, datatype: int, name: str) -> LayerInfo:
        r"""
        @brief The constructor for a named layer with layer and datatype.
        Creates a \LayerInfo object representing a named layer with layer and datatype.
        @param layer The layer number
        @param datatype The datatype number
        @param name The name

        This method was added in version 0.18.
        """
    def __copy__(self) -> LayerInfo:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> LayerInfo:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, b: object) -> bool:
        r"""
        @brief Compares two layer info objects
        @return True, if both are equal

        This method was added in version 0.18.
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given layer info object. This method enables layer info objects as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief The default constructor.
        Creates a default \LayerInfo object.

        This method was added in version 0.18.
        """
    @overload
    def __init__(self, name: str) -> None:
        r"""
        @brief The constructor for a named layer.
        Creates a \LayerInfo object representing a named layer.
        @param name The name

        This method was added in version 0.18.
        """
    @overload
    def __init__(self, layer: int, datatype: int) -> None:
        r"""
        @brief The constructor for a layer/datatype pair.
        Creates a \LayerInfo object representing a layer and datatype.
        @param layer The layer number
        @param datatype The datatype number

        This method was added in version 0.18.
        """
    @overload
    def __init__(self, layer: int, datatype: int, name: str) -> None:
        r"""
        @brief The constructor for a named layer with layer and datatype.
        Creates a \LayerInfo object representing a named layer with layer and datatype.
        @param layer The layer number
        @param datatype The datatype number
        @param name The name

        This method was added in version 0.18.
        """
    def __ne__(self, b: object) -> bool:
        r"""
        @brief Compares two layer info objects
        @return True, if both are not equal

        This method was added in version 0.18.
        """
    def __str__(self, as_target: Optional[bool] = ...) -> str:
        r"""
        @brief Convert the layer info object to a string
        @return The string

        If 'as_target' is true, wildcard and relative specifications are formatted such such.

        This method was added in version 0.18.
        The 'as_target' argument has been added in version 0.26.5.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def anonymous(self) -> bool:
        r"""
        @brief Returns true, if the layer has no specification (i.e. is created by the default constructor).
        @return True, if the layer does not have any specification.

        This method was added in version 0.23.
        """
    def assign(self, other: LayerInfo) -> None:
        r"""
        @brief Assigns another object to self
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> LayerInfo:
        r"""
        @brief Creates a copy of self
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given layer info object. This method enables layer info objects as hash keys.

        This method has been introduced in version 0.25.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_equivalent(self, b: LayerInfo) -> bool:
        r"""
        @brief Equivalence of two layer info objects
        @return True, if both are equivalent

        First, layer and datatype are compared. The name is of second order and used only if no layer or datatype is given.
        This is basically a weak comparison that reflects the search preferences.

        This method was added in version 0.18.
        """
    def is_named(self) -> bool:
        r"""
        @brief Returns true, if the layer is purely specified by name.
        @return True, if no layer or datatype is given.

        This method was added in version 0.18.
        """
    def to_s(self, as_target: Optional[bool] = ...) -> str:
        r"""
        @brief Convert the layer info object to a string
        @return The string

        If 'as_target' is true, wildcard and relative specifications are formatted such such.

        This method was added in version 0.18.
        The 'as_target' argument has been added in version 0.26.5.
        """

class LayoutMetaInfo:
    r"""
    @brief A piece of layout meta information
    Layout meta information is basically additional data that can be attached to a layout. Layout readers may generate meta information and some writers will add layout information to the layout object. Some writers will also read meta information to determine certain attributes.

    Multiple layout meta information objects can be attached to one layout using \Layout#add_meta_info. Meta information is identified by a unique name and carries a string value plus an optional description string. The description string is for information only and is not evaluated by code.

    See also \Layout#each_meta_info and \Layout#meta_info_value and \Layout#remove_meta_info
    This class has been introduced in version 0.25.
    """
    description: str
    r"""
    Getter:
    @brief Gets the description of the layout meta info object

    Setter:
    @brief Sets the description of the layout meta info object
    """
    name: str
    r"""
    Getter:
    @brief Gets the name of the layout meta info object

    Setter:
    @brief Sets the name of the layout meta info object
    """
    value: str
    r"""
    Getter:
    @brief Gets the value of the layout meta info object

    Setter:
    @brief Sets the value of the layout meta info object
    """
    @classmethod
    def new(cls, name: str, value: str, description: Optional[str] = ...) -> LayoutMetaInfo:
        r"""
        @brief Creates a layout meta info object
        @param name The name
        @param value The value
        @param description An optional description text
        """
    def __copy__(self) -> LayoutMetaInfo:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> LayoutMetaInfo:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self, name: str, value: str, description: Optional[str] = ...) -> None:
        r"""
        @brief Creates a layout meta info object
        @param name The name
        @param value The value
        @param description An optional description text
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: LayoutMetaInfo) -> None:
        r"""
        @brief Assigns another object to self
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> LayoutMetaInfo:
        r"""
        @brief Creates a copy of self
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """

class Layout:
    r"""
    @brief The layout object

    This object represents a layout.
    The layout object contains the cell hierarchy and
    adds functionality for managing cell names and layer names.
    The cell hierarchy can be changed by adding cells and cell instances.
    Cell instances will virtually put the content of a cell into another cell. Many cell instances can be put into a cell thus forming repetitions of the cell content. This process can be repeated over multiple levels. In effect a cell graphs is created with parent cells and child cells. The graph must not be recursive, so there is at least one top cell, which does not have a parent cell. Multiple top cells can be present.

    \Layout is the very basic class of the layout database. It has a rich set of methods to manipulate and query the layout hierarchy, the geometrical objects, the meta information and other features of the layout database. For a discussion of the basic API and the related classes see @<a href="/programming/database_api.xml">The Database API@</a>.

    Usually layout objects have already been created by KLayout's application core. You can address such a layout via the \CellView object inside the \LayoutView class. For example:

    @code
    active_layout = RBA::CellView::active.layout
    puts "Top cell of current layout is #{active_layout.top_cell.name}"
    @/code

    However, a layout can also be used standalone:

    @code
    layout = RBA::Layout::new
    cell = layout.create_cell("TOP")
    layer = layout.layer(RBA::LayerInfo::new(1, 0))
    cell.shapes(layer).insert(RBA::Box::new(0, 0, 1000, 1000))
    layout.write("single_rect.gds")
    @/code
    """
    dbu: float
    r"""
    Getter:
    @brief Gets the database unit

    The database unit is the value of one units distance in micrometers.
    For numerical reasons and to be compliant with the GDS2 format, the database objects use integer coordinates. The basic unit of these coordinates is the database unit.
    You can convert coordinates to micrometers by multiplying the integer value with the database unit.
    Typical values for the database unit are 0.001 micrometer (one nanometer).

    Setter:
    @brief Sets the database unit

    See \dbu for a description of the database unit.
    """
    prop_id: int
    r"""
    Getter:
    @brief Gets the properties ID associated with the layout

    This method has been introduced in version 0.24.
    Setter:
    @brief Sets the properties ID associated with the layout
    This method is provided, if a properties ID has been derived already. Usually it's more convenient to use \delete_property, \set_property or \property.

    This method has been introduced in version 0.24.
    """
    technology_name: str
    r"""
    Getter:
    @brief Gets the name of the technology this layout is associated with
    This method has been introduced in version 0.27. Before that, the technology has been kept in the 'technology' meta data element.
    Setter:
    @brief Sets the name of the technology this layout is associated with
    Changing the technology name will re-assess all library references because libraries can be technology specified. Cell layouts may be substituted during this re-assessment.

    This method has been introduced in version 0.27.
    """
    @overload
    @classmethod
    def new(cls) -> Layout:
        r"""
        @brief Creates a layout object

        Starting with version 0.25, layouts created with the default constructor are always editable. Before that version, they inherited the editable flag from the application.
        """
    @overload
    @classmethod
    def new(cls, editable: bool) -> Layout:
        r"""
        @brief Creates a layout object

        This constructor specifies whether the layout is editable. In editable mode, some optimizations are disabled and the layout can be manipulated through a variety of methods.

        This method was introduced in version 0.22.
        """
    @overload
    @classmethod
    def new(cls, manager: Manager) -> Layout:
        r"""
        @brief Creates a layout object attached to a manager

        This constructor specifies a manager object which is used to store undo information for example.

        Starting with version 0.25, layouts created with the default constructor are always editable. Before that version, they inherited the editable flag from the application.
        """
    @overload
    @classmethod
    def new(cls, editable: bool, manager: Manager) -> Layout:
        r"""
        @brief Creates a layout object attached to a manager

        This constructor specifies a manager object which is used to store undo information for example. It also allows one to specify whether the layout is editable. In editable mode, some optimizations are disabled and the layout can be manipulated through a variety of methods.

        This method was introduced in version 0.22.
        """
    def __copy__(self) -> Layout:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> Layout:
        r"""
        @brief Creates a copy of self
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Creates a layout object

        Starting with version 0.25, layouts created with the default constructor are always editable. Before that version, they inherited the editable flag from the application.
        """
    @overload
    def __init__(self, editable: bool) -> None:
        r"""
        @brief Creates a layout object

        This constructor specifies whether the layout is editable. In editable mode, some optimizations are disabled and the layout can be manipulated through a variety of methods.

        This method was introduced in version 0.22.
        """
    @overload
    def __init__(self, manager: Manager) -> None:
        r"""
        @brief Creates a layout object attached to a manager

        This constructor specifies a manager object which is used to store undo information for example.

        Starting with version 0.25, layouts created with the default constructor are always editable. Before that version, they inherited the editable flag from the application.
        """
    @overload
    def __init__(self, editable: bool, manager: Manager) -> None:
        r"""
        @brief Creates a layout object attached to a manager

        This constructor specifies a manager object which is used to store undo information for example. It also allows one to specify whether the layout is editable. In editable mode, some optimizations are disabled and the layout can be manipulated through a variety of methods.

        This method was introduced in version 0.22.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def add_cell(self, name: str) -> int:
        r"""
        @brief Adds a cell with the given name
        @return The index of the newly created cell.

        From version 0.23 on this method is deprecated because another method exists which is more convenient because is returns a \Cell object (\create_cell).
        """
    def add_lib_cell(self, library: Library, lib_cell_index: int) -> int:
        r"""
        @brief Imports a cell from the library
        @param library The reference to the library from which to import the cell
        @param lib_cell_index The index of the imported cell in the library
        @return The cell index of the new proxy cell in this layout
        This method imports the given cell from the library and creates a new proxy cell.
        The proxy cell acts as a pointer to the actual cell which still resides in the 
        library (precisely: in library.layout). The name of the new cell will be the name of
        library cell. 

        This method has been introduced in version 0.22.
        """
    def add_meta_info(self, info: LayoutMetaInfo) -> None:
        r"""
        @brief Adds meta information to the layout
        See \LayoutMetaInfo for details about layouts and meta information.
        This method has been introduced in version 0.25.
        """
    @overload
    def add_pcell_variant(self, pcell_id: int, parameters: Sequence[Any]) -> int:
        r"""
        @brief Creates a PCell variant for the given PCell ID with the given parameters
        @return The cell index of the pcell variant proxy cell
        This method will create a PCell variant proxy for a local PCell definition.
        It will create the PCell variant for the given parameters. Note that this method 
        does not allow one to create PCell instances for PCells located in a library. Use
        \add_pcell_variant with the library parameter for that purpose.

        The parameters are a sequence of variants which correspond to the parameters declared by the \PCellDeclaration object.

        The name of the new cell will be the name of the PCell. 
        If a cell with that name already exists, a new unique name is generated.

        This method has been introduced in version 0.22.
        """
    @overload
    def add_pcell_variant(self, library: Library, pcell_id: int, parameters: Sequence[Any]) -> int:
        r"""
        @brief Creates a PCell variant for a PCell located in an external library
        @return The cell index of the new proxy cell in this layout
        This method will import a PCell from a library and create a variant for the 
        given parameter set.
        Technically, this method creates a proxy to the library and creates the variant
        inside that library.  

        The parameters are a sequence of variants which correspond to the parameters declared by the \PCellDeclaration object.

        The name of the new cell will be the name of the PCell. 
        If a cell with that name already exists, a new unique name is generated.

        This method has been introduced in version 0.22.
        """
    @overload
    def add_pcell_variant(self, pcell_id: int, parameters: Dict[str, Any]) -> int:
        r"""
        @brief Creates a PCell variant for the given PCell ID with the parameters given as a name/value dictionary
        @return The cell index of the pcell variant proxy cell
        This method will create a PCell variant proxy for a local PCell definition.
        It will create the PCell variant for the given parameters. Note that this method 
        does not allow one to create PCell instances for PCells located in a library. Use
        \add_pcell_variant with the library parameter for that purpose.
        Unlike the variant using a list of parameters, this version allows specification
        of the parameters with a key/value dictionary. The keys are the parameter names
        as given by the PCell declaration.

        The parameters are a sequence of variants which correspond to the parameters declared by the \PCellDeclaration object.

        The name of the new cell will be the name of the PCell. 
        If a cell with that name already exists, a new unique name is generated.

        This method has been introduced in version 0.22.
        """
    @overload
    def add_pcell_variant(self, library: Library, pcell_id: int, parameters: Dict[str, Any]) -> int:
        r"""
        @brief Creates a PCell variant for a PCell located in an external library with the parameters given as a name/value dictionary
        @return The cell index of the new proxy cell in this layout
        This method will import a PCell from a library and create a variant for the 
        given parameter set.
        Technically, this method creates a proxy to the library and creates the variant
        inside that library.
        Unlike the variant using a list of parameters, this version allows specification
        of the parameters with a key/value dictionary. The keys are the parameter names
        as given by the PCell declaration.

        The parameters are a sequence of variants which correspond to the parameters declared by the \PCellDeclaration object.

        The name of the new cell will be the name of the PCell. 
        If a cell with that name already exists, a new unique name is generated.

        This method has been introduced in version 0.22.
        """
    def assign(self, other: Layout) -> None:
        r"""
        @brief Assigns another object to self
        """
    @overload
    def begin_shapes(self, cell: Cell, layer: int) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the given cell on the given layer
        @param cell The cell object of the initial (top) cell
        @param layer The layer from which to get the shapes
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.
        This version is convenience overload which takes a cell object instead of a cell index.

        This method is deprecated. Use \Cell#begin_shapes_rec instead.

        This method has been added in version 0.24.
        """
    @overload
    def begin_shapes(self, cell_index: int, layer: int) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the given cell on the given layer
        @param cell_index The index of the initial (top) cell
        @param layer The layer from which to get the shapes
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.

        This method is deprecated. Use \Cell#begin_shapes_rec instead.

        This method has been added in version 0.18.
        """
    @overload
    def begin_shapes_overlapping(self, cell: Cell, layer: int, region: DBox) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the given cell on the given layer using a region search, the region given in micrometer units
        @param cell The cell object for the starting cell
        @param layer The layer from which to get the shapes
        @param region The search region as a \DBox object in micrometer units
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.
        This version gives an iterator delivering shapes whose bounding box overlaps the given region.
        It is convenience overload which takes a cell object instead of a cell index.

        This method is deprecated. Use \Cell#begin_shapes_rec_overlapping instead.

        This variant has been added in version 0.25.
        """
    @overload
    def begin_shapes_overlapping(self, cell_index: int, layer: int, region: Box) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the given cell on the given layer using a region search
        @param cell_index The index of the starting cell
        @param layer The layer from which to get the shapes
        @param region The search region
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.
        This version gives an iterator delivering shapes whose bounding box overlaps the given region.

        This method is deprecated. Use \Cell#begin_shapes_rec_overlapping instead.

        This method has been added in version 0.18.
        """
    @overload
    def begin_shapes_overlapping(self, cell_index: int, layer: int, region: DBox) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the given cell on the given layer using a region search, the region given in micrometer units
        @param cell_index The index of the starting cell
        @param layer The layer from which to get the shapes
        @param region The search region as a \DBox object in micrometer units
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.
        This version gives an iterator delivering shapes whose bounding box overlaps the given region.

        This method is deprecated. Use \Cell#begin_shapes_rec_overlapping instead.

        This variant has been added in version 0.25.
        """
    @overload
    def begin_shapes_overlapping(self, cell_index: Cell, layer: int, region: Box) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the given cell on the given layer using a region search
        @param cell The cell object for the starting cell
        @param layer The layer from which to get the shapes
        @param region The search region
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.
        This version gives an iterator delivering shapes whose bounding box overlaps the given region.
        It is convenience overload which takes a cell object instead of a cell index.

        This method is deprecated. Use \Cell#begin_shapes_rec_overlapping instead.

        This method has been added in version 0.24.
        """
    @overload
    def begin_shapes_touching(self, cell: Cell, layer: int, region: Box) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the given cell on the given layer using a region search
        @param cell The cell object for the starting cell
        @param layer The layer from which to get the shapes
        @param region The search region
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.
        This version gives an iterator delivering shapes whose bounding box touches the given region.
        It is convenience overload which takes a cell object instead of a cell index.

        This method is deprecated. Use \Cell#begin_shapes_rec_touching instead.

        This method has been added in version 0.24.
        """
    @overload
    def begin_shapes_touching(self, cell: Cell, layer: int, region: DBox) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the given cell on the given layer using a region search, the region given in micrometer units
        @param cell The cell object for the starting cell
        @param layer The layer from which to get the shapes
        @param region The search region as a \DBox object in micrometer units
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.
        This version gives an iterator delivering shapes whose bounding box touches the given region.
        It is convenience overload which takes a cell object instead of a cell index.

        This method is deprecated. Use \Cell#begin_shapes_rec_touching instead.

        This variant has been added in version 0.25.
        """
    @overload
    def begin_shapes_touching(self, cell_index: int, layer: int, region: Box) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the given cell on the given layer using a region search
        @param cell_index The index of the starting cell
        @param layer The layer from which to get the shapes
        @param region The search region
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.
        This version gives an iterator delivering shapes whose bounding box touches the given region.

        This method is deprecated. Use \Cell#begin_shapes_rec_touching instead.

        This method has been added in version 0.18.
        """
    @overload
    def begin_shapes_touching(self, cell_index: int, layer: int, region: DBox) -> RecursiveShapeIterator:
        r"""
        @brief Delivers a recursive shape iterator for the shapes below the given cell on the given layer using a region search, the region given in micrometer units
        @param cell_index The index of the starting cell
        @param layer The layer from which to get the shapes
        @param region The search region as a \DBox object in micrometer units
        @return A suitable iterator

        For details see the description of the \RecursiveShapeIterator class.
        This version gives an iterator delivering shapes whose bounding box touches the given region.

        This method is deprecated. Use \Cell#begin_shapes_rec_touching instead.

        This variant has been added in version 0.25.
        """
    @overload
    def cell(self, i: int) -> Cell:
        r"""
        @brief Gets a cell object from the cell index

        @param i The cell index
        @return A reference to the cell (a \Cell object)

        If the cell index is not a valid cell index, this method will raise an error. Use \is_valid_cell_index? to test whether a given cell index is valid.
        """
    @overload
    def cell(self, name: str) -> Cell:
        r"""
        @brief Gets a cell object from the cell name

        @param name The cell name
        @return A reference to the cell (a \Cell object)

        If name is not a valid cell name, this method will return "nil".
        This method has been introduced in version 0.23 and replaces \cell_by_name.
        """
    def cell_by_name(self, name: str) -> int:
        r"""
        @brief Gets the cell index for a given name
        Returns the cell index for the cell with the given name. If no cell with this name exists, an exception is thrown.
        From version 0.23 on, a version of the \cell method is provided which returns a \Cell object for the cell with the given name or "nil" if the name is not valid. This method replaces \cell_by_name and \has_cell?
        """
    def cell_name(self, index: int) -> str:
        r"""
        @brief Gets the name for a cell with the given index
        """
    @overload
    def cells(self) -> int:
        r"""
        @brief Returns the number of cells

        @return The number of cells (the maximum cell index)
        """
    @overload
    def cells(self, name_filter: str) -> List[Cell]:
        r"""
        @brief Gets the cell objects for a given name filter

        @param name_filter The cell name filter (glob pattern)
        @return A list of \Cell object of the cells matching the pattern

        This method has been introduced in version 0.27.3.
        """
    def cleanup(self, cell_indexes_to_keep: Optional[Sequence[int]] = ...) -> None:
        r"""
        @brief Cleans up the layout
        This method will remove proxy objects that are no longer in use. After changing PCell parameters such proxy objects may still be present in the layout and are cached for later reuse. Usually they are cleaned up automatically, but in a scripting context it may be useful to clean up these cells explicitly.

        Use 'cell_indexes_to_keep' for specifying a list of cell indexes of PCell variants or library proxies you don't want to be cleaned up.

        This method has been introduced in version 0.25.
        """
    def clear(self) -> None:
        r"""
        @brief Clears the layout

        Clears the layout completely.
        """
    def clear_layer(self, layer_index: int) -> None:
        r"""
        @brief Clears a layer

        Clears the layer: removes all shapes.

        This method was introduced in version 0.19.

        @param layer_index The index of the layer to delete.
        """
    @overload
    def clip(self, cell: int, box: Box) -> int:
        r"""
        @brief Clips the given cell by the given rectangle and produce a new cell with the clip
        @param cell The cell index of the cell to clip
        @param box The clip box in database units
        @return The index of the new cell

        This method will cut a rectangular region given by the box from the given cell. The clip will be stored in a new cell whose index is returned. The clip will be performed hierarchically. The resulting cell will hold a hierarchy of child cells, which are potentially clipped versions of child cells of the original cell.
        This method has been added in version 0.21.
        """
    @overload
    def clip(self, cell: int, box: DBox) -> int:
        r"""
        @brief Clips the given cell by the given rectangle and produce a new cell with the clip
        @param cell The cell index of the cell to clip
        @param box The clip box in micrometer units
        @return The index of the new cell

        This variant which takes a micrometer-unit box has been added in version 0.28.
        """
    @overload
    def clip(self, cell: Cell, box: Box) -> Cell:
        r"""
        @brief Clips the given cell by the given rectangle and produce a new cell with the clip
        @param cell The cell reference of the cell to clip
        @param box The clip box in database units
        @return The reference to the new cell

        This variant which takes cell references instead of cell indexes has been added in version 0.28.
        """
    @overload
    def clip(self, cell: Cell, box: DBox) -> Cell:
        r"""
        @brief Clips the given cell by the given rectangle and produce a new cell with the clip
        @param cell The cell reference of the cell to clip
        @param box The clip box in micrometer units
        @return The reference to the new cell

        This variant which takes a micrometer-unit box and cell references has been added in version 0.28.
        """
    @overload
    def clip_into(self, cell: int, target: Layout, box: Box) -> int:
        r"""
        @brief Clips the given cell by the given rectangle and produce a new cell with the clip
        @param cell The cell index of the cell to clip
        @param box The clip box in database units
        @param target The target layout
        @return The index of the new cell in the target layout

        This method will cut a rectangular region given by the box from the given cell. The clip will be stored in a new cell in the target layout. The clip will be performed hierarchically. The resulting cell will hold a hierarchy of child cells, which are potentially clipped versions of child cells of the original cell.

        Please note that it is important that the database unit of the target layout is identical to the database unit of the source layout to achieve the desired results.This method also assumes that the target layout holds the same layers than the source layout. It will copy shapes to the same layers than they have been on the original layout. 
        This method has been added in version 0.21.
        """
    @overload
    def clip_into(self, cell: int, target: Layout, box: DBox) -> int:
        r"""
        @brief Clips the given cell by the given rectangle and produce a new cell with the clip
        @param cell The cell index of the cell to clip
        @param box The clip box in micrometer units
        @param target The target layout
        @return The index of the new cell in the target layout

        This variant which takes a micrometer-unit box has been added in version 0.28.
        """
    @overload
    def clip_into(self, cell: Cell, target: Layout, box: Box) -> Cell:
        r"""
        @brief Clips the given cell by the given rectangle and produce a new cell with the clip
        @param cell The reference to the cell to clip
        @param box The clip box in database units
        @param target The target layout
        @return The reference to the new cell in the target layout

        This variant which takes cell references instead of cell indexes has been added in version 0.28.
        """
    @overload
    def clip_into(self, cell: Cell, target: Layout, box: DBox) -> Cell:
        r"""
        @brief Clips the given cell by the given rectangle and produce a new cell with the clip
        @param cell The reference to the cell to clip
        @param box The clip box in micrometer units
        @param target The target layout
        @return The reference to the new cell in the target layout

        This variant which takes a micrometer-unit box and cell references has been added in version 0.28.
        """
    def convert_cell_to_static(self, cell_index: int) -> int:
        r"""
        @brief Converts a PCell or library cell to a usual (static) cell
        @return The index of the new cell
        This method will create a new cell which contains the static representation of the PCell or library proxy given by "cell_index". If that cell is not a PCell or library proxy, it won't be touched and the input cell index is returned.

        This method has been added in version 0.23.
        """
    def copy_layer(self, src: int, dest: int) -> None:
        r"""
        @brief Copies a layer

        This method was introduced in version 0.19.

        Copy a layer from the source to the target. The target is not cleared before, so that this method 
        merges shapes from the source with the target layer.

        @param src The layer index of the source layer.
        @param dest The layer index of the destination layer.
        """
    @overload
    def copy_tree_shapes(self, source_layout: Layout, cell_mapping: CellMapping) -> None:
        r"""
        @brief Copies the shapes for all given mappings in the \CellMapping object
        @param source_layout The layout where to take the shapes from
        @param cell_mapping The cell mapping object that determines how cells are identified between source and target layout

        Provide a \CellMapping object to specify pairs of cells which are mapped from the source layout to this layout. When constructing such a cell mapping object for example with \CellMapping#for_multi_cells_full, use self as the target layout. During the cell mapping construction, the cell mapper will usually create a suitable target hierarchy already. After having completed the cell mapping, use \copy_tree_shapes to copy over the shapes from the source to the target layout.

        This method has been added in version 0.26.8.
        """
    @overload
    def copy_tree_shapes(self, source_layout: Layout, cell_mapping: CellMapping, layer_mapping: LayerMapping) -> None:
        r"""
        @brief Copies the shapes for all given mappings in the \CellMapping object using the given layer mapping
        @param source_layout The layout where to take the shapes from
        @param cell_mapping The cell mapping object that determines how cells are identified between source and target layout
        @param layer_mapping Specifies which layers are copied from the source layout to the target layout

        Provide a \CellMapping object to specify pairs of cells which are mapped from the source layout to this layout. When constructing such a cell mapping object for example with \CellMapping#for_multi_cells_full, use self as the target layout. During the cell mapping construction, the cell mapper will usually create a suitable target hierarchy already. After having completed the cell mapping, use \copy_tree_shapes to copy over the shapes from the source to the target layout.

        This method has been added in version 0.26.8.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    @overload
    def create_cell(self, name: str) -> Cell:
        r"""
        @brief Creates a cell with the given name
        @param name The name of the cell to create
        @return The \Cell object of the newly created cell.

        If a cell with that name already exists, the unique name will be chosen for the new cell consisting of the given name plus a suitable suffix.

        This method has been introduce in version 0.23 and replaces \add_cell.
        """
    @overload
    def create_cell(self, name: str, lib_name: str) -> Cell:
        r"""
        @brief Creates a cell with the given name
        @param name The name of the library cell and the name of the cell to create
        @param lib_name The name of the library where to take the cell from
        @return The \Cell object of the newly created cell or an existing cell if the library cell has already been used in this layout.

        Library cells are imported by creating a 'library proxy'. This is a cell which represents the library cell in the framework of the current layout. The library proxy is linked to the library and will be updated if the library cell is changed.

        This method will look up the cell by the given name in the specified library and create a new library proxy for this cell.
        If the same library cell has already been used, the original library proxy is returned. Hence, strictly speaking this method does not always create a new cell but may return a reference to an existing cell.

        If the library name is not valid, nil is returned.

        This method has been introduce in version 0.24.
        """
    @overload
    def create_cell(self, pcell_name: str, params: Dict[str, Any]) -> Cell:
        r"""
        @brief Creates a cell as a PCell variant for the PCell with the given name
        @param pcell_name The name of the PCell and also the name of the cell to create
        @param params The PCell parameters (key/value dictionary)
        @return The \Cell object of the newly created cell or an existing cell if the PCell has already been used with these parameters.

        PCells are instantiated by creating a PCell variant. A PCell variant is linked to the PCell and represents this PCell with a particular parameter set.

        This method will look up the PCell by the PCell name and create a new PCell variant for the given parameters. If the PCell has already been instantiated with the same parameters, the original variant will be returned. Hence this method is not strictly creating a cell - only if the required variant has not been created yet.

        The parameters are specified as a key/value dictionary with the names being the ones from the PCell declaration.

        If no PCell with the given name exists, nil is returned.

        This method has been introduce in version 0.24.
        """
    @overload
    def create_cell(self, pcell_name: str, lib_name: str, params: Dict[str, Any]) -> Cell:
        r"""
        @brief Creates a cell for a PCell with the given PCell name from the given library
        @param pcell_name The name of the PCell and also the name of the cell to create
        @param lib_name The name of the library where to take the PCell from
        @param params The PCell parameters (key/value dictionary)
        @return The \Cell object of the newly created cell or an existing cell if this PCell has already been used with the given parameters

        This method will look up the PCell by the PCell name in the specified library and create a new PCell variant for the given parameters plus the library proxy. The parameters must be specified as a key/value dictionary with the names being the ones from the PCell declaration.

        If no PCell with the given name exists or the library name is not valid, nil is returned. Note that this function - despite the name - may not always create a new cell, but return an existing cell if the PCell from the library has already been used with the given parameters.

        This method has been introduce in version 0.24.
        """
    def delete_cell(self, cell_index: int) -> None:
        r"""
        @brief Deletes a cell 

        This deletes a cell but not the sub cells of the cell.
        These subcells will likely become new top cells unless they are used
        otherwise.
        All instances of this cell are deleted as well.
        Hint: to delete multiple cells, use "delete_cells" which is 
        far more efficient in this case.

        @param cell_index The index of the cell to delete

        This method has been introduced in version 0.20.
        """
    def delete_cell_rec(self, cell_index: int) -> None:
        r"""
        @brief Deletes a cell plus all subcells

        This deletes a cell and also all sub cells of the cell.
        In contrast to \prune_cell, all cells are deleted together with their instances even if they are used otherwise.

        @param cell_index The index of the cell to delete

        This method has been introduced in version 0.20.
        """
    def delete_cells(self, cell_index_list: Sequence[int]) -> None:
        r"""
        @brief Deletes multiple cells

        This deletes the cells but not the sub cells of these cells.
        These subcells will likely become new top cells unless they are used
        otherwise.
        All instances of these cells are deleted as well.

        @param cell_index_list An array of cell indices of the cells to delete

        This method has been introduced in version 0.20.
        """
    def delete_layer(self, layer_index: int) -> None:
        r"""
        @brief Deletes a layer

        This method frees the memory allocated for the shapes of this layer and remembers the
        layer's index for reuse when the next layer is allocated.

        @param layer_index The index of the layer to delete.
        """
    def delete_property(self, key: Any) -> None:
        r"""
        @brief Deletes the user property with the given key
        This method is a convenience method that deletes the property with the given key. It does nothing if no property with that key exists. Using that method is more convenient than creating a new property set with a new ID and assigning that properties ID.
        This method may change the properties ID.

        This method has been introduced in version 0.24.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dump_mem_statistics(self, detailed: Optional[bool] = ...) -> None:
        r"""
        @hide
        """
    def dup(self) -> Layout:
        r"""
        @brief Creates a copy of self
        """
    def each_cell(self) -> Iterator[Cell]:
        r"""
        @brief Iterates the unsorted cell list
        """
    def each_cell_bottom_up(self) -> Iterator[int]:
        r"""
        @brief Iterates the bottom-up sorted cell list

        In bottom-up traversal a cell is not delivered before
        the last child cell of this cell has been delivered.
        The bottom-up iterator does not deliver cells but cell
        indices actually.
        """
    def each_cell_top_down(self) -> Iterator[int]:
        r"""
        @brief Iterates the top-down sorted cell list

        The top-down cell list has the property of delivering all
        cells before they are instantiated. In addition the first
        cells are all top cells. There is at least one top cell.
        The top-down iterator does not deliver cells but cell
        indices actually.
        @brief begin iterator of the top-down sorted cell list
        """
    def each_meta_info(self) -> Iterator[LayoutMetaInfo]:
        r"""
        @brief Iterates over the meta information of the layout
        See \LayoutMetaInfo for details about layouts and meta information.

        This method has been introduced in version 0.25.
        """
    def each_top_cell(self) -> Iterator[int]:
        r"""
        @brief Iterates the top cells
        A layout may have an arbitrary number of top cells. The usual case however is that there is one top cell.
        """
    def end_changes(self) -> None:
        r"""
        @brief Cancels the "in changes" state (see "start_changes")
        """
    @overload
    def find_layer(self, info: LayerInfo) -> Any:
        r"""
        @brief Finds a layer with the given properties

        If a layer with the given properties already exists, this method will return the index of that layer.If no such layer exists, it will return nil.

        This method has been introduced in version 0.23.
        """
    @overload
    def find_layer(self, name: str) -> Any:
        r"""
        @brief Finds a layer with the given name

        If a layer with the given name already exists, this method will return the index of that layer.If no such layer exists, it will return nil.

        This method has been introduced in version 0.23.
        """
    @overload
    def find_layer(self, layer: int, datatype: int) -> Any:
        r"""
        @brief Finds a layer with the given layer and datatype number

        If a layer with the given layer/datatype already exists, this method will return the index of that layer.If no such layer exists, it will return nil.

        This method has been introduced in version 0.23.
        """
    @overload
    def find_layer(self, layer: int, datatype: int, name: str) -> Any:
        r"""
        @brief Finds a layer with the given layer and datatype number and name

        If a layer with the given layer/datatype/name already exists, this method will return the index of that layer.If no such layer exists, it will return nil.

        This method has been introduced in version 0.23.
        """
    def flatten(self, cell_index: int, levels: int, prune: bool) -> None:
        r"""
        @brief Flattens the given cell

        This method propagates all shapes and instances from the specified number of hierarchy levels below into the given cell.
        It also removes the instances of the cells from which the shapes came from, but does not remove the cells themselves if prune is set to false.
        If prune is set to true, these cells are removed if not used otherwise.

        @param cell_index The cell which should be flattened
        @param levels The number of hierarchy levels to flatten (-1: all, 0: none, 1: one level etc.)
        @param prune Set to true to remove orphan cells.

        This method has been introduced in version 0.20.
        """
    def flatten_into(self, source_cell_index: int, target_cell_index: int, trans: ICplxTrans, levels: int) -> None:
        r"""
        @brief Flattens the given cell into another cell

        This method works like 'flatten', but allows specification of a target cell which can be different from the source cell plus a transformation which is applied for all shapes and instances in the target cell.

        In contrast to the 'flatten' method, the source cell is not modified.

        @param source_cell_index The source cell which should be flattened
        @param target_cell_index The target cell into which the resulting objects are written
        @param trans The transformation to apply on the output shapes and instances
        @param levels The number of hierarchy levels to flatten (-1: all, 0: none, 1: one level etc.)

        This method has been introduced in version 0.24.
        """
    def get_info(self, index: int) -> LayerInfo:
        r"""
        @brief Gets the info structure for a specified layer
        """
    def guiding_shape_layer(self) -> int:
        r"""
        @brief Returns the index of the guiding shape layer
        The guiding shape layer is used to store guiding shapes for PCells.

        This method has been added in version 0.22.
        """
    def has_cell(self, name: str) -> bool:
        r"""
        @brief Returns true if a cell with a given name exists
        Returns true, if the layout has a cell with the given name
        """
    def has_prop_id(self) -> bool:
        r"""
        @brief Returns true, if the layout has user properties

        This method has been introduced in version 0.24.
        """
    @overload
    def insert(self, cell_index: int, layer: int, edge_pairs: EdgePairs) -> None:
        r"""
        @brief Inserts an edge pair collection into the given cell and layer
        If the edge pair collection is (conceptionally) flat, it will be inserted into the cell's shapes list as a flat sequence of edge pairs.
        If the edge pair collection is deep (hierarchical), it will create a subhierarchy below the given cell and it's edge pairs will be put into the respective cells. Suitable subcells will be picked for inserting the edge pairs. If a hierarchy already exists below the given cell, the algorithm will try to reuse this hierarchy.

        This method has been introduced in version 0.27.
        """
    @overload
    def insert(self, cell_index: int, layer: int, edges: Edges) -> None:
        r"""
        @brief Inserts an edge collection into the given cell and layer
        If the edge collection is (conceptionally) flat, it will be inserted into the cell's shapes list as a flat sequence of edges.
        If the edge collection is deep (hierarchical), it will create a subhierarchy below the given cell and it's edges will be put into the respective cells. Suitable subcells will be picked for inserting the edges. If a hierarchy already exists below the given cell, the algorithm will try to reuse this hierarchy.

        This method has been introduced in version 0.26.
        """
    @overload
    def insert(self, cell_index: int, layer: int, region: Region) -> None:
        r"""
        @brief Inserts a region into the given cell and layer
        If the region is (conceptionally) a flat region, it will be inserted into the cell's shapes list as a flat sequence of polygons.
        If the region is a deep (hierarchical) region, it will create a subhierarchy below the given cell and it's shapes will be put into the respective cells. Suitable subcells will be picked for inserting the shapes. If a hierarchy already exists below the given cell, the algorithm will try to reuse this hierarchy.

        This method has been introduced in version 0.26.
        """
    @overload
    def insert(self, cell_index: int, layer: int, texts: Texts) -> None:
        r"""
        @brief Inserts an text collection into the given cell and layer
        If the text collection is (conceptionally) flat, it will be inserted into the cell's shapes list as a flat sequence of texts.
        If the text collection is deep (hierarchical), it will create a subhierarchy below the given cell and it's texts will be put into the respective cells. Suitable subcells will be picked for inserting the texts. If a hierarchy already exists below the given cell, the algorithm will try to reuse this hierarchy.

        This method has been introduced in version 0.27.
        """
    def insert_layer(self, props: LayerInfo) -> int:
        r"""
        @brief Inserts a new layer with the given properties
        @return The index of the newly created layer
        """
    def insert_layer_at(self, index: int, props: LayerInfo) -> None:
        r"""
        @brief Inserts a new layer with the given properties at the given index
        This method will associate the given layer info with the given layer index. If a layer with that index already exists, this method will change the properties of the layer with that index. Otherwise a new layer is created.
        """
    def insert_special_layer(self, props: LayerInfo) -> int:
        r"""
        @brief Inserts a new special layer with the given properties

        Special layers can be used to represent objects that should not participate in normal viewing or other related operations. Special layers are not reported as valid layers.

        @return The index of the newly created layer
        """
    def insert_special_layer_at(self, index: int, props: LayerInfo) -> None:
        r"""
        @brief Inserts a new special layer with the given properties at the given index

        See \insert_special_layer for a description of special layers.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_editable(self) -> bool:
        r"""
        @brief Returns a value indicating whether the layout is editable.
        @return True, if the layout is editable.
        If a layout is editable, in general manipulation methods are enabled and some optimizations are disabled (i.e. shape arrays are expanded).

        This method has been introduced in version 0.22.
        """
    def is_free_layer(self, layer_index: int) -> bool:
        r"""
        @brief Returns true, if a layer index is a free (unused) layer index

        @return true, if this is the case

        This method has been introduced in version 0.26.
        """
    def is_special_layer(self, layer_index: int) -> bool:
        r"""
        @brief Returns true, if a layer index is a special layer index

        @return true, if this is the case
        """
    def is_valid_cell_index(self, cell_index: int) -> bool:
        r"""
        @brief Returns true, if a cell index is a valid index

        @return true, if this is the case
        This method has been added in version 0.20.
        """
    def is_valid_layer(self, layer_index: int) -> bool:
        r"""
        @brief Returns true, if a layer index is a valid normal layout layer index

        @return true, if this is the case
        """
    @overload
    def layer(self) -> int:
        r"""
        @brief Creates a new internal layer

        This method will create a new internal layer and return the layer index for this layer.
        The layer does not have any properties attached to it. That means, it is not going to be saved to a layout file unless it is given database properties with \set_info.

        This method is equivalent to "layer(RBA::LayerInfo::new())".

        This method has been introduced in version 0.25.
        """
    @overload
    def layer(self, info: LayerInfo) -> int:
        r"""
        @brief Finds or creates a layer with the given properties

        If a layer with the given properties already exists, this method will return the index of that layer.If no such layer exists, a new one with these properties will be created and its index will be returned. If "info" is anonymous (info.anonymous? is true), a new layer will always be created.

        This method has been introduced in version 0.23.
        """
    @overload
    def layer(self, name: str) -> int:
        r"""
        @brief Finds or creates a layer with the given name

        If a layer with the given name already exists, this method will return the index of that layer.If no such layer exists, a new one with this name will be created and its index will be returned.

        This method has been introduced in version 0.23.
        """
    @overload
    def layer(self, layer: int, datatype: int) -> int:
        r"""
        @brief Finds or creates a layer with the given layer and datatype number

        If a layer with the given layer/datatype already exists, this method will return the index of that layer.If no such layer exists, a new one with these properties will be created and its index will be returned.

        This method has been introduced in version 0.23.
        """
    @overload
    def layer(self, layer: int, datatype: int, name: str) -> int:
        r"""
        @brief Finds or creates a layer with the given layer and datatype number and name

        If a layer with the given layer/datatype/name already exists, this method will return the index of that layer.If no such layer exists, a new one with these properties will be created and its index will be returned.

        This method has been introduced in version 0.23.
        """
    def layer_indexes(self) -> List[int]:
        r"""
        @brief Gets a list of valid layer's indices
        This method returns an array with layer indices representing valid layers.

        This method has been introduced in version 0.19.
        """
    def layer_indices(self) -> List[int]:
        r"""
        @brief Gets a list of valid layer's indices
        This method returns an array with layer indices representing valid layers.

        This method has been introduced in version 0.19.
        """
    def layer_infos(self) -> List[LayerInfo]:
        r"""
        @brief Gets a list of valid layer's properties
        The method returns an array with layer properties representing valid layers.
        The sequence and length of this list corresponds to that of \layer_indexes.

        This method has been introduced in version 0.25.
        """
    def layers(self) -> int:
        r"""
        @brief Returns the number of layers
        The number of layers reports the maximum (plus 1) layer index used so far. Not all of the layers with an index in the range of 0 to layers-1 needs to be a valid layer. These layers can be either valid, special or unused. Use \is_valid_layer? and \is_special_layer? to test for the first two states.
        """
    def library(self) -> Library:
        r"""
        @brief Gets the library this layout lives in or nil if the layout is not part of a library
        This attribute has been introduced in version 0.27.5.
        """
    def meta_info_value(self, name: str) -> str:
        r"""
        @brief Gets the meta information value for a given name
        See \LayoutMetaInfo for details about layouts and meta information.

        If no meta information with the given name exists, an empty string will be returned.

        This method has been introduced in version 0.25.
        """
    def move_layer(self, src: int, dest: int) -> None:
        r"""
        @brief Moves a layer

        This method was introduced in version 0.19.

        Move a layer from the source to the target. The target is not cleared before, so that this method 
        merges shapes from the source with the target layer. The source layer is empty after that operation.

        @param src The layer index of the source layer.
        @param dest The layer index of the destination layer.
        """
    @overload
    def move_tree_shapes(self, source_layout: Layout, cell_mapping: CellMapping) -> None:
        r"""
        @brief Moves the shapes for all given mappings in the \CellMapping object

        This method acts like the corresponding \copy_tree_shapes method, but removes the shapes from the source layout after they have been copied.

        This method has been added in version 0.26.8.
        """
    @overload
    def move_tree_shapes(self, source_layout: Layout, cell_mapping: CellMapping, layer_mapping: LayerMapping) -> None:
        r"""
        @brief Moves the shapes for all given mappings in the \CellMapping object using the given layer mapping

        This method acts like the corresponding \copy_tree_shapes method, but removes the shapes from the source layout after they have been copied.

        This method has been added in version 0.26.8.
        """
    @overload
    def multi_clip(self, cell: int, boxes: Sequence[Box]) -> List[int]:
        r"""
        @brief Clips the given cell by the given rectangles and produces new cells with the clips, one for each rectangle.
        @param cell The cell index of the cell to clip
        @param boxes The clip boxes in database units
        @return The indexes of the new cells

        This method will cut rectangular regions given by the boxes from the given cell. The clips will be stored in a new cells whose indexed are returned. The clips will be performed hierarchically. The resulting cells will hold a hierarchy of child cells, which are potentially clipped versions of child cells of the original cell. This version is somewhat more efficient than doing individual clips because the clip cells may share clipped versions of child cells.

        This method has been added in version 0.21.
        """
    @overload
    def multi_clip(self, cell: int, boxes: Sequence[DBox]) -> List[int]:
        r"""
        @brief Clips the given cell by the given rectangles and produces new cells with the clips, one for each rectangle.
        @param cell The cell index of the cell to clip
        @param boxes The clip boxes in micrometer units
        @return The indexes of the new cells

        This variant which takes micrometer-unit boxes has been added in version 0.28.
        """
    @overload
    def multi_clip(self, cell: Cell, boxes: Sequence[Box]) -> List[Cell]:
        r"""
        @brief Clips the given cell by the given rectangles and produces new cells with the clips, one for each rectangle.
        @param cell The reference to the cell to clip
        @param boxes The clip boxes in database units
        @return The references to the new cells

        This variant which takes cell references has been added in version 0.28.
        """
    @overload
    def multi_clip(self, cell: Cell, boxes: Sequence[DBox]) -> List[Cell]:
        r"""
        @brief Clips the given cell by the given rectangles and produces new cells with the clips, one for each rectangle.
        @param cell The reference to the cell to clip
        @param boxes The clip boxes in micrometer units
        @return The references to the new cells

        This variant which takes cell references and micrometer-unit boxes has been added in version 0.28.
        """
    @overload
    def multi_clip_into(self, cell: int, target: Layout, boxes: Sequence[Box]) -> List[int]:
        r"""
        @brief Clips the given cell by the given rectangles and produces new cells with the clips, one for each rectangle.
        @param cell The cell index of the cell to clip
        @param boxes The clip boxes in database units
        @param target The target layout
        @return The indexes of the new cells

        This method will cut rectangular regions given by the boxes from the given cell. The clips will be stored in a new cells in the given target layout. The clips will be performed hierarchically. The resulting cells will hold a hierarchy of child cells, which are potentially clipped versions of child cells of the original cell. This version is somewhat more efficient than doing individual clips because the clip cells may share clipped versions of child cells.

        Please note that it is important that the database unit of the target layout is identical to the database unit of the source layout to achieve the desired results. This method also assumes that the target layout holds the same layers than the source layout. It will copy shapes to the same layers than they have been on the original layout.

        This method has been added in version 0.21.
        """
    @overload
    def multi_clip_into(self, cell: int, target: Layout, boxes: Sequence[DBox]) -> List[int]:
        r"""
        @brief Clips the given cell by the given rectangles and produces new cells with the clips, one for each rectangle.
        @param cell The cell index of the cell to clip
        @param boxes The clip boxes in database units
        @param target The target layout
        @return The indexes of the new cells

        This variant which takes micrometer-unit boxes has been added in version 0.28.
        """
    @overload
    def multi_clip_into(self, cell: Cell, target: Layout, boxes: Sequence[Box]) -> List[Cell]:
        r"""
        @brief Clips the given cell by the given rectangles and produces new cells with the clips, one for each rectangle.
        @param cell The reference the cell to clip
        @param boxes The clip boxes in database units
        @param target The target layout
        @return The references to the new cells

        This variant which takes cell references boxes has been added in version 0.28.
        """
    @overload
    def multi_clip_into(self, cell: Cell, target: Layout, boxes: Sequence[DBox]) -> List[Cell]:
        r"""
        @brief Clips the given cell by the given rectangles and produces new cells with the clips, one for each rectangle.
        @param cell The reference the cell to clip
        @param boxes The clip boxes in micrometer units
        @param target The target layout
        @return The references to the new cells

        This variant which takes cell references and micrometer-unit boxes has been added in version 0.28.
        """
    @overload
    def pcell_declaration(self, name: str) -> PCellDeclaration_Native:
        r"""
        @brief Gets a reference to the PCell declaration for the PCell with the given name
        Returns a reference to the local PCell declaration with the given name. If the name
        is not a valid PCell name, this method returns nil.

        Usually this method is used on library layouts that define
        PCells. Note that this method cannot be used on the layouts using the PCell from 
        a library.

        This method has been introduced in version 0.22.
        """
    @overload
    def pcell_declaration(self, pcell_id: int) -> PCellDeclaration_Native:
        r"""
        @brief Gets a reference to the PCell declaration for the PCell with the given PCell ID.
        Returns a reference to the local PCell declaration with the given PCell id. If the parameter
        is not a valid PCell ID, this method returns nil. The PCell ID is the number returned 
        by \register_pcell for example.

        Usually this method is used on library layouts that define
        PCells. Note that this method cannot be used on the layouts using the PCell from 
        a library.

        This method has been introduced in version 0.22.
        """
    def pcell_id(self, name: str) -> int:
        r"""
        @brief Gets the ID of the PCell with the given name
        This method is equivalent to 'pcell_declaration(name).id'.

        This method has been introduced in version 0.22.
        """
    def pcell_ids(self) -> List[int]:
        r"""
        @brief Gets the IDs of the PCells registered in the layout
        Returns an array of PCell IDs.

        This method has been introduced in version 0.24.
        """
    def pcell_names(self) -> List[str]:
        r"""
        @brief Gets the names of the PCells registered in the layout
        Returns an array of PCell names.

        This method has been introduced in version 0.24.
        """
    def properties(self, properties_id: int) -> List[Any]:
        r"""
        @brief Gets the properties set for a given properties ID

        Basically performs the backward conversion of the 'properties_id' method. Given a properties ID, returns the properties set as an array of pairs of variants. In this array, each key and the value are stored as pairs (arrays with two elements).
        If the properties ID is not valid, an empty array is returned.

        @param properties_id The properties ID to get the properties for
        @return The array of variants (see \properties_id)
        """
    def properties_id(self, properties: Sequence[Any]) -> int:
        r"""
        @brief Gets the properties ID for a given properties set

        Before a set of properties can be attached to a shape, it must be converted into an ID that is unique for that set. The properties set must be given as a list of pairs of variants, each pair describing a name and a value. The name acts as the key for the property and does not need to be a string (it can be an integer or double value as well).
        The backward conversion can be performed with the 'properties' method.

        @param properties The array of pairs of variants (both elements can be integer, double or string)
        @return The unique properties ID for that set
        """
    def property(self, key: Any) -> Any:
        r"""
        @brief Gets the user property with the given key
        This method is a convenience method that gets the property with the given key. If no property with that key exists, it will return nil. Using that method is more convenient than using the properties ID to retrieve the property value. 
        This method has been introduced in version 0.24.
        """
    def prune_cell(self, cell_index: int, levels: int) -> None:
        r"""
        @brief Deletes a cell plus subcells not used otherwise

        This deletes a cell and also all sub cells of the cell which are not used otherwise.
        The number of hierarchy levels to consider can be specified as well. One level of hierarchy means that only the direct children of the cell are deleted with the cell itself.
        All instances of this cell are deleted as well.

        @param cell_index The index of the cell to delete
        @param levels The number of hierarchy levels to consider (-1: all, 0: none, 1: one level etc.)

        This method has been introduced in version 0.20.
        """
    def prune_subcells(self, cell_index: int, levels: int) -> None:
        r"""
        @brief Deletes all sub cells of the cell which are not used otherwise down to the specified level of hierarchy

        This deletes all sub cells of the cell which are not used otherwise.
        All instances of the deleted cells are deleted as well.
        It is possible to specify how many levels of hierarchy below the given root cell are considered.

        @param cell_index The root cell from which to delete a sub cells
        @param levels The number of hierarchy levels to consider (-1: all, 0: none, 1: one level etc.)

        This method has been introduced in version 0.20.
        """
    @overload
    def read(self, filename: str) -> LayerMap:
        r"""
        @brief Load the layout from the given file
        The format of the file is determined automatically and automatic unzipping is provided. No particular options can be specified.
        @param filename The name of the file to load.
        @return A layer map that contains the mapping used by the reader including the layers that have been created.
        This method has been added in version 0.18.
        """
    @overload
    def read(self, filename: str, options: LoadLayoutOptions) -> LayerMap:
        r"""
        @brief Load the layout from the given file with options
        The format of the file is determined automatically and automatic unzipping is provided. In this version, some reader options can be specified. @param filename The name of the file to load.
        @param options The options object specifying further options for the reader.
        @return A layer map that contains the mapping used by the reader including the layers that have been created.
        This method has been added in version 0.18.
        """
    def refresh(self) -> None:
        r"""
        @brief Calls \Cell#refresh on all cells inside this layout
        This method is useful to recompute all PCells from a layout. Note that this does not update PCells which are linked from a library. To recompute PCells from a library, you need to use \Library#refresh on the library object from which the PCells are imported.

        This method has been introduced in version 0.27.9.
        """
    def register_pcell(self, name: str, declaration: PCellDeclaration_Native) -> int:
        r"""
        @brief Registers a PCell declaration under the given name
        Registers a local PCell in the current layout. If a declaration with that name
        already exists, it is replaced with the new declaration.

        This method has been introduced in version 0.22.
        """
    def remove_meta_info(self, name: str) -> None:
        r"""
        @brief Removes meta information from the layout
        See \LayoutMetaInfo for details about layouts and meta information.
        This method has been introduced in version 0.25.
        """
    def rename_cell(self, index: int, name: str) -> None:
        r"""
        @brief Renames the cell with given index
        The cell with the given index is renamed to the given name. NOTE: it is not ensured that the name is unique. This method allows assigning identical names to different cells which usually breaks things.
        Consider using \unique_cell_name to generate truely unique names.
        """
    @overload
    def scale_and_snap(self, cell: Cell, grid: int, mult: int, div: int) -> None:
        r"""
        @brief Scales and snaps the layout below a given cell by the given rational factor and snaps to the given grid

        This method is useful to scale a layout by a non-integer factor. The scale factor is given by the rational number mult / div. After scaling, the layout will be snapped to the given grid.

        Snapping happens 'as-if-flat' - that is, touching edges will stay touching, regardless of their hierarchy path. To achieve this, this method usually needs to produce cell variants.

        This method has been introduced in version 0.26.1.
        """
    @overload
    def scale_and_snap(self, cell_index: int, grid: int, mult: int, div: int) -> None:
        r"""
        @brief Scales and snaps the layout below a given cell by the given rational factor and snaps to the given grid

        Like the other version of \scale_and_snap, but taking a cell index for the argument.

        This method has been introduced in version 0.26.1.
        """
    def set_info(self, index: int, props: LayerInfo) -> None:
        r"""
        @brief Sets the info structure for a specified layer
        """
    def set_property(self, key: Any, value: Any) -> None:
        r"""
        @brief Sets the user property with the given key to the given value
        This method is a convenience method that sets the property with the given key to the given value. If no property with that key exists, it will create one. Using that method is more convenient than creating a new property set with a new ID and assigning that properties ID.
        This method may change the properties ID. Note: GDS only supports integer keys. OASIS supports numeric and string keys. 
        This method has been introduced in version 0.24.
        """
    def start_changes(self) -> None:
        r"""
        @brief Signals the start of an operation bringing the layout into invalid state

        This method should be called whenever the layout is
        about to be brought into an invalid state. After calling
        this method, \under_construction? returns true which 
        tells foreign code (i.e. the asynchronous painter or the cell tree view)
        not to use this layout object.

        This state is cancelled by the \end_changes method.
        The start_changes method can be called multiple times
        and must be cancelled the same number of times.

        This method can be used to speed up certain operations. For example iterating over the layout with a \RecursiveShapeIterator while modifying other layers of the layout can be very inefficient, because inside the loop the layout's state is invalidate and updated frequently.
        Putting a update and start_changes sequence before the loop (use both methods in that order!) and a end_changes call after the loop can improve the performance dramatically.

        In addition, it can be necessary to prevent redraw operations in certain cases by using start_changes .. end_changes, in particular when it is possible to put a layout object into an invalid state temporarily.

        While the layout is under construction \update can be called to update the internal state explicitly if required.
        This for example might be necessary to update the cell bounding boxes or to redo the sorting for region queries.
        """
    def swap_layers(self, a: int, b: int) -> None:
        r"""
        @brief Swap two layers

        Swaps the shapes of both layers.

        This method was introduced in version 0.19.

        @param a The first of the layers to swap.
        @param b The second of the layers to swap.
        """
    def technology(self) -> Technology:
        r"""
        @brief Gets the \Technology object of the technology this layout is associated with or nil if the layout is not associated with a technology
        This method has been introduced in version 0.27. Before that, the technology has been kept in the 'technology' meta data element.
        """
    def top_cell(self) -> Cell:
        r"""
        @brief Returns the top cell object
        @return The \Cell object of the top cell
        If the layout has a single top cell, this method returns the top cell's \Cell object.
        If the layout does not have a top cell, this method returns "nil". If the layout has multiple
        top cells, this method raises an error.

        This method has been introduced in version 0.23.
        """
    def top_cells(self) -> List[Cell]:
        r"""
        @brief Returns the top cell objects
        @return The \Cell objects of the top cells
        This method returns and array of \Cell objects representing the top cells of the layout.
        This array can be empty, if the layout does not have a top cell (i.e. no cell at all).

        This method has been introduced in version 0.23.
        """
    @overload
    def transform(self, trans: DCplxTrans) -> None:
        r"""
        @brief Transforms the layout with the given complex integer transformation, which is in micrometer units
        This variant will internally translate the transformation's displacement into database units. Apart from that, it behaves identical to the version with a \ICplxTrans argument.

        This method has been introduced in version 0.23.
        """
    @overload
    def transform(self, trans: DTrans) -> None:
        r"""
        @brief Transforms the layout with the given transformation, which is in micrometer units
        This variant will internally translate the transformation's displacement into database units. Apart from that, it behaves identical to the version with a \Trans argument.

        This variant has been introduced in version 0.25.
        """
    @overload
    def transform(self, trans: ICplxTrans) -> None:
        r"""
        @brief Transforms the layout with the given complex integer transformation

        This method has been introduced in version 0.23.
        """
    @overload
    def transform(self, trans: Trans) -> None:
        r"""
        @brief Transforms the layout with the given transformation

        This method has been introduced in version 0.23.
        """
    def under_construction(self) -> bool:
        r"""
        @brief Returns true if the layout object is under construction

        A layout object is either under construction if a transaction
        is ongoing or the layout is brought into invalid state by
        "start_changes".
        """
    def unique_cell_name(self, name: str) -> str:
        r"""
        @brief Creates a new unique cell name from the given name
        @return A unique name derived from the argument

        If a cell with the given name exists, a suffix will be added to make the name unique. Otherwise, the argument will be returned unchanged.

        The returned name can be used to rename cells without risk of creating name clashes.

        This method has been introduced in version 0.28.
        """
    def update(self) -> None:
        r"""
        @brief Updates the internals of the layout
        This method updates the internal state of the layout. Usually this is done automatically
        This method is provided to ensure this explicitly. This can be useful while using \start_changes and \end_changes to wrap a performance-critical operation. See \start_changes for more details.
        """
    @overload
    def write(self, filename: str) -> None:
        r"""
        @brief Writes the layout to a stream file
        @param filename The file to which to write the layout
        """
    @overload
    def write(self, filename: str, options: SaveLayoutOptions) -> None:
        r"""
        @brief Writes the layout to a stream file
        @param filename The file to which to write the layout
        @param options The option set to use for writing. See \SaveLayoutOptions for details

        This version automatically determines the compression mode from the file name. The file is written with zlib compression if the suffix is ".gz" or ".gzip".

        This variant has been introduced in version 0.23.
        """
    @overload
    def write(self, filename: str, gzip: bool, options: SaveLayoutOptions) -> None:
        r"""
        @brief Writes the layout to a stream file
        @param filename The file to which to write the layout
        @param gzip Ignored
        @param options The option set to use for writing. See \SaveLayoutOptions for details

        Starting with version 0.23, this variant is deprecated since the more convenient variant with two parameters automatically determines the compression mode from the file name. The gzip parameter is ignored staring with version 0.23.
        """

class SaveLayoutOptions:
    r"""
    @brief Options for saving layouts

    This class describes the various options for saving a layout to a stream file (GDS2, OASIS and others).
    There are: layers to be saved, cell or cells to be saved, scale factor, format, database unit
    and format specific options.

    Usually the default constructor provides a suitable object. Please note, that the format written is "GDS2" by default. Either explicitly set a format using \format= or derive the format from the file name using \set_format_from_filename.

    The layers are specified by either selecting all layers or by defining layer by layer using the
    \add_layer method. \select_all_layers will explicitly select all layers for saving, \deselect_all_layers will explicitly clear the list of layers.

    Cells are selected in a similar fashion: by default, all cells are selected. Using \add_cell, specific
    cells can be selected for saving. All these cells plus their hierarchy will then be written to the stream file.

    """
    cif_blank_separator: bool
    r"""
    Getter:
    @brief Gets a flag indicating whether blanks shall be used as x/y separator characters
    See \cif_blank_separator= method for a description of that property.
    This property has been added in version 0.23.10.

    The predicate version (cif_blank_separator?) has been added in version 0.25.1.

    Setter:
    @brief Sets a flag indicating whether blanks shall be used as x/y separator characters
    If this property is set to true, the x and y coordinates are separated with blank characters rather than comma characters.
    This property has been added in version 0.23.10.
    """
    cif_dummy_calls: bool
    r"""
    Getter:
    @brief Gets a flag indicating whether dummy calls shall be written
    See \cif_dummy_calls= method for a description of that property.
    This property has been added in version 0.23.10.

    The predicate version (cif_blank_separator?) has been added in version 0.25.1.

    Setter:
    @brief Sets a flag indicating whether dummy calls shall be written
    If this property is set to true, dummy calls will be written in the top level entity of the CIF file calling every top cell.
    This option is useful for enhanced compatibility with other tools.

    This property has been added in version 0.23.10.
    """
    dbu: float
    r"""
    Getter:
    @brief Get the explicit database unit if one is set

    See \dbu= for a description of that attribute.

    Setter:
    @brief Set the database unit to be used in the stream file

    By default, the database unit of the layout is used. This method allows one to explicitly use a different
    database unit. A scale factor is introduced automatically which scales all layout objects accordingly so their physical dimensions remain the same. When scaling to a larger database unit or one that is not an integer fraction of the original one, rounding errors may occur and the layout may become slightly distorted.
    """
    dxf_polygon_mode: int
    r"""
    Getter:
    @brief Specifies how to write polygons.
    See \dxf_polygon_mode= for a description of this property.

    This property has been added in version 0.21.3.

    Setter:
    @brief Specifies how to write polygons.
    The mode is 0 (write POLYLINE entities), 1 (write LWPOLYLINE entities), 2 (decompose into SOLID entities), 3 (write HATCH entities), or 4 (write LINE entities).

    This property has been added in version 0.21.3. '4', in version 0.25.6.
    """
    format: str
    r"""
    Getter:
    @brief Gets the format name

    See \format= for a description of that method.

    Setter:
    @brief Select a format
    The format string can be either "GDS2", "OASIS", "CIF" or "DXF". Other formats may be available if
    a suitable plugin is installed.
    """
    gds2_libname: str
    r"""
    Getter:
    @brief Get the library name
    See \gds2_libname= method for a description of the library name.
    This property has been added in version 0.18.

    Setter:
    @brief Set the library name

    The library name is the string written into the LIBNAME records of the GDS file.
    The library name should not be an empty string and is subject to certain limitations in the character choice.

    This property has been added in version 0.18.
    """
    gds2_max_cellname_length: int
    r"""
    Getter:
    @brief Get the maximum length of cell names
    See \gds2_max_cellname_length= method for a description of the maximum cell name length.
    This property has been added in version 0.18.

    Setter:
    @brief Maximum length of cell names

    This property describes the maximum number of characters for cell names. 
    Longer cell names will be shortened.

    This property has been added in version 0.18.
    """
    gds2_max_vertex_count: int
    r"""
    Getter:
    @brief Gets the maximum number of vertices for polygons to write
    See \gds2_max_vertex_count= method for a description of the maximum vertex count.
    This property has been added in version 0.18.

    Setter:
    @brief Sets the maximum number of vertices for polygons to write
    This property describes the maximum number of point for polygons in GDS2 files.
    Polygons with more points will be split.
    The minimum value for this property is 4. The maximum allowed value is about 4000 or 8000, depending on the
    GDS2 interpretation. If \gds2_multi_xy_records is true, this
    property is not used. Instead, the number of points is unlimited.

    This property has been added in version 0.18.
    """
    gds2_multi_xy_records: bool
    r"""
    Getter:
    @brief Gets the property enabling multiple XY records for BOUNDARY elements
    See \gds2_multi_xy_records= method for a description of this property.
    This property has been added in version 0.18.

    Setter:
    @brief Uses multiple XY records in BOUNDARY elements for unlimited large polygons

    Setting this property to true allows producing polygons with an unlimited number of points 
    at the cost of incompatible formats. Setting it to true disables the \gds2_max_vertex_count setting.

    This property has been added in version 0.18.
    """
    gds2_no_zero_length_paths: bool
    r"""
    Getter:
    @brief Gets a value indicating whether zero-length paths are eliminated

    This property has been added in version 0.23.

    Setter:
    @brief Eliminates zero-length paths if true

    If this property is set to true, paths with zero length will be converted to BOUNDARY objects.


    This property has been added in version 0.23.
    """
    gds2_resolve_skew_arrays: bool
    r"""
    Getter:
    @brief Gets a value indicating whether to resolve skew arrays into single instances
    See \gds2_resolve_skew_arrays= method for a description of this property.
    This property has been added in version 0.27.1.

    Setter:
    @brief Resolves skew arrays into single instances

    Setting this property to true will make skew (non-orthogonal) arrays being resolved into single instances.
    Skew arrays happen if either the row or column vector isn't parallel to x or y axis. Such arrays can cause problems with some legacy software and can be disabled with this option.

    This property has been added in version 0.27.1.
    """
    gds2_user_units: float
    r"""
    Getter:
    @brief Get the user units
    See \gds2_user_units= method for a description of the user units.
    This property has been added in version 0.18.

    Setter:
    @brief Set the users units to write into the GDS file

    The user units of a GDS file are rarely used and usually are set to 1 (micron).
    The intention of the user units is to specify the display units. KLayout ignores the user unit and uses microns as the display unit.
    The user unit must be larger than zero.

    This property has been added in version 0.18.
    """
    gds2_write_cell_properties: bool
    r"""
    Getter:
    @brief Gets a value indicating whether cell properties are written

    This property has been added in version 0.23.

    Setter:
    @brief Enables writing of cell properties if set to true

    If this property is set to true, cell properties will be written as PROPATTR/PROPVALUE records immediately following the BGNSTR records. This is a non-standard extension and is therefore disabled by default.


    This property has been added in version 0.23.
    """
    gds2_write_file_properties: bool
    r"""
    Getter:
    @brief Gets a value indicating whether layout properties are written

    This property has been added in version 0.24.

    Setter:
    @brief Enables writing of file properties if set to true

    If this property is set to true, layout properties will be written as PROPATTR/PROPVALUE records immediately following the BGNLIB records. This is a non-standard extension and is therefore disabled by default.


    This property has been added in version 0.24.
    """
    gds2_write_timestamps: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the current time is written into the GDS2 timestamp fields

    This property has been added in version 0.21.16.

    Setter:
    @brief Writes the current time into the GDS2 timestamps if set to true

    If this property is set to false, the time fields will all be zero. This somewhat simplifies compare and diff applications.


    This property has been added in version 0.21.16.
    """
    keep_instances: bool
    r"""
    Getter:
    @brief Gets a flag indicating whether instances will be kept even if the target cell is dropped

    See \keep_instances= for details about this flag.

    This method was introduced in version 0.23.

    Setter:
    @brief Enables or disables instances for dropped cells

    If this flag is set to true, instances for cells will be written, even if the cell is dropped. That may happen, if cells are selected with \select_this_cell or \add_this_cell or \no_empty_cells is used. Even if cells called by such cells are not selected, instances will be written for that cell if "keep_instances" is true. That feature is supported by the GDS format currently and results in "ghost cells" which have instances but no cell definition.

    The default value is false (instances of dropped cells are not written).

    This method was introduced in version 0.23.
    """
    mag_lambda: float
    r"""
    Getter:
    @brief Gets the lambda value
    See \mag_lambda= method for a description of this attribute.
    This property has been added in version 0.26.2.

    Setter:
    @brief Specifies the lambda value to used for writing

    The lambda value is the basic unit of the layout.
    The layout is brought to units of this value. If the layout is not on-grid on this unit, snapping will happen. If the value is less or equal to zero, KLayout will use the lambda value stored inside the layout set by a previous read operation of a MAGIC file. The lambda value is stored in the Layout object as the "lambda" metadata attribute.

    This property has been added in version 0.26.2.
    """
    mag_tech: str
    r"""
    Getter:
    @brief Gets the technology string used for writing
    See \mag_tech= method for a description of this attribute.
    This property has been added in version 0.26.2.

    Setter:
    @brief Specifies the technology string used for writing

    If this string is empty, the writer will try to obtain the technology from the "technology" metadata attribute of the layout.

    This property has been added in version 0.26.2.
    """
    mag_write_timestamp: bool
    r"""
    Getter:
    @brief Gets a value indicating whether to write a timestamp
    See \write_timestamp= method for a description of this attribute.

    This property has been added in version 0.26.2.

    Setter:
    @brief Specifies whether to write a timestamp

    If this attribute is set to false, the timestamp written is 0. This is not permitted in the strict sense, but simplifies comparison of Magic files.

    This property has been added in version 0.26.2.
    """
    no_empty_cells: bool
    r"""
    Getter:
    @brief Returns a flag indicating whether empty cells are not written.

    Setter:
    @brief Don't write empty cells if this flag is set

    By default, all cells are written (no_empty_cells is false).
    This applies to empty cells which do not contain shapes for the specified layers as well as cells which are empty because they reference empty cells only.
    """
    oasis_compression_level: int
    r"""
    Getter:
    @brief Get the OASIS compression level
    See \oasis_compression_level= method for a description of the OASIS compression level.
    Setter:
    @brief Set the OASIS compression level
    The OASIS compression level is an integer number between 0 and 10. 0 basically is no compression, 1 produces shape arrays in a simple fashion. 2 and higher compression levels will use a more elaborate algorithm to find shape arrays which uses 2nd and further neighbor distances. The higher the level, the higher the memory requirements and run times.
    """
    oasis_permissive: bool
    r"""
    Getter:
    @brief Gets the OASIS permissive mode
    See \oasis_permissive= method for a description of this predicate.
    This method has been introduced in version 0.25.1.
    Setter:
    @brief Sets OASIS permissive mode
    If this flag is true, certain shapes which cannot be written to OASIS are reported as warnings, not as errors. For example, paths with odd width (are rounded) or polygons with less than three points (are skipped).

    This method has been introduced in version 0.25.1.
    """
    oasis_recompress: bool
    r"""
    Getter:
    @brief Gets the OASIS recompression mode
    See \oasis_recompress= method for a description of this predicate.
    This method has been introduced in version 0.23.
    Setter:
    @brief Sets OASIS recompression mode
    If this flag is true, shape arrays already existing will be resolved and compression is applied to the individual shapes again. If this flag is false (the default), shape arrays already existing will be written as such.

    This method has been introduced in version 0.23.
    """
    oasis_strict_mode: bool
    r"""
    Getter:
    @brief Gets a value indicating whether to write strict-mode OASIS files

    Setter:
    @brief Sets a value indicating whether to write strict-mode OASIS files
    Setting this property clears all format specific options for other formats such as GDS.
    """
    oasis_substitution_char: str
    r"""
    Getter:
    @brief Gets the substitution character

    See \oasis_substitution_char for details. This attribute has been introduced in version 0.23.

    Setter:
    @brief Sets the substitution character for a-strings and n-strings
    The substitution character is used in place of invalid characters. The value of this attribute is a string which is either empty or a single character. If the string is empty, no substitution is made at the risk of producing invalid OASIS files.

    This attribute has been introduce in version 0.23.
    """
    oasis_write_cblocks: bool
    r"""
    Getter:
    @brief Gets a value indicating whether to write compressed CBLOCKS per cell

    Setter:
    @brief Sets a value indicating whether to write compressed CBLOCKS per cell
    Setting this property clears all format specific options for other formats such as GDS.
    """
    oasis_write_cell_bounding_boxes: bool
    r"""
    Getter:
    @brief Gets a value indicating whether cell bounding boxes are written
    See \oasis_write_cell_bounding_boxes= method for a description of this flag.
    This method has been introduced in version 0.24.3.
    Setter:
    @brief Sets a value indicating whether cell bounding boxes are written
    If this value is set to true, cell bounding boxes are written (S_BOUNDING_BOX). The S_BOUNDING_BOX properties will be attached to the CELLNAME records.

    Setting this value to true will also enable writing of other standard properties like S_TOP_CELL (see \oasis_write_std_properties=).
    By default, cell bounding boxes are not written, but standard properties are.

    This method has been introduced in version 0.24.3.
    """
    oasis_write_std_properties: bool
    r"""
    Getter:
    @brief Gets a value indicating whether standard properties will be written
    See \oasis_write_std_properties= method for a description of this flag.
    This method has been introduced in version 0.24.
    Setter:
    @brief Sets a value indicating whether standard properties will be written
    If this value is false, no standard properties are written. If true, S_TOP_CELL and some other global standard properties are written. In addition, \oasis_write_cell_bounding_boxes= can be used to write cell bounding boxes using S_BOUNDING_BOX.

    By default, this flag is true and standard properties are written.

    Setting this property to false clears the oasis_write_cell_bounding_boxes flag too.

    This method has been introduced in version 0.24.
    """
    oasis_write_std_properties_ext: int
    r"""
    Getter:
    @hide
    Setter:
    @hide
    """
    scale_factor: float
    r"""
    Getter:
    @brief Gets the scaling factor currently set

    Setter:
    @brief Set the scaling factor for the saving 

    Using a scaling factor will scale all objects accordingly. This scale factor adds to a potential scaling implied by using an explicit database unit.

    Be aware that rounding effects may occur if fractional scaling factors are used.

    By default, no scaling is applied.
    """
    write_context_info: bool
    r"""
    Getter:
    @brief Gets a flag indicating whether context information will be stored

    See \write_context_info= for details about this flag.

    This method was introduced in version 0.23.

    Setter:
    @brief Enables or disables context information

    If this flag is set to false, no context information for PCell or library cell instances is written. Those cells will be converted to plain cells and KLayout will not be able to restore the identity of those cells. Use this option to enforce compatibility with other tools that don't understand the context information of KLayout.

    The default value is true (context information is stored). Not all formats support context information, hence that flag has no effect for formats like CIF or DXF.

    This method was introduced in version 0.23.
    """
    @classmethod
    def new(cls) -> SaveLayoutOptions:
        r"""
        @brief Default constructor

        This will initialize the scale factor to 1.0, the database unit is set to
        "same as original" and all layers are selected as well as all cells.
        The default format is GDS2.
        """
    def __copy__(self) -> SaveLayoutOptions:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> SaveLayoutOptions:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self) -> None:
        r"""
        @brief Default constructor

        This will initialize the scale factor to 1.0, the database unit is set to
        "same as original" and all layers are selected as well as all cells.
        The default format is GDS2.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def add_cell(self, cell_index: int) -> None:
        r"""
        @brief Add a cell (plus hierarchy) to be saved


        The index of the cell must be a valid index in the context of the layout that will be saved.
        This method clears the 'select all cells' flag.

        This method also implicitly adds the children of that cell. A method that does not add the children in \add_this_cell.
        """
    def add_layer(self, layer_index: int, properties: LayerInfo) -> None:
        r"""
        @brief Add a layer to be saved 


        Adds the layer with the given index to the layer list that will be written.
        If all layers have been selected previously, all layers will 
        be unselected first and only the new layer remains.

        The 'properties' argument can be used to assign different layer properties than the ones
        present in the layout. Pass a default \LayerInfo object to this argument to use the
        properties from the layout object. Construct a valid \LayerInfo object with explicit layer,
        datatype and possibly a name to override the properties stored in the layout.
        """
    def add_this_cell(self, cell_index: int) -> None:
        r"""
        @brief Adds a cell to be saved


        The index of the cell must be a valid index in the context of the layout that will be saved.
        This method clears the 'select all cells' flag.
        Unlike \add_cell, this method does not implicitly add all children of that cell.

        This method has been added in version 0.23.
        """
    def assign(self, other: SaveLayoutOptions) -> None:
        r"""
        @brief Assigns another object to self
        """
    def clear_cells(self) -> None:
        r"""
        @brief Clears all cells to be saved

        This method can be used to ensure that no cell is selected before \add_cell is called to specify a cell.
        This method clears the 'select all cells' flag.

        This method has been added in version 0.22.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def deselect_all_layers(self) -> None:
        r"""
        @brief Unselect all layers: no layer will be saved

        This method will clear all layers selected with \add_layer so far and clear the 'select all layers' flag.
        Using this method is the only way to save a layout without any layers.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> SaveLayoutOptions:
        r"""
        @brief Creates a copy of self
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def select_all_cells(self) -> None:
        r"""
        @brief Select all cells to save

        This method will clear all cells specified with \add_cells so far and set the 'select all cells' flag.
        This is the default.
        """
    def select_all_layers(self) -> None:
        r"""
        @brief Select all layers to be saved

        This method will clear all layers selected with \add_layer so far and set the 'select all layers' flag.
        This is the default.
        """
    def select_cell(self, cell_index: int) -> None:
        r"""
        @brief Selects a cell to be saved (plus hierarchy below)


        This method is basically a convenience method that combines \clear_cells and \add_cell.
        This method clears the 'select all cells' flag.

        This method has been added in version 0.22.
        """
    def select_this_cell(self, cell_index: int) -> None:
        r"""
        @brief Selects a cell to be saved


        This method is basically a convenience method that combines \clear_cells and \add_this_cell.
        This method clears the 'select all cells' flag.

        This method has been added in version 0.23.
        """
    def set_format_from_filename(self, filename: str) -> bool:
        r"""
        @brief Select a format from the given file name

        This method will set the format according to the file's extension.

        This method has been introduced in version 0.22. Beginning with version 0.23, this method always returns true, since the only consumer for the return value, Layout#write, now ignores that parameter and automatically determines the compression mode from the file name.
        """

class LayoutQueryIterator:
    r"""
    @brief Provides the results of the query

    This object is used by \LayoutQuery#each to deliver the results of a query in an iterative fashion. See \LayoutQuery for a detailed description of the query interface.

    The LayoutQueryIterator class has been introduced in version 0.25.
    """
    @classmethod
    def new(cls) -> LayoutQueryIterator:
        r"""
        @brief Creates a new object of this class
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def cell(self) -> Any:
        r"""
        @brief A shortcut for 'get("cell")'
        """
    def cell_index(self) -> Any:
        r"""
        @brief A shortcut for 'get("cell_index")'
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def data(self) -> Any:
        r"""
        @brief A shortcut for 'get("data")'
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dtrans(self) -> Any:
        r"""
        @brief A shortcut for 'get("dtrans")'
        """
    def get(self, name: str) -> Any:
        r"""
        @brief Gets the query property with the given name
        The query properties available can be obtained from the query object using \LayoutQuery#property_names.
        Some shortcut methods are available. For example, the \data method provides a shortcut for 'get("data")'.

        If a property with the given name is not available, nil will be returned.
        """
    def initial_cell(self) -> Any:
        r"""
        @brief A shortcut for 'get("initial_cell")'
        """
    def initial_cell_index(self) -> Any:
        r"""
        @brief A shortcut for 'get("initial_cell_index")'
        """
    def inst(self) -> Any:
        r"""
        @brief A shortcut for 'get("inst")'
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def layer_index(self) -> Any:
        r"""
        @brief A shortcut for 'get("layer_index")'
        """
    def layout(self) -> Layout:
        r"""
        @brief Gets the layout the query acts on
        """
    def parent_cell(self) -> Any:
        r"""
        @brief A shortcut for 'get("parent_cell")'
        """
    def parent_cell_index(self) -> Any:
        r"""
        @brief A shortcut for 'get("parent_cell_index")'
        """
    def path_dtrans(self) -> Any:
        r"""
        @brief A shortcut for 'get("path_dtrans")'
        """
    def path_trans(self) -> Any:
        r"""
        @brief A shortcut for 'get("path_trans")'
        """
    def query(self) -> LayoutQuery:
        r"""
        @brief Gets the query the iterator follows on
        """
    def shape(self) -> Any:
        r"""
        @brief A shortcut for 'get("shape")'
        """
    def trans(self) -> Any:
        r"""
        @brief A shortcut for 'get("trans")'
        """

class LayoutQuery:
    r"""
    @brief A layout query
    Layout queries are the backbone of the "Search & replace" feature. Layout queries allow retrieval of data from layouts and manipulation of layouts. This object provides script binding for this feature.
    Layout queries are used by first creating a query object. Depending on the nature of the query, either \execute or \each can be used to execute the query. \execute will run the query and return once the query is finished. \execute is useful for running queries that don't return results such as "delete" or "with ... do" queries.
    \each can be used when the results of the query need to be retrieved.

    The \each method will call a block a of code for every result available. It will provide a \LayoutQueryIterator object that allows accessing the results of the query. Depending on the query, different attributes of the iterator object will be available. For example, "select" queries will fill the "data" attribute with an array of values corresponding to the columns of the selection.

    Here is some sample code:
    @code
    ly = RBA::CellView::active.layout
    q = RBA::LayoutQuery::new("select cell.name, cell.bbox from *")
    q.each(ly) do |iter|
      puts "cell name: #{iter.data[0]}, bounding box: #{iter.data[1]}"
    end
    @/code

    The LayoutQuery class has been introduced in version 0.25.
    """
    @classmethod
    def new(cls, query: str) -> LayoutQuery:
        r"""
        @brief Creates a new query object from the given query string
        """
    def __init__(self, query: str) -> None:
        r"""
        @brief Creates a new query object from the given query string
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def each(self, layout: Layout, context: Optional[tl.ExpressionContext] = ...) -> Iterator[LayoutQueryIterator]:
        r"""
        @brief Executes the query and delivered the results iteratively.
        The argument to the block is a \LayoutQueryIterator object which can be asked for specific results.

        The context argument allows supplying an expression execution context. This context can be used for example to supply variables for the execution. It has been added in version 0.26.
        """
    def execute(self, layout: Layout, context: Optional[tl.ExpressionContext] = ...) -> None:
        r"""
        @brief Executes the query

        This method can be used to execute "active" queries such
        as "delete" or "with ... do".
        It is basically equivalent to iterating over the query until it is
        done.

        The context argument allows supplying an expression execution context. This context can be used for example to supply variables for the execution. It has been added in version 0.26.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def property_names(self) -> List[str]:
        r"""
        @brief Gets a list of property names available.
        The list of properties available from the query depends on the nature of the query. This method allows detection of the properties available. Within the query, all of these properties can be obtained from the query iterator using \LayoutQueryIterator#get.
        """

class Library:
    r"""
    @brief A Library 

    A library is basically a wrapper around a layout object. The layout object
    provides cells and potentially PCells that can be imported into other layouts.

    The library provides a name which is used to identify the library and a description
    which is used for identifying the library in a user interface. 

    After a library is created and the layout is filled, it must be registered using the register method.

    This class has been introduced in version 0.22.
    """
    description: str
    r"""
    Getter:
    @brief Returns the libraries' description text

    Setter:
    @brief Sets the libraries' description text
    """
    technology: str
    r"""
    Getter:
    @brief Returns name of the technology the library is associated with
    If this attribute is a non-empty string, this library is only offered for selection if the current layout uses this technology.

    This attribute has been introduced in version 0.25. In version 0.27 this attribute is deprecated as a library can now be associated with multiple technologies.
    Setter:
    @brief sets the name of the technology the library is associated with

    See \technology for details. This attribute has been introduced in version 0.25. In version 0.27, a library can be associated with multiple technologies and this method will revert the selection to a single one. Passing an empty string is equivalent to \clear_technologies.
    """
    @classmethod
    def library_by_id(cls, id: int) -> Library:
        r"""
        @brief Gets the library object for the given ID
        If the ID is not valid, nil is returned.

        This method has been introduced in version 0.27.
        """
    @classmethod
    def library_by_name(cls, name: str, for_technology: Optional[str] = ...) -> Library:
        r"""
        @brief Gets a library by name
        Returns the library object for the given name. If the name is not a valid
        library name, nil is returned.

        Different libraries can be registered under the same names for different technologies. When a technology name is given in 'for_technologies', the first library matching this technology is returned. If no technology is given, the first library is returned.

        The technology selector has been introduced in version 0.27.
        """
    @classmethod
    def library_ids(cls) -> List[int]:
        r"""
        @brief Returns a list of valid library IDs.
        See \library_names for the reasoning behind this method.
        This method has been introduced in version 0.27.
        """
    @classmethod
    def library_names(cls) -> List[str]:
        r"""
        @brief Returns a list of the names of all libraries registered in the system.

        NOTE: starting with version 0.27, the name of a library does not need to be unique if libraries are associated with specific technologies. This method will only return the names and it's not possible not unambiguously derive the library object. It is recommended to use \library_ids and \library_by_id to obtain the library unambiguously.
        """
    @classmethod
    def new(cls) -> Library:
        r"""
        @brief Creates a new, empty library
        """
    def __copy__(self) -> Library:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> Library:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new, empty library
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def add_technology(self, tech: str) -> None:
        r"""
        @brief Additionally associates the library with the given technology.
        See also \clear_technologies.

        This method has been introduced in version 0.27
        """
    def assign(self, other: Library) -> None:
        r"""
        @brief Assigns another object to self
        """
    def clear_technologies(self) -> None:
        r"""
        @brief Clears the list of technologies the library is associated with.
        See also \add_technology.

        This method has been introduced in version 0.27
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def delete(self) -> None:
        r"""
        @brief Deletes the library

        This method will delete the library object. Library proxies pointing to this library will become invalid and the library object cannot be used any more after calling this method.

        This method has been introduced in version 0.25.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> Library:
        r"""
        @brief Creates a copy of self
        """
    def for_technologies(self) -> bool:
        r"""
        @brief Returns a value indicating whether the library is associated with any technology.
        The method is equivalent to checking whether the \technologies list is empty.

        This method has been introduced in version 0.27
        """
    def id(self) -> int:
        r"""
        @brief Returns the library's ID
        The ID is set when the library is registered and cannot be changed 
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_for_technology(self, tech: str) -> bool:
        r"""
        @brief Returns a value indicating whether the library is associated with the given technology.
        This method has been introduced in version 0.27
        """
    def layout(self) -> Layout:
        r"""
        @brief The layout object where the cells reside that this library defines
        """
    def layout_const(self) -> Layout:
        r"""
        @brief The layout object where the cells reside that this library defines (const version)
        """
    def name(self) -> str:
        r"""
        @brief Returns the libraries' name
        The name is set when the library is registered and cannot be changed
        """
    def refresh(self) -> None:
        r"""
        @brief Updates all layouts using this library.
        This method will retire cells or update layouts in the attached clients.
        It will also recompute the PCells inside the library. 
        This method has been introduced in version 0.27.8.
        """
    def register(self, name: str) -> None:
        r"""
        @brief Registers the library with the given name

        This method can be called in the constructor to register the library after 
        the layout object has been filled with content. If a library with that name
        already exists for the same technologies, it will be replaced with this library. 

        This method will set the libraries' name.

        The technology specific behaviour has been introduced in version 0.27.
        """
    def technologies(self) -> List[str]:
        r"""
        @brief Gets the list of technologies this library is associated with.
        This method has been introduced in version 0.27
        """

class PCellDeclaration_Native:
    r"""
    @hide
    @alias PCellDeclaration
    """
    @classmethod
    def new(cls) -> PCellDeclaration_Native:
        r"""
        @brief Creates a new object of this class
        """
    def __copy__(self) -> PCellDeclaration_Native:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> PCellDeclaration_Native:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: PCellDeclaration_Native) -> None:
        r"""
        @brief Assigns another object to self
        """
    def callback(self, layout: Layout, name: str, states: PCellParameterStates) -> None:
        r"""
        """
    def can_create_from_shape(self, layout: Layout, shape: Shape, layer: int) -> bool:
        r"""
        """
    def coerce_parameters(self, layout: Layout, parameters: Sequence[Any]) -> List[Any]:
        r"""
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def display_text(self, parameters: Sequence[Any]) -> str:
        r"""
        """
    def dup(self) -> PCellDeclaration_Native:
        r"""
        @brief Creates a copy of self
        """
    def get_layers(self, parameters: Sequence[Any]) -> List[LayerInfo]:
        r"""
        """
    def get_parameters(self) -> List[PCellParameterDeclaration]:
        r"""
        """
    def id(self) -> int:
        r"""
        @brief Gets the integer ID of the PCell declaration
        This ID is used to identify the PCell in the context of a Layout object for example
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def layout(self) -> Layout:
        r"""
        @brief Gets the Layout object the PCell is registered in or nil if it is not registered yet.
        This attribute has been added in version 0.27.5.
        """
    def name(self) -> str:
        r"""
        @brief Gets the name of the PCell
        """
    def parameters_from_shape(self, layout: Layout, shape: Shape, layer: int) -> List[Any]:
        r"""
        """
    def produce(self, layout: Layout, layers: Sequence[int], parameters: Sequence[Any], cell: Cell) -> None:
        r"""
        """
    def transformation_from_shape(self, layout: Layout, shape: Shape, layer: int) -> Trans:
        r"""
        """
    def wants_lazy_evaluation(self) -> bool:
        r"""
        """

class PCellParameterState:
    r"""
    @brief Provides access to the attributes of a single parameter within \PCellParameterStates.

    See \PCellParameterStates for details about this feature.

    This class has been introduced in version 0.28.
    """
    class ParameterStateIcon:
        r"""
        @brief This enum specifies the icon shown next to the parameter in PCell parameter list.

        This enum was introduced in version 0.28.
        """
        ErrorIcon: ClassVar[PCellParameterState.ParameterStateIcon]
        r"""
        @brief An icon indicating an error is shown
        """
        InfoIcon: ClassVar[PCellParameterState.ParameterStateIcon]
        r"""
        @brief A general 'information' icon is shown
        """
        NoIcon: ClassVar[PCellParameterState.ParameterStateIcon]
        r"""
        @brief No icon is shown for the parameter
        """
        WarningIcon: ClassVar[PCellParameterState.ParameterStateIcon]
        r"""
        @brief An icon indicating a warning is shown
        """
        @overload
        @classmethod
        def new(cls, i: int) -> PCellParameterState.ParameterStateIcon:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        @classmethod
        def new(cls, s: str) -> PCellParameterState.ParameterStateIcon:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer value
            """
        @overload
        def __eq__(self, other: object) -> bool:
            r"""
            @brief Compares two enums
            """
        @overload
        def __init__(self, i: int) -> None:
            r"""
            @brief Creates an enum from an integer value
            """
        @overload
        def __init__(self, s: str) -> None:
            r"""
            @brief Creates an enum from a string value
            """
        @overload
        def __lt__(self, other: int) -> bool:
            r"""
            @brief Returns true if the enum is less (in the enum symbol order) than the integer value
            """
        @overload
        def __lt__(self, other: PCellParameterState.ParameterStateIcon) -> bool:
            r"""
            @brief Returns true if the first enum is less (in the enum symbol order) than the second
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares two enums for inequality
            """
        @overload
        def __ne__(self, other: object) -> bool:
            r"""
            @brief Compares an enum with an integer for inequality
            """
        def __repr__(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def __str__(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
        def inspect(self) -> str:
            r"""
            @brief Converts an enum to a visual string
            """
        def to_i(self) -> int:
            r"""
            @brief Gets the integer value from the enum
            """
        def to_s(self) -> str:
            r"""
            @brief Gets the symbolic string from an enum
            """
    ErrorIcon: ClassVar[PCellParameterState.ParameterStateIcon]
    r"""
    @brief An icon indicating an error is shown
    """
    InfoIcon: ClassVar[PCellParameterState.ParameterStateIcon]
    r"""
    @brief A general 'information' icon is shown
    """
    NoIcon: ClassVar[PCellParameterState.ParameterStateIcon]
    r"""
    @brief No icon is shown for the parameter
    """
    WarningIcon: ClassVar[PCellParameterState.ParameterStateIcon]
    r"""
    @brief An icon indicating a warning is shown
    """
    enabled: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the parameter is enabled in the parameter form

    Setter:
    @brief Sets a value indicating whether the parameter is enabled in the parameter form
    """
    @property
    def icon(self) -> None:
        r"""
        WARNING: This variable can only be set, not retrieved.
        @brief Sets the icon for the parameter
        """
    readonly: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the parameter is read-only (not editable) in the parameter form

    Setter:
    @brief Sets a value indicating whether the parameter is made read-only (not editable) in the parameter form
    """
    tooltip: str
    r"""
    Getter:
    @brief Gets the tool tip text

    Setter:
    @brief Sets the tool tip text

    The tool tip is shown when hovering over the parameter label or edit field.
    """
    value: Any
    r"""
    Getter:
    @brief Gets the value of the parameter

    Setter:
    @brief Sets the value of the parameter
    """
    visible: bool
    r"""
    Getter:
    @brief Gets a value indicating whether the parameter is visible in the parameter form

    Setter:
    @brief Sets a value indicating whether the parameter is visible in the parameter form
    """
    @classmethod
    def new(cls) -> PCellParameterState:
        r"""
        @brief Creates a new object of this class
        """
    def __copy__(self) -> PCellParameterState:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> PCellParameterState:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: PCellParameterState) -> None:
        r"""
        @brief Assigns another object to self
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> PCellParameterState:
        r"""
        @brief Creates a copy of self
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_enabled(self) -> bool:
        r"""
        @brief Gets a value indicating whether the parameter is enabled in the parameter form
        """
    def is_readonly(self) -> bool:
        r"""
        @brief Gets a value indicating whether the parameter is read-only (not editable) in the parameter form
        """
    def is_visible(self) -> bool:
        r"""
        @brief Gets a value indicating whether the parameter is visible in the parameter form
        """

class PCellParameterStates:
    r"""
    @brief Provides access to the parameter states inside a 'callback' implementation of a PCell

    Example: enables or disables a parameter 'n' based on the value:

    @code
    n_param = states.parameter("n")
    n_param.enabled = n_param.value > 1.0
    @/code

    This class has been introduced in version 0.28.
    """
    @classmethod
    def new(cls) -> PCellParameterStates:
        r"""
        @brief Creates a new object of this class
        """
    def __copy__(self) -> PCellParameterStates:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> PCellParameterStates:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: PCellParameterStates) -> None:
        r"""
        @brief Assigns another object to self
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> PCellParameterStates:
        r"""
        @brief Creates a copy of self
        """
    def has_parameter(self, name: str) -> bool:
        r"""
        @brief Gets a value indicating whether a parameter with that name exists
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def parameter(self, name: str) -> PCellParameterState:
        r"""
        @brief Gets the parameter by name

        This will return a \PCellParameterState object that can be used to manipulate the parameter state.
        """

class PCellDeclaration(PCellDeclaration_Native):
    r"""
    @brief A PCell declaration providing the parameters and code to produce the PCell

    A PCell declaration is basically the recipe of how to create a PCell layout from
    a parameter set. The declaration includes

    @ul
    @li Parameters: names, types, default values @/li
    @li Layers: the layers the PCell wants to create @/li
    @li Code: a production callback that is called whenever a PCell is instantiated with a certain parameter set @/li
    @li Display name: the name that is shown for a given PCell instance @/li
    @/ul

    All these declarations are implemented by deriving from the PCellDeclaration class
    and reimplementing the specific methods. Reimplementing the \display_name method is 
    optional. The default implementation creates a name from the PCell name plus the 
    parameters.

    By supplying the information about the layers it wants to create, KLayout is able to
    call the production callback with a defined set of the layer ID's which are already
    mapped to valid actual layout layers.

    This class has been introduced in version 0.22.
    """
    def _assign(self, other: PCellDeclaration_Native) -> None:
        r"""
        @brief Assigns another object to self
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _dup(self) -> PCellDeclaration:
        r"""
        @brief Creates a copy of self
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def callback(self, arg0: Layout, arg1: str, arg2: PCellParameterStates) -> None:
        r"""
        @hide
        """
    def can_create_from_shape(self, arg0: Layout, arg1: Shape, arg2: int) -> bool:
        r"""
        @hide
        """
    def display_text(self, arg0: Sequence[Any]) -> str:
        r"""
        @hide
        """
    def get_parameters(self) -> List[PCellParameterDeclaration]:
        r"""
        @hide
        """
    def parameters_from_shape(self, arg0: Layout, arg1: Shape, arg2: int) -> List[Any]:
        r"""
        @hide
        """
    def produce(self, arg0: Layout, arg1: Sequence[int], arg2: Sequence[Any], arg3: Cell) -> None:
        r"""
        @hide
        """
    def transformation_from_shape(self, arg0: Layout, arg1: Shape, arg2: int) -> Trans:
        r"""
        @hide
        """
    def wants_lazy_evaluation(self) -> bool:
        r"""
        @hide
        """

class PCellParameterDeclaration:
    r"""
    @brief A PCell parameter declaration

    This class declares a PCell parameter by providing a name, the type and a value 
    and additional 
    information like description, unit string and default value. It is used in the \PCellDeclaration class to 
    deliver the necessary information.

    This class has been introduced in version 0.22.
    """
    TypeBoolean: ClassVar[int]
    r"""
    @brief Type code: boolean data
    """
    TypeCallback: ClassVar[int]
    r"""
    @brief Type code: a button triggering a callback

    This code has been introduced in version 0.28.
    """
    TypeDouble: ClassVar[int]
    r"""
    @brief Type code: floating-point data
    """
    TypeInt: ClassVar[int]
    r"""
    @brief Type code: integer data
    """
    TypeLayer: ClassVar[int]
    r"""
    @brief Type code: a layer (a \LayerInfo object)
    """
    TypeList: ClassVar[int]
    r"""
    @brief Type code: a list of variants
    """
    TypeNone: ClassVar[int]
    r"""
    @brief Type code: unspecific type
    """
    TypeShape: ClassVar[int]
    r"""
    @brief Type code: a guiding shape (Box, Edge, Point, Polygon or Path)
    """
    TypeString: ClassVar[int]
    r"""
    @brief Type code: string data
    """
    default: Any
    r"""
    Getter:
    @brief Gets the default value

    Setter:
    @brief Sets the default value
    If a default value is defined, it will be used to initialize the parameter value
    when a PCell is created.
    """
    description: str
    r"""
    Getter:
    @brief Gets the description text

    Setter:
    @brief Sets the description
    """
    hidden: bool
    r"""
    Getter:
    @brief Returns true, if the parameter is a hidden parameter that should not be shown in the user interface
    By making a parameter hidden, it is possible to create internal parameters which cannot be
    edited.

    Setter:
    @brief Makes the parameter hidden if this attribute is set to true
    """
    name: str
    r"""
    Getter:
    @brief Gets the name

    Setter:
    @brief Sets the name
    """
    readonly: bool
    r"""
    Getter:
    @brief Returns true, if the parameter is a read-only parameter
    By making a parameter read-only, it is shown but cannot be
    edited.

    Setter:
    @brief Makes the parameter read-only if this attribute is set to true
    """
    type: int
    r"""
    Getter:
    @brief Gets the type
    The type is one of the T... constants.
    Setter:
    @brief Sets the type
    """
    unit: str
    r"""
    Getter:
    @brief Gets the unit string

    Setter:
    @brief Sets the unit string
    The unit string is shown right to the edit fields for numeric parameters.
    """
    @overload
    @classmethod
    def new(cls, name: str, type: int, description: str) -> PCellParameterDeclaration:
        r"""
        @brief Create a new parameter declaration with the given name and type
        @param name The parameter name
        @param type One of the Type... constants describing the type of the parameter
        @param description The description text
        """
    @overload
    @classmethod
    def new(cls, name: str, type: int, description: str, default: Any) -> PCellParameterDeclaration:
        r"""
        @brief Create a new parameter declaration with the given name, type and default value
        @param name The parameter name
        @param type One of the Type... constants describing the type of the parameter
        @param description The description text
        @param default The default (initial) value
        """
    @overload
    @classmethod
    def new(cls, name: str, type: int, description: str, default: Any, unit: str) -> PCellParameterDeclaration:
        r"""
        @brief Create a new parameter declaration with the given name, type, default value and unit string
        @param name The parameter name
        @param type One of the Type... constants describing the type of the parameter
        @param description The description text
        @param default The default (initial) value
        @param unit The unit string
        """
    def __copy__(self) -> PCellParameterDeclaration:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> PCellParameterDeclaration:
        r"""
        @brief Creates a copy of self
        """
    @overload
    def __init__(self, name: str, type: int, description: str) -> None:
        r"""
        @brief Create a new parameter declaration with the given name and type
        @param name The parameter name
        @param type One of the Type... constants describing the type of the parameter
        @param description The description text
        """
    @overload
    def __init__(self, name: str, type: int, description: str, default: Any) -> None:
        r"""
        @brief Create a new parameter declaration with the given name, type and default value
        @param name The parameter name
        @param type One of the Type... constants describing the type of the parameter
        @param description The description text
        @param default The default (initial) value
        """
    @overload
    def __init__(self, name: str, type: int, description: str, default: Any, unit: str) -> None:
        r"""
        @brief Create a new parameter declaration with the given name, type, default value and unit string
        @param name The parameter name
        @param type One of the Type... constants describing the type of the parameter
        @param description The description text
        @param default The default (initial) value
        @param unit The unit string
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def add_choice(self, description: str, value: Any) -> None:
        r"""
        @brief Add a new value to the list of choices
        This method will add the given value with the given description to the list of
        choices. If choices are defined, KLayout will show a drop-down box instead of an
        entry field in the parameter user interface.
        """
    def assign(self, other: PCellParameterDeclaration) -> None:
        r"""
        @brief Assigns another object to self
        """
    def choice_descriptions(self) -> List[str]:
        r"""
        @brief Returns a list of choice descriptions
        """
    def choice_values(self) -> List[Any]:
        r"""
        @brief Returns a list of choice values
        """
    def clear_choices(self) -> None:
        r"""
        @brief Clears the list of choices
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> PCellParameterDeclaration:
        r"""
        @brief Creates a copy of self
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """

class Manager:
    r"""
    @brief A transaction manager class

    Manager objects control layout and potentially other objects in the layout database and queue operations to form transactions. A transaction is a sequence of operations that can be undone or redone.

    In order to equip a layout object with undo/redo support, instantiate the layout object with a manager attached and embrace the operations to undo/redo with transaction/commit calls.

    The use of transactions is subject to certain constraints, i.e. transacted sequences may not be mixed with non-transacted ones.

    This class has been introduced in version 0.19.
    """
    @classmethod
    def new(cls) -> Manager:
        r"""
        @brief Creates a new object of this class
        """
    def __copy__(self) -> Manager:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> Manager:
        r"""
        @brief Creates a copy of self
        """
    def __init__(self) -> None:
        r"""
        @brief Creates a new object of this class
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def assign(self, other: Manager) -> None:
        r"""
        @brief Assigns another object to self
        """
    def commit(self) -> None:
        r"""
        @brief Close a transaction.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> Manager:
        r"""
        @brief Creates a copy of self
        """
    def has_redo(self) -> bool:
        r"""
        @brief Determine if a transaction is available for 'redo'

        @return True, if a transaction is available.
        """
    def has_undo(self) -> bool:
        r"""
        @brief Determine if a transaction is available for 'undo'

        @return True, if a transaction is available.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def redo(self) -> None:
        r"""
        @brief Redo the next available transaction

        The next transaction is redone with this method.
        The 'has_redo' method can be used to determine whether
        there are transactions to undo.
        """
    @overload
    def transaction(self, description: str) -> int:
        r"""
        @brief Begin a transaction


        This call will open a new transaction. A transaction consists
        of a set of operations issued with the 'queue' method.
        A transaction is closed with the 'commit' method.

        @param description The description for this transaction.

        @return The ID of the transaction (can be used to join other transactions with this one)
        """
    @overload
    def transaction(self, description: str, join_with: int) -> int:
        r"""
        @brief Begin a joined transaction


        This call will open a new transaction and join if with the previous transaction.
        The ID of the previous transaction must be equal to the ID given with 'join_with'.

        This overload was introduced in version 0.22.

        @param description The description for this transaction (ignored if joined).
        @param description The ID of the previous transaction.

        @return The ID of the new transaction (can be used to join more)
        """
    def transaction_for_redo(self) -> str:
        r"""
        @brief Return the description of the next transaction for 'redo'
        """
    def transaction_for_undo(self) -> str:
        r"""
        @brief Return the description of the next transaction for 'undo'
        """
    def undo(self) -> None:
        r"""
        @brief Undo the current transaction

        The current transaction is undone with this method.
        The 'has_undo' method can be used to determine whether
        there are transactions to undo.
        """

class Matrix2d:
    r"""
    @brief A 2d matrix object used mainly for representing rotation and shear transformations.

    This object represents a 2x2 matrix. This matrix is used to implement affine transformations in the 2d space mainly. It can be decomposed into basic transformations: mirroring, rotation and shear. In that case, the assumed execution order of the basic transformations is mirroring at the x axis, rotation, magnification and shear.

    The matrix is a generalization of the transformations and is of limited use in a layout database context. It is useful however to implement shear transformations on polygons, edges and polygon or edge collections.

    This class was introduced in version 0.22.
    """
    @overload
    @classmethod
    def new(cls) -> Matrix2d:
        r"""
        @brief Create a new Matrix2d representing a unit transformation
        """
    @overload
    @classmethod
    def new(cls, m: float) -> Matrix2d:
        r"""
        @brief Create a new Matrix2d representing an isotropic magnification
        @param m The magnification
        """
    @overload
    @classmethod
    def new(cls, t: DCplxTrans) -> Matrix2d:
        r"""
        @brief Create a new Matrix2d from the given complex transformation@param t The transformation from which to create the matrix (not taking into account the displacement)
        """
    @overload
    @classmethod
    def new(cls, mx: float, my: float) -> Matrix2d:
        r"""
        @brief Create a new Matrix2d representing an anisotropic magnification
        @param mx The magnification in x direction
        @param my The magnification in y direction
        """
    @overload
    @classmethod
    def new(cls, m11: float, m12: float, m21: float, m22: float) -> Matrix2d:
        r"""
        @brief Create a new Matrix2d from the four coefficients
        """
    @overload
    @classmethod
    def newc(cls, mag: float, rotation: float, mirror: bool) -> Matrix2d:
        r"""
        @brief Create a new Matrix2d representing an isotropic magnification, rotation and mirroring
        @param mag The magnification in x direction
        @param rotation The rotation angle (in degree)
        @param mirror The mirror flag (at x axis)

        This constructor is provided to construct a matrix similar to the complex transformation.
        This constructor is called 'newc' to distinguish it from the constructors taking matrix coefficients ('c' is for composite).
        The order of execution of the operations is mirror, magnification, rotation (as for complex transformations).
        """
    @overload
    @classmethod
    def newc(cls, shear: float, mx: float, my: float, rotation: float, mirror: bool) -> Matrix2d:
        r"""
        @brief Create a new Matrix2d representing a shear, anisotropic magnification, rotation and mirroring
        @param shear The shear angle
        @param mx The magnification in x direction
        @param my The magnification in y direction
        @param rotation The rotation angle (in degree)
        @param mirror The mirror flag (at x axis)

        The order of execution of the operations is mirror, magnification, shear and rotation.
        This constructor is called 'newc' to distinguish it from the constructor taking the four matrix coefficients ('c' is for composite).
        """
    def __add__(self, m: Matrix2d) -> Matrix2d:
        r"""
        @brief Sum of two matrices.
        @param m The other matrix.
        @return The (element-wise) sum of self+m
        """
    def __copy__(self) -> Matrix2d:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> Matrix2d:
        r"""
        @brief Creates a copy of self
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Create a new Matrix2d representing a unit transformation
        """
    @overload
    def __init__(self, m: float) -> None:
        r"""
        @brief Create a new Matrix2d representing an isotropic magnification
        @param m The magnification
        """
    @overload
    def __init__(self, t: DCplxTrans) -> None:
        r"""
        @brief Create a new Matrix2d from the given complex transformation@param t The transformation from which to create the matrix (not taking into account the displacement)
        """
    @overload
    def __init__(self, mx: float, my: float) -> None:
        r"""
        @brief Create a new Matrix2d representing an anisotropic magnification
        @param mx The magnification in x direction
        @param my The magnification in y direction
        """
    @overload
    def __init__(self, m11: float, m12: float, m21: float, m22: float) -> None:
        r"""
        @brief Create a new Matrix2d from the four coefficients
        """
    @overload
    def __mul__(self, box: DBox) -> DBox:
        r"""
        @brief Transforms a box with this matrix.
        @param box The box to transform.
        @return The transformed box

        Please note that the box remains a box, even though the matrix supports shear and rotation. The returned box will be the bounding box of the sheared and rotated rectangle.
        """
    @overload
    def __mul__(self, e: DEdge) -> DEdge:
        r"""
        @brief Transforms an edge with this matrix.
        @param e The edge to transform.
        @return The transformed edge
        """
    @overload
    def __mul__(self, m: Matrix2d) -> Matrix2d:
        r"""
        @brief Product of two matrices.
        @param m The other matrix.
        @return The matrix product self*m
        """
    @overload
    def __mul__(self, p: DPoint) -> DPoint:
        r"""
        @brief Transforms a point with this matrix.
        @param p The point to transform.
        @return The transformed point
        """
    @overload
    def __mul__(self, p: DPolygon) -> DPolygon:
        r"""
        @brief Transforms a polygon with this matrix.
        @param p The polygon to transform.
        @return The transformed polygon
        """
    @overload
    def __mul__(self, p: DSimplePolygon) -> DSimplePolygon:
        r"""
        @brief Transforms a simple polygon with this matrix.
        @param p The simple polygon to transform.
        @return The transformed simple polygon
        """
    @overload
    def __mul__(self, v: DVector) -> DVector:
        r"""
        @brief Transforms a vector with this matrix.
        @param v The vector to transform.
        @return The transformed vector
        """
    @overload
    def __rmul__(self, box: DBox) -> DBox:
        r"""
        @brief Transforms a box with this matrix.
        @param box The box to transform.
        @return The transformed box

        Please note that the box remains a box, even though the matrix supports shear and rotation. The returned box will be the bounding box of the sheared and rotated rectangle.
        """
    @overload
    def __rmul__(self, e: DEdge) -> DEdge:
        r"""
        @brief Transforms an edge with this matrix.
        @param e The edge to transform.
        @return The transformed edge
        """
    @overload
    def __rmul__(self, m: Matrix2d) -> Matrix2d:
        r"""
        @brief Product of two matrices.
        @param m The other matrix.
        @return The matrix product self*m
        """
    @overload
    def __rmul__(self, p: DPoint) -> DPoint:
        r"""
        @brief Transforms a point with this matrix.
        @param p The point to transform.
        @return The transformed point
        """
    @overload
    def __rmul__(self, p: DPolygon) -> DPolygon:
        r"""
        @brief Transforms a polygon with this matrix.
        @param p The polygon to transform.
        @return The transformed polygon
        """
    @overload
    def __rmul__(self, p: DSimplePolygon) -> DSimplePolygon:
        r"""
        @brief Transforms a simple polygon with this matrix.
        @param p The simple polygon to transform.
        @return The transformed simple polygon
        """
    @overload
    def __rmul__(self, v: DVector) -> DVector:
        r"""
        @brief Transforms a vector with this matrix.
        @param v The vector to transform.
        @return The transformed vector
        """
    def __str__(self) -> str:
        r"""
        @brief Convert the matrix to a string.
        @return The string representing this matrix
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def angle(self) -> float:
        r"""
        @brief Returns the rotation angle of the rotation component of this matrix.
        @return The angle in degree.
        The matrix is decomposed into basic transformations assuming an execution order of mirroring at the x axis, rotation, magnification and shear.
        """
    def assign(self, other: Matrix2d) -> None:
        r"""
        @brief Assigns another object to self
        """
    def cplx_trans(self) -> DCplxTrans:
        r"""
        @brief Converts this matrix to a complex transformation (if possible).
        @return The complex transformation.
        This method is successful only if the matrix does not contain shear components and the magnification must be isotropic.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> Matrix2d:
        r"""
        @brief Creates a copy of self
        """
    def inverted(self) -> Matrix2d:
        r"""
        @brief The inverse of this matrix.
        @return The inverse of this matrix
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_mirror(self) -> bool:
        r"""
        @brief Returns the mirror flag of this matrix.
        @return True if this matrix has a mirror component.
        The matrix is decomposed into basic transformations assuming an execution order of mirroring at the x axis, rotation, magnification and shear.
        """
    def m(self, i: int, j: int) -> float:
        r"""
        @brief Gets the m coefficient with the given index.
        @return The coefficient [i,j]
        """
    def m11(self) -> float:
        r"""
        @brief Gets the m11 coefficient.
        @return The value of the m11 coefficient
        """
    def m12(self) -> float:
        r"""
        @brief Gets the m12 coefficient.
        @return The value of the m12 coefficient
        """
    def m21(self) -> float:
        r"""
        @brief Gets the m21 coefficient.
        @return The value of the m21 coefficient
        """
    def m22(self) -> float:
        r"""
        @brief Gets the m22 coefficient.
        @return The value of the m22 coefficient
        """
    def mag_x(self) -> float:
        r"""
        @brief Returns the x magnification of the magnification component of this matrix.
        @return The magnification factor.
        The matrix is decomposed into basic transformations assuming an execution order of mirroring at the x axis, magnification, shear and rotation.
        """
    def mag_y(self) -> float:
        r"""
        @brief Returns the y magnification of the magnification component of this matrix.
        @return The magnification factor.
        The matrix is decomposed into basic transformations assuming an execution order of mirroring at the x axis, magnification, shear and rotation.
        """
    def shear_angle(self) -> float:
        r"""
        @brief Returns the magnitude of the shear component of this matrix.
        @return The shear angle in degree.
        The matrix is decomposed into basic transformations assuming an execution order of mirroring at the x axis, rotation, magnification and shear.
        The shear basic transformation will tilt the x axis towards the y axis and vice versa. The shear angle gives the tilt angle of the axes towards the other one. The possible range for this angle is -45 to 45 degree.
        """
    def to_s(self) -> str:
        r"""
        @brief Convert the matrix to a string.
        @return The string representing this matrix
        """
    def trans(self, p: DPoint) -> DPoint:
        r"""
        @brief Transforms a point with this matrix.
        @param p The point to transform.
        @return The transformed point
        """

class IMatrix2d:
    r"""
    @brief A 2d matrix object used mainly for representing rotation and shear transformations (integer coordinate version).

    This object represents a 2x2 matrix. This matrix is used to implement affine transformations in the 2d space mainly. It can be decomposed into basic transformations: mirroring, rotation and shear. In that case, the assumed execution order of the basic transformations is mirroring at the x axis, rotation, magnification and shear.

    The integer variant was introduced in version 0.27.
    """
    @overload
    @classmethod
    def new(cls) -> IMatrix2d:
        r"""
        @brief Create a new Matrix2d representing a unit transformation
        """
    @overload
    @classmethod
    def new(cls, m: float) -> IMatrix2d:
        r"""
        @brief Create a new Matrix2d representing an isotropic magnification
        @param m The magnification
        """
    @overload
    @classmethod
    def new(cls, t: DCplxTrans) -> IMatrix2d:
        r"""
        @brief Create a new Matrix2d from the given complex transformation@param t The transformation from which to create the matrix (not taking into account the displacement)
        """
    @overload
    @classmethod
    def new(cls, mx: float, my: float) -> IMatrix2d:
        r"""
        @brief Create a new Matrix2d representing an anisotropic magnification
        @param mx The magnification in x direction
        @param my The magnification in y direction
        """
    @overload
    @classmethod
    def new(cls, m11: float, m12: float, m21: float, m22: float) -> IMatrix2d:
        r"""
        @brief Create a new Matrix2d from the four coefficients
        """
    @overload
    @classmethod
    def newc(cls, mag: float, rotation: float, mirror: bool) -> IMatrix2d:
        r"""
        @brief Create a new Matrix2d representing an isotropic magnification, rotation and mirroring
        @param mag The magnification in x direction
        @param rotation The rotation angle (in degree)
        @param mirror The mirror flag (at x axis)

        This constructor is provided to construct a matrix similar to the complex transformation.
        This constructor is called 'newc' to distinguish it from the constructors taking matrix coefficients ('c' is for composite).
        The order of execution of the operations is mirror, magnification, rotation (as for complex transformations).
        """
    @overload
    @classmethod
    def newc(cls, shear: float, mx: float, my: float, rotation: float, mirror: bool) -> IMatrix2d:
        r"""
        @brief Create a new Matrix2d representing a shear, anisotropic magnification, rotation and mirroring
        @param shear The shear angle
        @param mx The magnification in x direction
        @param my The magnification in y direction
        @param rotation The rotation angle (in degree)
        @param mirror The mirror flag (at x axis)

        The order of execution of the operations is mirror, magnification, shear and rotation.
        This constructor is called 'newc' to distinguish it from the constructor taking the four matrix coefficients ('c' is for composite).
        """
    def __add__(self, m: IMatrix2d) -> IMatrix2d:
        r"""
        @brief Sum of two matrices.
        @param m The other matrix.
        @return The (element-wise) sum of self+m
        """
    def __copy__(self) -> IMatrix2d:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> IMatrix2d:
        r"""
        @brief Creates a copy of self
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Create a new Matrix2d representing a unit transformation
        """
    @overload
    def __init__(self, m: float) -> None:
        r"""
        @brief Create a new Matrix2d representing an isotropic magnification
        @param m The magnification
        """
    @overload
    def __init__(self, t: DCplxTrans) -> None:
        r"""
        @brief Create a new Matrix2d from the given complex transformation@param t The transformation from which to create the matrix (not taking into account the displacement)
        """
    @overload
    def __init__(self, mx: float, my: float) -> None:
        r"""
        @brief Create a new Matrix2d representing an anisotropic magnification
        @param mx The magnification in x direction
        @param my The magnification in y direction
        """
    @overload
    def __init__(self, m11: float, m12: float, m21: float, m22: float) -> None:
        r"""
        @brief Create a new Matrix2d from the four coefficients
        """
    @overload
    def __mul__(self, box: Box) -> Box:
        r"""
        @brief Transforms a box with this matrix.
        @param box The box to transform.
        @return The transformed box

        Please note that the box remains a box, even though the matrix supports shear and rotation. The returned box will be the bounding box of the sheared and rotated rectangle.
        """
    @overload
    def __mul__(self, e: Edge) -> Edge:
        r"""
        @brief Transforms an edge with this matrix.
        @param e The edge to transform.
        @return The transformed edge
        """
    @overload
    def __mul__(self, m: IMatrix2d) -> IMatrix2d:
        r"""
        @brief Product of two matrices.
        @param m The other matrix.
        @return The matrix product self*m
        """
    @overload
    def __mul__(self, p: Point) -> Point:
        r"""
        @brief Transforms a point with this matrix.
        @param p The point to transform.
        @return The transformed point
        """
    @overload
    def __mul__(self, p: Polygon) -> Polygon:
        r"""
        @brief Transforms a polygon with this matrix.
        @param p The polygon to transform.
        @return The transformed polygon
        """
    @overload
    def __mul__(self, p: SimplePolygon) -> SimplePolygon:
        r"""
        @brief Transforms a simple polygon with this matrix.
        @param p The simple polygon to transform.
        @return The transformed simple polygon
        """
    @overload
    def __mul__(self, v: Vector) -> Vector:
        r"""
        @brief Transforms a vector with this matrix.
        @param v The vector to transform.
        @return The transformed vector
        """
    @overload
    def __rmul__(self, box: Box) -> Box:
        r"""
        @brief Transforms a box with this matrix.
        @param box The box to transform.
        @return The transformed box

        Please note that the box remains a box, even though the matrix supports shear and rotation. The returned box will be the bounding box of the sheared and rotated rectangle.
        """
    @overload
    def __rmul__(self, e: Edge) -> Edge:
        r"""
        @brief Transforms an edge with this matrix.
        @param e The edge to transform.
        @return The transformed edge
        """
    @overload
    def __rmul__(self, m: IMatrix2d) -> IMatrix2d:
        r"""
        @brief Product of two matrices.
        @param m The other matrix.
        @return The matrix product self*m
        """
    @overload
    def __rmul__(self, p: Point) -> Point:
        r"""
        @brief Transforms a point with this matrix.
        @param p The point to transform.
        @return The transformed point
        """
    @overload
    def __rmul__(self, p: Polygon) -> Polygon:
        r"""
        @brief Transforms a polygon with this matrix.
        @param p The polygon to transform.
        @return The transformed polygon
        """
    @overload
    def __rmul__(self, p: SimplePolygon) -> SimplePolygon:
        r"""
        @brief Transforms a simple polygon with this matrix.
        @param p The simple polygon to transform.
        @return The transformed simple polygon
        """
    @overload
    def __rmul__(self, v: Vector) -> Vector:
        r"""
        @brief Transforms a vector with this matrix.
        @param v The vector to transform.
        @return The transformed vector
        """
    def __str__(self) -> str:
        r"""
        @brief Convert the matrix to a string.
        @return The string representing this matrix
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def angle(self) -> float:
        r"""
        @brief Returns the rotation angle of the rotation component of this matrix.
        @return The angle in degree.
        The matrix is decomposed into basic transformations assuming an execution order of mirroring at the x axis, rotation, magnification and shear.
        """
    def assign(self, other: IMatrix2d) -> None:
        r"""
        @brief Assigns another object to self
        """
    def cplx_trans(self) -> ICplxTrans:
        r"""
        @brief Converts this matrix to a complex transformation (if possible).
        @return The complex transformation.
        This method is successful only if the matrix does not contain shear components and the magnification must be isotropic.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> IMatrix2d:
        r"""
        @brief Creates a copy of self
        """
    def inverted(self) -> IMatrix2d:
        r"""
        @brief The inverse of this matrix.
        @return The inverse of this matrix
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_mirror(self) -> bool:
        r"""
        @brief Returns the mirror flag of this matrix.
        @return True if this matrix has a mirror component.
        The matrix is decomposed into basic transformations assuming an execution order of mirroring at the x axis, rotation, magnification and shear.
        """
    def m(self, i: int, j: int) -> float:
        r"""
        @brief Gets the m coefficient with the given index.
        @return The coefficient [i,j]
        """
    def m11(self) -> float:
        r"""
        @brief Gets the m11 coefficient.
        @return The value of the m11 coefficient
        """
    def m12(self) -> float:
        r"""
        @brief Gets the m12 coefficient.
        @return The value of the m12 coefficient
        """
    def m21(self) -> float:
        r"""
        @brief Gets the m21 coefficient.
        @return The value of the m21 coefficient
        """
    def m22(self) -> float:
        r"""
        @brief Gets the m22 coefficient.
        @return The value of the m22 coefficient
        """
    def mag_x(self) -> float:
        r"""
        @brief Returns the x magnification of the magnification component of this matrix.
        @return The magnification factor.
        The matrix is decomposed into basic transformations assuming an execution order of mirroring at the x axis, magnification, shear and rotation.
        """
    def mag_y(self) -> float:
        r"""
        @brief Returns the y magnification of the magnification component of this matrix.
        @return The magnification factor.
        The matrix is decomposed into basic transformations assuming an execution order of mirroring at the x axis, magnification, shear and rotation.
        """
    def shear_angle(self) -> float:
        r"""
        @brief Returns the magnitude of the shear component of this matrix.
        @return The shear angle in degree.
        The matrix is decomposed into basic transformations assuming an execution order of mirroring at the x axis, rotation, magnification and shear.
        The shear basic transformation will tilt the x axis towards the y axis and vice versa. The shear angle gives the tilt angle of the axes towards the other one. The possible range for this angle is -45 to 45 degree.
        """
    def to_s(self) -> str:
        r"""
        @brief Convert the matrix to a string.
        @return The string representing this matrix
        """
    def trans(self, p: Point) -> Point:
        r"""
        @brief Transforms a point with this matrix.
        @param p The point to transform.
        @return The transformed point
        """

class Matrix3d:
    r"""
    @brief A 3d matrix object used mainly for representing rotation, shear, displacement and perspective transformations.

    This object represents a 3x3 matrix. This matrix is used to implement generic geometrical transformations in the 2d space mainly. It can be decomposed into basic transformations: mirroring, rotation, shear, displacement and perspective distortion. In that case, the assumed execution order of the basic transformations is mirroring at the x axis, rotation, magnification, shear, displacement and perspective distortion.

    This class was introduced in version 0.22.
    """
    AdjustAll: ClassVar[int]
    r"""
    @brief Mode for \adjust: currently equivalent to \adjust_perspective
    """
    AdjustDisplacement: ClassVar[int]
    r"""
    @brief Mode for \adjust: adjust displacement only
    """
    AdjustMagnification: ClassVar[int]
    r"""
    @brief Mode for \adjust: adjust rotation, mirror option and magnification
    """
    AdjustNone: ClassVar[int]
    r"""
    @brief Mode for \adjust: adjust nothing
    """
    AdjustPerspective: ClassVar[int]
    r"""
    @brief Mode for \adjust: adjust whole matrix including perspective transformation
    """
    AdjustRotation: ClassVar[int]
    r"""
    @brief Mode for \adjust: adjust rotation only
    """
    AdjustRotationMirror: ClassVar[int]
    r"""
    @brief Mode for \adjust: adjust rotation and mirror option
    """
    AdjustShear: ClassVar[int]
    r"""
    @brief Mode for \adjust: adjust rotation, mirror option, magnification and shear
    """
    @overload
    @classmethod
    def new(cls) -> Matrix3d:
        r"""
        @brief Create a new Matrix3d representing a unit transformation
        """
    @overload
    @classmethod
    def new(cls, m: float) -> Matrix3d:
        r"""
        @brief Create a new Matrix3d representing a magnification
        @param m The magnification
        """
    @overload
    @classmethod
    def new(cls, t: DCplxTrans) -> Matrix3d:
        r"""
        @brief Create a new Matrix3d from the given complex transformation@param t The transformation from which to create the matrix
        """
    @overload
    @classmethod
    def new(cls, m11: float, m12: float, m21: float, m22: float) -> Matrix3d:
        r"""
        @brief Create a new Matrix3d from the four coefficients of a Matrix2d
        """
    @overload
    @classmethod
    def new(cls, m11: float, m12: float, m21: float, m22: float, dx: float, dy: float) -> Matrix3d:
        r"""
        @brief Create a new Matrix3d from the four coefficients of a Matrix2d plus a displacement
        """
    @overload
    @classmethod
    def new(cls, m11: float, m12: float, m13: float, m21: float, m22: float, m23: float, m31: float, m32: float, m33: float) -> Matrix3d:
        r"""
        @brief Create a new Matrix3d from the nine matrix coefficients
        """
    @overload
    @classmethod
    def newc(cls, mag: float, rotation: float, mirrx: bool) -> Matrix3d:
        r"""
        @brief Create a new Matrix3d representing a isotropic magnification, rotation and mirroring
        @param mag The magnification
        @param rotation The rotation angle (in degree)
        @param mirrx The mirror flag (at x axis)

        The order of execution of the operations is mirror, magnification and rotation.
        This constructor is called 'newc' to distinguish it from the constructors taking coefficients ('c' is for composite).
        """
    @overload
    @classmethod
    def newc(cls, shear: float, mx: float, my: float, rotation: float, mirrx: bool) -> Matrix3d:
        r"""
        @brief Create a new Matrix3d representing a shear, anisotropic magnification, rotation and mirroring
        @param shear The shear angle
        @param mx The magnification in x direction
        @param mx The magnification in y direction
        @param rotation The rotation angle (in degree)
        @param mirrx The mirror flag (at x axis)

        The order of execution of the operations is mirror, magnification, rotation and shear.
        This constructor is called 'newc' to distinguish it from the constructor taking the four matrix coefficients ('c' is for composite).
        """
    @overload
    @classmethod
    def newc(cls, u: DVector, shear: float, mx: float, my: float, rotation: float, mirrx: bool) -> Matrix3d:
        r"""
        @brief Create a new Matrix3d representing a displacement, shear, anisotropic magnification, rotation and mirroring
        @param u The displacement
        @param shear The shear angle
        @param mx The magnification in x direction
        @param mx The magnification in y direction
        @param rotation The rotation angle (in degree)
        @param mirrx The mirror flag (at x axis)

        The order of execution of the operations is mirror, magnification, rotation, shear and displacement.
        This constructor is called 'newc' to distinguish it from the constructor taking the four matrix coefficients ('c' is for composite).

        Starting with version 0.25 the displacement is of vector type.
        """
    @overload
    @classmethod
    def newc(cls, tx: float, ty: float, z: float, u: DVector, shear: float, mx: float, my: float, rotation: float, mirrx: bool) -> Matrix3d:
        r"""
        @brief Create a new Matrix3d representing a perspective distortion, displacement, shear, anisotropic magnification, rotation and mirroring
        @param tx The perspective tilt angle x (around the y axis)
        @param ty The perspective tilt angle y (around the x axis)
        @param z The observer distance at which the tilt angles are given
        @param u The displacement
        @param shear The shear angle
        @param mx The magnification in x direction
        @param mx The magnification in y direction
        @param rotation The rotation angle (in degree)
        @param mirrx The mirror flag (at x axis)

        The order of execution of the operations is mirror, magnification, rotation, shear, perspective distortion and displacement.
        This constructor is called 'newc' to distinguish it from the constructor taking the four matrix coefficients ('c' is for composite).

        The tx and ty parameters represent the perspective distortion. They denote a tilt of the xy plane around the y axis (tx) or the x axis (ty) in degree. The same effect is achieved for different tilt angles for different observer distances. Hence, the observer distance must be given at which the tilt angles are given. If the magnitude of the tilt angle is not important, z can be set to 1.

        Starting with version 0.25 the displacement is of vector type.
        """
    def __add__(self, m: Matrix3d) -> Matrix3d:
        r"""
        @brief Sum of two matrices.
        @param m The other matrix.
        @return The (element-wise) sum of self+m
        """
    def __copy__(self) -> Matrix3d:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> Matrix3d:
        r"""
        @brief Creates a copy of self
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Create a new Matrix3d representing a unit transformation
        """
    @overload
    def __init__(self, m: float) -> None:
        r"""
        @brief Create a new Matrix3d representing a magnification
        @param m The magnification
        """
    @overload
    def __init__(self, t: DCplxTrans) -> None:
        r"""
        @brief Create a new Matrix3d from the given complex transformation@param t The transformation from which to create the matrix
        """
    @overload
    def __init__(self, m11: float, m12: float, m21: float, m22: float) -> None:
        r"""
        @brief Create a new Matrix3d from the four coefficients of a Matrix2d
        """
    @overload
    def __init__(self, m11: float, m12: float, m21: float, m22: float, dx: float, dy: float) -> None:
        r"""
        @brief Create a new Matrix3d from the four coefficients of a Matrix2d plus a displacement
        """
    @overload
    def __init__(self, m11: float, m12: float, m13: float, m21: float, m22: float, m23: float, m31: float, m32: float, m33: float) -> None:
        r"""
        @brief Create a new Matrix3d from the nine matrix coefficients
        """
    @overload
    def __mul__(self, box: DBox) -> DBox:
        r"""
        @brief Transforms a box with this matrix.
        @param box The box to transform.
        @return The transformed box

        Please note that the box remains a box, even though the matrix supports shear and rotation. The returned box will be the bounding box of the sheared and rotated rectangle.
        """
    @overload
    def __mul__(self, e: DEdge) -> DEdge:
        r"""
        @brief Transforms an edge with this matrix.
        @param e The edge to transform.
        @return The transformed edge
        """
    @overload
    def __mul__(self, m: Matrix3d) -> Matrix3d:
        r"""
        @brief Product of two matrices.
        @param m The other matrix.
        @return The matrix product self*m
        """
    @overload
    def __mul__(self, p: DPoint) -> DPoint:
        r"""
        @brief Transforms a point with this matrix.
        @param p The point to transform.
        @return The transformed point
        """
    @overload
    def __mul__(self, p: DPolygon) -> DPolygon:
        r"""
        @brief Transforms a polygon with this matrix.
        @param p The polygon to transform.
        @return The transformed polygon
        """
    @overload
    def __mul__(self, p: DSimplePolygon) -> DSimplePolygon:
        r"""
        @brief Transforms a simple polygon with this matrix.
        @param p The simple polygon to transform.
        @return The transformed simple polygon
        """
    @overload
    def __mul__(self, v: DVector) -> DVector:
        r"""
        @brief Transforms a vector with this matrix.
        @param v The vector to transform.
        @return The transformed vector
        """
    @overload
    def __rmul__(self, box: DBox) -> DBox:
        r"""
        @brief Transforms a box with this matrix.
        @param box The box to transform.
        @return The transformed box

        Please note that the box remains a box, even though the matrix supports shear and rotation. The returned box will be the bounding box of the sheared and rotated rectangle.
        """
    @overload
    def __rmul__(self, e: DEdge) -> DEdge:
        r"""
        @brief Transforms an edge with this matrix.
        @param e The edge to transform.
        @return The transformed edge
        """
    @overload
    def __rmul__(self, m: Matrix3d) -> Matrix3d:
        r"""
        @brief Product of two matrices.
        @param m The other matrix.
        @return The matrix product self*m
        """
    @overload
    def __rmul__(self, p: DPoint) -> DPoint:
        r"""
        @brief Transforms a point with this matrix.
        @param p The point to transform.
        @return The transformed point
        """
    @overload
    def __rmul__(self, p: DPolygon) -> DPolygon:
        r"""
        @brief Transforms a polygon with this matrix.
        @param p The polygon to transform.
        @return The transformed polygon
        """
    @overload
    def __rmul__(self, p: DSimplePolygon) -> DSimplePolygon:
        r"""
        @brief Transforms a simple polygon with this matrix.
        @param p The simple polygon to transform.
        @return The transformed simple polygon
        """
    @overload
    def __rmul__(self, v: DVector) -> DVector:
        r"""
        @brief Transforms a vector with this matrix.
        @param v The vector to transform.
        @return The transformed vector
        """
    def __str__(self) -> str:
        r"""
        @brief Convert the matrix to a string.
        @return The string representing this matrix
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def adjust(self, landmarks_before: Sequence[DPoint], landmarks_after: Sequence[DPoint], flags: int, fixed_point: int) -> None:
        r"""
        @brief Adjust a 3d matrix to match the given set of landmarks

        This function tries to adjust the matrix
        such, that either the matrix is changed as little as possible (if few landmarks are given) 
        or that the "after" landmarks will match as close as possible to the "before" landmarks 
        (if the problem is overdetermined).

        @param landmarks_before The points before the transformation.
        @param landmarks_after The points after the transformation.
        @param mode Selects the adjustment mode. Must be one of the Adjust... constants.
        @param fixed_point The index of the fixed point (one that is definitely mapped to the target) or -1 if there is none
        """
    def angle(self) -> float:
        r"""
        @brief Returns the rotation angle of the rotation component of this matrix.
        @return The angle in degree.
        See the description of this class for details about the basic transformations.
        """
    def assign(self, other: Matrix3d) -> None:
        r"""
        @brief Assigns another object to self
        """
    def cplx_trans(self) -> DCplxTrans:
        r"""
        @brief Converts this matrix to a complex transformation (if possible).
        @return The complex transformation.
        This method is successful only if the matrix does not contain shear or perspective distortion components and the magnification must be isotropic.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def disp(self) -> DVector:
        r"""
        @brief Returns the displacement vector of this transformation.

        Starting with version 0.25 this method returns a vector type instead of a point.
        @return The displacement vector.
        """
    def dup(self) -> Matrix3d:
        r"""
        @brief Creates a copy of self
        """
    def inverted(self) -> Matrix3d:
        r"""
        @brief The inverse of this matrix.
        @return The inverse of this matrix
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_mirror(self) -> bool:
        r"""
        @brief Returns the mirror flag of this matrix.
        @return True if this matrix has a mirror component.
        See the description of this class for details about the basic transformations.
        """
    def m(self, i: int, j: int) -> float:
        r"""
        @brief Gets the m coefficient with the given index.
        @return The coefficient [i,j]
        """
    def mag_x(self) -> float:
        r"""
        @brief Returns the x magnification of the magnification component of this matrix.
        @return The magnification factor.
        """
    def mag_y(self) -> float:
        r"""
        @brief Returns the y magnification of the magnification component of this matrix.
        @return The magnification factor.
        """
    def shear_angle(self) -> float:
        r"""
        @brief Returns the magnitude of the shear component of this matrix.
        @return The shear angle in degree.
        The shear basic transformation will tilt the x axis towards the y axis and vice versa. The shear angle gives the tilt angle of the axes towards the other one. The possible range for this angle is -45 to 45 degree.See the description of this class for details about the basic transformations.
        """
    def to_s(self) -> str:
        r"""
        @brief Convert the matrix to a string.
        @return The string representing this matrix
        """
    def trans(self, p: DPoint) -> DPoint:
        r"""
        @brief Transforms a point with this matrix.
        @param p The point to transform.
        @return The transformed point
        """
    def tx(self, z: float) -> float:
        r"""
        @brief Returns the perspective tilt angle tx.
        @param z The observer distance at which the tilt angle is computed.
        @return The tilt angle tx.
        The tx and ty parameters represent the perspective distortion. They denote a tilt of the xy plane around the y axis (tx) or the x axis (ty) in degree. The same effect is achieved for different tilt angles at different observer distances. Hence, the observer distance must be specified at which the tilt angle is computed. If the magnitude of the tilt angle is not important, z can be set to 1.
        """
    def ty(self, z: float) -> float:
        r"""
        @brief Returns the perspective tilt angle ty.
        @param z The observer distance at which the tilt angle is computed.
        @return The tilt angle ty.
        The tx and ty parameters represent the perspective distortion. They denote a tilt of the xy plane around the y axis (tx) or the x axis (ty) in degree. The same effect is achieved for different tilt angles at different observer distances. Hence, the observer distance must be specified at which the tilt angle is computed. If the magnitude of the tilt angle is not important, z can be set to 1.
        """

class IMatrix3d:
    r"""
    @brief A 3d matrix object used mainly for representing rotation, shear, displacement and perspective transformations (integer coordinate version).

    This object represents a 3x3 matrix. This matrix is used to implement generic geometrical transformations in the 2d space mainly. It can be decomposed into basic transformations: mirroring, rotation, shear, displacement and perspective distortion. In that case, the assumed execution order of the basic transformations is mirroring at the x axis, rotation, magnification, shear, displacement and perspective distortion.

    The integer variant was introduced in version 0.27.
    """
    @overload
    @classmethod
    def new(cls) -> IMatrix3d:
        r"""
        @brief Create a new Matrix3d representing a unit transformation
        """
    @overload
    @classmethod
    def new(cls, m: float) -> IMatrix3d:
        r"""
        @brief Create a new Matrix3d representing a magnification
        @param m The magnification
        """
    @overload
    @classmethod
    def new(cls, t: ICplxTrans) -> IMatrix3d:
        r"""
        @brief Create a new Matrix3d from the given complex transformation@param t The transformation from which to create the matrix
        """
    @overload
    @classmethod
    def new(cls, m11: float, m12: float, m21: float, m22: float) -> IMatrix3d:
        r"""
        @brief Create a new Matrix3d from the four coefficients of a Matrix2d
        """
    @overload
    @classmethod
    def new(cls, m11: float, m12: float, m21: float, m22: float, dx: float, dy: float) -> IMatrix3d:
        r"""
        @brief Create a new Matrix3d from the four coefficients of a Matrix2d plus a displacement
        """
    @overload
    @classmethod
    def new(cls, m11: float, m12: float, m13: float, m21: float, m22: float, m23: float, m31: float, m32: float, m33: float) -> IMatrix3d:
        r"""
        @brief Create a new Matrix3d from the nine matrix coefficients
        """
    @overload
    @classmethod
    def newc(cls, mag: float, rotation: float, mirrx: bool) -> IMatrix3d:
        r"""
        @brief Create a new Matrix3d representing a isotropic magnification, rotation and mirroring
        @param mag The magnification
        @param rotation The rotation angle (in degree)
        @param mirrx The mirror flag (at x axis)

        The order of execution of the operations is mirror, magnification and rotation.
        This constructor is called 'newc' to distinguish it from the constructors taking coefficients ('c' is for composite).
        """
    @overload
    @classmethod
    def newc(cls, shear: float, mx: float, my: float, rotation: float, mirrx: bool) -> IMatrix3d:
        r"""
        @brief Create a new Matrix3d representing a shear, anisotropic magnification, rotation and mirroring
        @param shear The shear angle
        @param mx The magnification in x direction
        @param mx The magnification in y direction
        @param rotation The rotation angle (in degree)
        @param mirrx The mirror flag (at x axis)

        The order of execution of the operations is mirror, magnification, rotation and shear.
        This constructor is called 'newc' to distinguish it from the constructor taking the four matrix coefficients ('c' is for composite).
        """
    @overload
    @classmethod
    def newc(cls, u: Vector, shear: float, mx: float, my: float, rotation: float, mirrx: bool) -> IMatrix3d:
        r"""
        @brief Create a new Matrix3d representing a displacement, shear, anisotropic magnification, rotation and mirroring
        @param u The displacement
        @param shear The shear angle
        @param mx The magnification in x direction
        @param mx The magnification in y direction
        @param rotation The rotation angle (in degree)
        @param mirrx The mirror flag (at x axis)

        The order of execution of the operations is mirror, magnification, rotation, shear and displacement.
        This constructor is called 'newc' to distinguish it from the constructor taking the four matrix coefficients ('c' is for composite).

        Starting with version 0.25 the displacement is of vector type.
        """
    @overload
    @classmethod
    def newc(cls, tx: float, ty: float, z: float, u: Vector, shear: float, mx: float, my: float, rotation: float, mirrx: bool) -> IMatrix3d:
        r"""
        @brief Create a new Matrix3d representing a perspective distortion, displacement, shear, anisotropic magnification, rotation and mirroring
        @param tx The perspective tilt angle x (around the y axis)
        @param ty The perspective tilt angle y (around the x axis)
        @param z The observer distance at which the tilt angles are given
        @param u The displacement
        @param shear The shear angle
        @param mx The magnification in x direction
        @param mx The magnification in y direction
        @param rotation The rotation angle (in degree)
        @param mirrx The mirror flag (at x axis)

        The order of execution of the operations is mirror, magnification, rotation, shear, perspective distortion and displacement.
        This constructor is called 'newc' to distinguish it from the constructor taking the four matrix coefficients ('c' is for composite).

        The tx and ty parameters represent the perspective distortion. They denote a tilt of the xy plane around the y axis (tx) or the x axis (ty) in degree. The same effect is achieved for different tilt angles for different observer distances. Hence, the observer distance must be given at which the tilt angles are given. If the magnitude of the tilt angle is not important, z can be set to 1.

        Starting with version 0.25 the displacement is of vector type.
        """
    def __add__(self, m: IMatrix3d) -> IMatrix3d:
        r"""
        @brief Sum of two matrices.
        @param m The other matrix.
        @return The (element-wise) sum of self+m
        """
    def __copy__(self) -> IMatrix3d:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> IMatrix3d:
        r"""
        @brief Creates a copy of self
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Create a new Matrix3d representing a unit transformation
        """
    @overload
    def __init__(self, m: float) -> None:
        r"""
        @brief Create a new Matrix3d representing a magnification
        @param m The magnification
        """
    @overload
    def __init__(self, t: ICplxTrans) -> None:
        r"""
        @brief Create a new Matrix3d from the given complex transformation@param t The transformation from which to create the matrix
        """
    @overload
    def __init__(self, m11: float, m12: float, m21: float, m22: float) -> None:
        r"""
        @brief Create a new Matrix3d from the four coefficients of a Matrix2d
        """
    @overload
    def __init__(self, m11: float, m12: float, m21: float, m22: float, dx: float, dy: float) -> None:
        r"""
        @brief Create a new Matrix3d from the four coefficients of a Matrix2d plus a displacement
        """
    @overload
    def __init__(self, m11: float, m12: float, m13: float, m21: float, m22: float, m23: float, m31: float, m32: float, m33: float) -> None:
        r"""
        @brief Create a new Matrix3d from the nine matrix coefficients
        """
    @overload
    def __mul__(self, box: Box) -> Box:
        r"""
        @brief Transforms a box with this matrix.
        @param box The box to transform.
        @return The transformed box

        Please note that the box remains a box, even though the matrix supports shear and rotation. The returned box will be the bounding box of the sheared and rotated rectangle.
        """
    @overload
    def __mul__(self, e: Edge) -> Edge:
        r"""
        @brief Transforms an edge with this matrix.
        @param e The edge to transform.
        @return The transformed edge
        """
    @overload
    def __mul__(self, m: IMatrix3d) -> IMatrix3d:
        r"""
        @brief Product of two matrices.
        @param m The other matrix.
        @return The matrix product self*m
        """
    @overload
    def __mul__(self, p: Point) -> Point:
        r"""
        @brief Transforms a point with this matrix.
        @param p The point to transform.
        @return The transformed point
        """
    @overload
    def __mul__(self, p: Polygon) -> Polygon:
        r"""
        @brief Transforms a polygon with this matrix.
        @param p The polygon to transform.
        @return The transformed polygon
        """
    @overload
    def __mul__(self, p: SimplePolygon) -> SimplePolygon:
        r"""
        @brief Transforms a simple polygon with this matrix.
        @param p The simple polygon to transform.
        @return The transformed simple polygon
        """
    @overload
    def __mul__(self, v: Vector) -> Vector:
        r"""
        @brief Transforms a vector with this matrix.
        @param v The vector to transform.
        @return The transformed vector
        """
    @overload
    def __rmul__(self, box: Box) -> Box:
        r"""
        @brief Transforms a box with this matrix.
        @param box The box to transform.
        @return The transformed box

        Please note that the box remains a box, even though the matrix supports shear and rotation. The returned box will be the bounding box of the sheared and rotated rectangle.
        """
    @overload
    def __rmul__(self, e: Edge) -> Edge:
        r"""
        @brief Transforms an edge with this matrix.
        @param e The edge to transform.
        @return The transformed edge
        """
    @overload
    def __rmul__(self, m: IMatrix3d) -> IMatrix3d:
        r"""
        @brief Product of two matrices.
        @param m The other matrix.
        @return The matrix product self*m
        """
    @overload
    def __rmul__(self, p: Point) -> Point:
        r"""
        @brief Transforms a point with this matrix.
        @param p The point to transform.
        @return The transformed point
        """
    @overload
    def __rmul__(self, p: Polygon) -> Polygon:
        r"""
        @brief Transforms a polygon with this matrix.
        @param p The polygon to transform.
        @return The transformed polygon
        """
    @overload
    def __rmul__(self, p: SimplePolygon) -> SimplePolygon:
        r"""
        @brief Transforms a simple polygon with this matrix.
        @param p The simple polygon to transform.
        @return The transformed simple polygon
        """
    @overload
    def __rmul__(self, v: Vector) -> Vector:
        r"""
        @brief Transforms a vector with this matrix.
        @param v The vector to transform.
        @return The transformed vector
        """
    def __str__(self) -> str:
        r"""
        @brief Convert the matrix to a string.
        @return The string representing this matrix
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def angle(self) -> float:
        r"""
        @brief Returns the rotation angle of the rotation component of this matrix.
        @return The angle in degree.
        See the description of this class for details about the basic transformations.
        """
    def assign(self, other: IMatrix3d) -> None:
        r"""
        @brief Assigns another object to self
        """
    def cplx_trans(self) -> DCplxTrans:
        r"""
        @brief Converts this matrix to a complex transformation (if possible).
        @return The complex transformation.
        This method is successful only if the matrix does not contain shear or perspective distortion components and the magnification must be isotropic.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def disp(self) -> Vector:
        r"""
        @brief Returns the displacement vector of this transformation.

        Starting with version 0.25 this method returns a vector type instead of a point.
        @return The displacement vector.
        """
    def dup(self) -> IMatrix3d:
        r"""
        @brief Creates a copy of self
        """
    def inverted(self) -> IMatrix3d:
        r"""
        @brief The inverse of this matrix.
        @return The inverse of this matrix
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_mirror(self) -> bool:
        r"""
        @brief Returns the mirror flag of this matrix.
        @return True if this matrix has a mirror component.
        See the description of this class for details about the basic transformations.
        """
    def m(self, i: int, j: int) -> float:
        r"""
        @brief Gets the m coefficient with the given index.
        @return The coefficient [i,j]
        """
    def mag_x(self) -> float:
        r"""
        @brief Returns the x magnification of the magnification component of this matrix.
        @return The magnification factor.
        """
    def mag_y(self) -> float:
        r"""
        @brief Returns the y magnification of the magnification component of this matrix.
        @return The magnification factor.
        """
    def shear_angle(self) -> float:
        r"""
        @brief Returns the magnitude of the shear component of this matrix.
        @return The shear angle in degree.
        The shear basic transformation will tilt the x axis towards the y axis and vice versa. The shear angle gives the tilt angle of the axes towards the other one. The possible range for this angle is -45 to 45 degree.See the description of this class for details about the basic transformations.
        """
    def to_s(self) -> str:
        r"""
        @brief Convert the matrix to a string.
        @return The string representing this matrix
        """
    def trans(self, p: Point) -> Point:
        r"""
        @brief Transforms a point with this matrix.
        @param p The point to transform.
        @return The transformed point
        """
    def tx(self, z: float) -> float:
        r"""
        @brief Returns the perspective tilt angle tx.
        @param z The observer distance at which the tilt angle is computed.
        @return The tilt angle tx.
        The tx and ty parameters represent the perspective distortion. They denote a tilt of the xy plane around the y axis (tx) or the x axis (ty) in degree. The same effect is achieved for different tilt angles at different observer distances. Hence, the observer distance must be specified at which the tilt angle is computed. If the magnitude of the tilt angle is not important, z can be set to 1.
        """
    def ty(self, z: float) -> float:
        r"""
        @brief Returns the perspective tilt angle ty.
        @param z The observer distance at which the tilt angle is computed.
        @return The tilt angle ty.
        The tx and ty parameters represent the perspective distortion. They denote a tilt of the xy plane around the y axis (tx) or the x axis (ty) in degree. The same effect is achieved for different tilt angles at different observer distances. Hence, the observer distance must be specified at which the tilt angle is computed. If the magnitude of the tilt angle is not important, z can be set to 1.
        """

class Path:
    r"""
    @brief A path class

    A path consists of an sequence of line segments forming the 'spine' of the path and a width. In addition, the starting point can be drawn back by a certain extent (the 'begin extension') and the end point can be pulled forward somewhat (by the 'end extension').

    A path may have round ends for special purposes. In particular, a round-ended path with a single point can represent a circle. Round-ended paths should have being and end extensions equal to half the width. Non-round-ended paths with a single point are allowed but the definition of the resulting shape in not well defined and may differ in other tools.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    bgn_ext: int
    r"""
    Getter:
    @brief Get the begin extension

    Setter:
    @brief Set the begin extension
    """
    end_ext: int
    r"""
    Getter:
    @brief Get the end extension

    Setter:
    @brief Set the end extension
    """
    points: int
    r"""
    Getter:
    @brief Get the number of points
    Setter:
    @brief Set the points of the path
    @param p An array of points to assign to the path's spine
    """
    round: bool
    r"""
    Getter:
    @brief Returns true, if the path has round ends

    Setter:
    @brief Set the 'round ends' flag
    A path with round ends show half circles at the ends, instead of square or rectangular ends. Paths with this flag set should use a begin and end extension of half the width (see \bgn_ext and \end_ext). The interpretation of such paths in other tools may differ otherwise.
    """
    width: int
    r"""
    Getter:
    @brief Get the width

    Setter:
    @brief Set the width
    """
    @classmethod
    def from_dpath(cls, dpath: DPath) -> Path:
        r"""
        @brief Creates an integer coordinate path from a floating-point coordinate path

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dpath'.
        """
    @classmethod
    def from_s(cls, s: str) -> Path:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> Path:
        r"""
        @brief Default constructor: creates an empty (invalid) path with width 0
        """
    @overload
    @classmethod
    def new(cls, dpath: DPath) -> Path:
        r"""
        @brief Creates an integer coordinate path from a floating-point coordinate path

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dpath'.
        """
    @overload
    @classmethod
    def new(cls, pts: Sequence[Point], width: int) -> Path:
        r"""
        @brief Constructor given the points of the path's spine and the width


        @param pts The points forming the spine of the path
        @param width The width of the path
        """
    @overload
    @classmethod
    def new(cls, pts: Sequence[Point], width: int, bgn_ext: int, end_ext: int) -> Path:
        r"""
        @brief Constructor given the points of the path's spine, the width and the extensions


        @param pts The points forming the spine of the path
        @param width The width of the path
        @param bgn_ext The begin extension of the path
        @param end_ext The end extension of the path
        """
    @overload
    @classmethod
    def new(cls, pts: Sequence[Point], width: int, bgn_ext: int, end_ext: int, round: bool) -> Path:
        r"""
        @brief Constructor given the points of the path's spine, the width, the extensions and the round end flag


        @param pts The points forming the spine of the path
        @param width The width of the path
        @param bgn_ext The begin extension of the path
        @param end_ext The end extension of the path
        @param round If this flag is true, the path will get rounded ends
        """
    @classmethod
    def new_pw(cls, pts: Sequence[Point], width: int) -> Path:
        r"""
        @brief Constructor given the points of the path's spine and the width


        @param pts The points forming the spine of the path
        @param width The width of the path
        """
    @classmethod
    def new_pwx(cls, pts: Sequence[Point], width: int, bgn_ext: int, end_ext: int) -> Path:
        r"""
        @brief Constructor given the points of the path's spine, the width and the extensions


        @param pts The points forming the spine of the path
        @param width The width of the path
        @param bgn_ext The begin extension of the path
        @param end_ext The end extension of the path
        """
    @classmethod
    def new_pwxr(cls, pts: Sequence[Point], width: int, bgn_ext: int, end_ext: int, round: bool) -> Path:
        r"""
        @brief Constructor given the points of the path's spine, the width, the extensions and the round end flag


        @param pts The points forming the spine of the path
        @param width The width of the path
        @param bgn_ext The begin extension of the path
        @param end_ext The end extension of the path
        @param round If this flag is true, the path will get rounded ends
        """
    def __copy__(self) -> Path:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> Path:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, p: object) -> bool:
        r"""
        @brief Equality test
        @param p The object to compare against
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given polygon. This method enables polygons as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor: creates an empty (invalid) path with width 0
        """
    @overload
    def __init__(self, dpath: DPath) -> None:
        r"""
        @brief Creates an integer coordinate path from a floating-point coordinate path

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dpath'.
        """
    @overload
    def __init__(self, pts: Sequence[Point], width: int) -> None:
        r"""
        @brief Constructor given the points of the path's spine and the width


        @param pts The points forming the spine of the path
        @param width The width of the path
        """
    @overload
    def __init__(self, pts: Sequence[Point], width: int, bgn_ext: int, end_ext: int) -> None:
        r"""
        @brief Constructor given the points of the path's spine, the width and the extensions


        @param pts The points forming the spine of the path
        @param width The width of the path
        @param bgn_ext The begin extension of the path
        @param end_ext The end extension of the path
        """
    @overload
    def __init__(self, pts: Sequence[Point], width: int, bgn_ext: int, end_ext: int, round: bool) -> None:
        r"""
        @brief Constructor given the points of the path's spine, the width, the extensions and the round end flag


        @param pts The points forming the spine of the path
        @param width The width of the path
        @param bgn_ext The begin extension of the path
        @param end_ext The end extension of the path
        @param round If this flag is true, the path will get rounded ends
        """
    def __lt__(self, p: Path) -> bool:
        r"""
        @brief Less operator
        @param p The object to compare against
        This operator is provided to establish some, not necessarily a certain sorting order
        """
    def __mul__(self, f: float) -> Path:
        r"""
        @brief Scaling by some factor


        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __ne__(self, p: object) -> bool:
        r"""
        @brief Inequality test
        @param p The object to compare against
        """
    def __rmul__(self, f: float) -> Path:
        r"""
        @brief Scaling by some factor


        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __str__(self) -> str:
        r"""
        @brief Convert to a string
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def area(self) -> int:
        r"""
        @brief Returns the approximate area of the path
        This method returns the approximate value of the area. It is computed from the length times the width. end extensions are taken into account correctly, but not effects of the corner interpolation.
        This method was added in version 0.22.
        """
    def assign(self, other: Path) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> Box:
        r"""
        @brief Returns the bounding box of the path
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> Path:
        r"""
        @brief Creates a copy of self
        """
    def each_point(self) -> Iterator[Point]:
        r"""
        @brief Get the points that make up the path's spine
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given polygon. This method enables polygons as hash keys.

        This method has been introduced in version 0.25.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_round(self) -> bool:
        r"""
        @brief Returns true, if the path has round ends
        """
    def length(self) -> int:
        r"""
        @brief Returns the length of the path
        the length of the path is determined by summing the lengths of the segments and adding begin and end extensions. For round-ended paths the length of the paths between the tips of the ends.

        This method was added in version 0.23.
        """
    @overload
    def move(self, p: Vector) -> Path:
        r"""
        @brief Moves the path.

        Moves the path by the given offset and returns the 
        moved path. The path is overwritten.

        @param p The distance to move the path.

        @return The moved path.
        """
    @overload
    def move(self, dx: int, dy: int) -> Path:
        r"""
        @brief Moves the path.

        Moves the path by the given offset and returns the 
        moved path. The path is overwritten.

        @param dx The x distance to move the path.
        @param dy The y distance to move the path.

        @return The moved path.

        This version has been added in version 0.23.
        """
    @overload
    def moved(self, p: Vector) -> Path:
        r"""
        @brief Returns the moved path (does not change self)

        Moves the path by the given offset and returns the 
        moved path. The path is not modified.

        @param p The distance to move the path.

        @return The moved path.
        """
    @overload
    def moved(self, dx: int, dy: int) -> Path:
        r"""
        @brief Returns the moved path (does not change self)

        Moves the path by the given offset and returns the 
        moved path. The path is not modified.

        @param dx The x distance to move the path.
        @param dy The y distance to move the path.

        @return The moved path.

        This version has been added in version 0.23.
        """
    def num_points(self) -> int:
        r"""
        @brief Get the number of points
        """
    def perimeter(self) -> int:
        r"""
        @brief Returns the approximate perimeter of the path
        This method returns the approximate value of the perimeter. It is computed from the length and the width. end extensions are taken into account correctly, but not effects of the corner interpolation.
        This method was added in version 0.24.4.
        """
    def polygon(self) -> Polygon:
        r"""
        @brief Convert the path to a polygon
        The returned polygon is not guaranteed to be non-self overlapping. This may happen if the path overlaps itself or contains very short segments.
        """
    def round_corners(self, radius: float, npoints: int) -> Path:
        r"""
        @brief Creates a new path whose corners are interpolated with circular bends

        @param radius The radius of the bends
        @param npoints The number of points (per full circle) used for interpolating the bends

        This method has been introduced in version 0.25.
        """
    def simple_polygon(self) -> SimplePolygon:
        r"""
        @brief Convert the path to a simple polygon
        The returned polygon is not guaranteed to be non-selfoverlapping. This may happen if the path overlaps itself or contains very short segments.
        """
    def to_dtype(self, dbu: Optional[float] = ...) -> DPath:
        r"""
        @brief Converts the path to a floating-point coordinate path

        The database unit can be specified to translate the integer-coordinate path into a floating-point coordinate path in micron units. The database unit is basically a scaling factor.

        This method has been introduced in version 0.25.
        """
    def to_s(self) -> str:
        r"""
        @brief Convert to a string
        """
    @overload
    def transformed(self, t: CplxTrans) -> DPath:
        r"""
        @brief Transform the path.

        Transforms the path with the given complex transformation.
        Does not modify the path but returns the transformed path.

        @param t The transformation to apply.

        @return The transformed path.
        """
    @overload
    def transformed(self, t: ICplxTrans) -> Path:
        r"""
        @brief Transform the path.

        Transforms the path with the given complex transformation.
        Does not modify the path but returns the transformed path.

        @param t The transformation to apply.

        @return The transformed path (in this case an integer coordinate path).

        This method has been introduced in version 0.18.
        """
    @overload
    def transformed(self, t: Trans) -> Path:
        r"""
        @brief Transform the path.

        Transforms the path with the given transformation.
        Does not modify the path but returns the transformed path.

        @param t The transformation to apply.

        @return The transformed path.
        """
    def transformed_cplx(self, t: CplxTrans) -> DPath:
        r"""
        @brief Transform the path.

        Transforms the path with the given complex transformation.
        Does not modify the path but returns the transformed path.

        @param t The transformation to apply.

        @return The transformed path.
        """

class DPath:
    r"""
    @brief A path class

    A path consists of an sequence of line segments forming the 'spine' of the path and a width. In addition, the starting point can be drawn back by a certain extent (the 'begin extension') and the end point can be pulled forward somewhat (by the 'end extension').

    A path may have round ends for special purposes. In particular, a round-ended path with a single point can represent a circle. Round-ended paths should have being and end extensions equal to half the width. Non-round-ended paths with a single point are allowed but the definition of the resulting shape in not well defined and may differ in other tools.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    bgn_ext: float
    r"""
    Getter:
    @brief Get the begin extension

    Setter:
    @brief Set the begin extension
    """
    end_ext: float
    r"""
    Getter:
    @brief Get the end extension

    Setter:
    @brief Set the end extension
    """
    points: int
    r"""
    Getter:
    @brief Get the number of points
    Setter:
    @brief Set the points of the path
    @param p An array of points to assign to the path's spine
    """
    round: bool
    r"""
    Getter:
    @brief Returns true, if the path has round ends

    Setter:
    @brief Set the 'round ends' flag
    A path with round ends show half circles at the ends, instead of square or rectangular ends. Paths with this flag set should use a begin and end extension of half the width (see \bgn_ext and \end_ext). The interpretation of such paths in other tools may differ otherwise.
    """
    width: float
    r"""
    Getter:
    @brief Get the width

    Setter:
    @brief Set the width
    """
    @classmethod
    def from_ipath(cls, path: Path) -> DPath:
        r"""
        @brief Creates a floating-point coordinate path from an integer coordinate path

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipath'.
        """
    @classmethod
    def from_s(cls, s: str) -> DPath:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> DPath:
        r"""
        @brief Default constructor: creates an empty (invalid) path with width 0
        """
    @overload
    @classmethod
    def new(cls, path: Path) -> DPath:
        r"""
        @brief Creates a floating-point coordinate path from an integer coordinate path

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipath'.
        """
    @overload
    @classmethod
    def new(cls, pts: Sequence[DPoint], width: float) -> DPath:
        r"""
        @brief Constructor given the points of the path's spine and the width


        @param pts The points forming the spine of the path
        @param width The width of the path
        """
    @overload
    @classmethod
    def new(cls, pts: Sequence[DPoint], width: float, bgn_ext: float, end_ext: float) -> DPath:
        r"""
        @brief Constructor given the points of the path's spine, the width and the extensions


        @param pts The points forming the spine of the path
        @param width The width of the path
        @param bgn_ext The begin extension of the path
        @param end_ext The end extension of the path
        """
    @overload
    @classmethod
    def new(cls, pts: Sequence[DPoint], width: float, bgn_ext: float, end_ext: float, round: bool) -> DPath:
        r"""
        @brief Constructor given the points of the path's spine, the width, the extensions and the round end flag


        @param pts The points forming the spine of the path
        @param width The width of the path
        @param bgn_ext The begin extension of the path
        @param end_ext The end extension of the path
        @param round If this flag is true, the path will get rounded ends
        """
    @classmethod
    def new_pw(cls, pts: Sequence[DPoint], width: float) -> DPath:
        r"""
        @brief Constructor given the points of the path's spine and the width


        @param pts The points forming the spine of the path
        @param width The width of the path
        """
    @classmethod
    def new_pwx(cls, pts: Sequence[DPoint], width: float, bgn_ext: float, end_ext: float) -> DPath:
        r"""
        @brief Constructor given the points of the path's spine, the width and the extensions


        @param pts The points forming the spine of the path
        @param width The width of the path
        @param bgn_ext The begin extension of the path
        @param end_ext The end extension of the path
        """
    @classmethod
    def new_pwxr(cls, pts: Sequence[DPoint], width: float, bgn_ext: float, end_ext: float, round: bool) -> DPath:
        r"""
        @brief Constructor given the points of the path's spine, the width, the extensions and the round end flag


        @param pts The points forming the spine of the path
        @param width The width of the path
        @param bgn_ext The begin extension of the path
        @param end_ext The end extension of the path
        @param round If this flag is true, the path will get rounded ends
        """
    def __copy__(self) -> DPath:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DPath:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, p: object) -> bool:
        r"""
        @brief Equality test
        @param p The object to compare against
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given polygon. This method enables polygons as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor: creates an empty (invalid) path with width 0
        """
    @overload
    def __init__(self, path: Path) -> None:
        r"""
        @brief Creates a floating-point coordinate path from an integer coordinate path

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipath'.
        """
    @overload
    def __init__(self, pts: Sequence[DPoint], width: float) -> None:
        r"""
        @brief Constructor given the points of the path's spine and the width


        @param pts The points forming the spine of the path
        @param width The width of the path
        """
    @overload
    def __init__(self, pts: Sequence[DPoint], width: float, bgn_ext: float, end_ext: float) -> None:
        r"""
        @brief Constructor given the points of the path's spine, the width and the extensions


        @param pts The points forming the spine of the path
        @param width The width of the path
        @param bgn_ext The begin extension of the path
        @param end_ext The end extension of the path
        """
    @overload
    def __init__(self, pts: Sequence[DPoint], width: float, bgn_ext: float, end_ext: float, round: bool) -> None:
        r"""
        @brief Constructor given the points of the path's spine, the width, the extensions and the round end flag


        @param pts The points forming the spine of the path
        @param width The width of the path
        @param bgn_ext The begin extension of the path
        @param end_ext The end extension of the path
        @param round If this flag is true, the path will get rounded ends
        """
    def __lt__(self, p: DPath) -> bool:
        r"""
        @brief Less operator
        @param p The object to compare against
        This operator is provided to establish some, not necessarily a certain sorting order
        """
    def __mul__(self, f: float) -> DPath:
        r"""
        @brief Scaling by some factor


        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __ne__(self, p: object) -> bool:
        r"""
        @brief Inequality test
        @param p The object to compare against
        """
    def __rmul__(self, f: float) -> DPath:
        r"""
        @brief Scaling by some factor


        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __str__(self) -> str:
        r"""
        @brief Convert to a string
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def area(self) -> float:
        r"""
        @brief Returns the approximate area of the path
        This method returns the approximate value of the area. It is computed from the length times the width. end extensions are taken into account correctly, but not effects of the corner interpolation.
        This method was added in version 0.22.
        """
    def assign(self, other: DPath) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> DBox:
        r"""
        @brief Returns the bounding box of the path
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DPath:
        r"""
        @brief Creates a copy of self
        """
    def each_point(self) -> Iterator[DPoint]:
        r"""
        @brief Get the points that make up the path's spine
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given polygon. This method enables polygons as hash keys.

        This method has been introduced in version 0.25.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_round(self) -> bool:
        r"""
        @brief Returns true, if the path has round ends
        """
    def length(self) -> float:
        r"""
        @brief Returns the length of the path
        the length of the path is determined by summing the lengths of the segments and adding begin and end extensions. For round-ended paths the length of the paths between the tips of the ends.

        This method was added in version 0.23.
        """
    @overload
    def move(self, p: DVector) -> DPath:
        r"""
        @brief Moves the path.

        Moves the path by the given offset and returns the 
        moved path. The path is overwritten.

        @param p The distance to move the path.

        @return The moved path.
        """
    @overload
    def move(self, dx: float, dy: float) -> DPath:
        r"""
        @brief Moves the path.

        Moves the path by the given offset and returns the 
        moved path. The path is overwritten.

        @param dx The x distance to move the path.
        @param dy The y distance to move the path.

        @return The moved path.

        This version has been added in version 0.23.
        """
    @overload
    def moved(self, p: DVector) -> DPath:
        r"""
        @brief Returns the moved path (does not change self)

        Moves the path by the given offset and returns the 
        moved path. The path is not modified.

        @param p The distance to move the path.

        @return The moved path.
        """
    @overload
    def moved(self, dx: float, dy: float) -> DPath:
        r"""
        @brief Returns the moved path (does not change self)

        Moves the path by the given offset and returns the 
        moved path. The path is not modified.

        @param dx The x distance to move the path.
        @param dy The y distance to move the path.

        @return The moved path.

        This version has been added in version 0.23.
        """
    def num_points(self) -> int:
        r"""
        @brief Get the number of points
        """
    def perimeter(self) -> float:
        r"""
        @brief Returns the approximate perimeter of the path
        This method returns the approximate value of the perimeter. It is computed from the length and the width. end extensions are taken into account correctly, but not effects of the corner interpolation.
        This method was added in version 0.24.4.
        """
    def polygon(self) -> DPolygon:
        r"""
        @brief Convert the path to a polygon
        The returned polygon is not guaranteed to be non-self overlapping. This may happen if the path overlaps itself or contains very short segments.
        """
    def round_corners(self, radius: float, npoints: int, accuracy: float) -> DPath:
        r"""
        @brief Creates a new path whose corners are interpolated with circular bends

        @param radius The radius of the bends
        @param npoints The number of points (per full circle) used for interpolating the bends
        @param accuracy The numerical accuracy of the computation

        The accuracy parameter controls the numerical resolution of the approximation process and should be in the order of half the database unit. This accuracy is used for suppressing redundant points and simplification of the resulting path.

        This method has been introduced in version 0.25.
        """
    def simple_polygon(self) -> DSimplePolygon:
        r"""
        @brief Convert the path to a simple polygon
        The returned polygon is not guaranteed to be non-selfoverlapping. This may happen if the path overlaps itself or contains very short segments.
        """
    def to_itype(self, dbu: Optional[float] = ...) -> Path:
        r"""
        @brief Converts the path to an integer coordinate path

        The database unit can be specified to translate the floating-point coordinate path in micron units to an integer-coordinate path in database units. The path's' coordinates will be divided by the database unit.

        This method has been introduced in version 0.25.
        """
    def to_s(self) -> str:
        r"""
        @brief Convert to a string
        """
    @overload
    def transformed(self, t: DCplxTrans) -> DPath:
        r"""
        @brief Transform the path.

        Transforms the path with the given complex transformation.
        Does not modify the path but returns the transformed path.

        @param t The transformation to apply.

        @return The transformed path.
        """
    @overload
    def transformed(self, t: DTrans) -> DPath:
        r"""
        @brief Transform the path.

        Transforms the path with the given transformation.
        Does not modify the path but returns the transformed path.

        @param t The transformation to apply.

        @return The transformed path.
        """
    @overload
    def transformed(self, t: VCplxTrans) -> Path:
        r"""
        @brief Transforms the polygon with the given complex transformation


        @param t The magnifying transformation to apply
        @return The transformed path (in this case an integer coordinate path)

        This method has been introduced in version 0.25.
        """
    def transformed_cplx(self, t: DCplxTrans) -> DPath:
        r"""
        @brief Transform the path.

        Transforms the path with the given complex transformation.
        Does not modify the path but returns the transformed path.

        @param t The transformation to apply.

        @return The transformed path.
        """

class DPoint:
    r"""
    @brief A point class with double (floating-point) coordinates
    Points represent a coordinate in the two-dimensional coordinate space of layout. They are not geometrical objects by itself. But they are frequently used in the database API for various purposes. Other than the integer variant (\Point), points with floating-point coordinates can represent fractions of a database unit.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    x: float
    r"""
    Getter:
    @brief Accessor to the x coordinate

    Setter:
    @brief Write accessor to the x coordinate
    """
    y: float
    r"""
    Getter:
    @brief Accessor to the y coordinate

    Setter:
    @brief Write accessor to the y coordinate
    """
    @classmethod
    def from_ipoint(cls, point: Point) -> DPoint:
        r"""
        @brief Creates a floating-point coordinate point from an integer coordinate point

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipoint'.
        """
    @classmethod
    def from_s(cls, s: str) -> DPoint:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> DPoint:
        r"""
        @brief Default constructor: creates a point at 0,0
        """
    @overload
    @classmethod
    def new(cls, point: Point) -> DPoint:
        r"""
        @brief Creates a floating-point coordinate point from an integer coordinate point

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipoint'.
        """
    @overload
    @classmethod
    def new(cls, v: DVector) -> DPoint:
        r"""
        @brief Default constructor: creates a point at from an vector
        This constructor is equivalent to computing point(0,0)+v.
        This method has been introduced in version 0.25.
        """
    @overload
    @classmethod
    def new(cls, x: float, y: float) -> DPoint:
        r"""
        @brief Constructor for a point from two coordinate values

        """
    def __add__(self, v: DVector) -> DPoint:
        r"""
        @brief Adds a vector to a point


        Adds vector v to self by adding the coordinates.

        Starting with version 0.25, this method expects a vector argument.
        """
    def __copy__(self) -> DPoint:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DPoint:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, p: object) -> bool:
        r"""
        @brief Equality test operator

        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given point. This method enables points as hash keys.

        This method has been introduced in version 0.25.
        """
    def __imul__(self, f: float) -> DPoint:
        r"""
        @brief Scaling by some factor


        Scales object in place. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor: creates a point at 0,0
        """
    @overload
    def __init__(self, point: Point) -> None:
        r"""
        @brief Creates a floating-point coordinate point from an integer coordinate point

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipoint'.
        """
    @overload
    def __init__(self, v: DVector) -> None:
        r"""
        @brief Default constructor: creates a point at from an vector
        This constructor is equivalent to computing point(0,0)+v.
        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self, x: float, y: float) -> None:
        r"""
        @brief Constructor for a point from two coordinate values

        """
    def __itruediv__(self, d: float) -> DPoint:
        r"""
        @brief Division by some divisor


        Divides the object in place. All coordinates are divided with the given divisor and if necessary rounded.
        """
    def __lt__(self, p: DPoint) -> bool:
        r"""
        @brief "less" comparison operator


        This operator is provided to establish a sorting
        order
        """
    def __mul__(self, f: float) -> DPoint:
        r"""
        @brief Scaling by some factor


        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __ne__(self, p: object) -> bool:
        r"""
        @brief Inequality test operator

        """
    def __neg__(self) -> DPoint:
        r"""
        @brief Compute the negative of a point


        Returns a new point with -x, -y.

        This method has been added in version 0.23.
        """
    def __rmul__(self, f: float) -> DPoint:
        r"""
        @brief Scaling by some factor


        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __str__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief String conversion.
        If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    @overload
    def __sub__(self, p: DPoint) -> DVector:
        r"""
        @brief Subtract one point from another


        Subtract point p from self by subtracting the coordinates. This renders a vector.

        Starting with version 0.25, this method renders a vector.
        """
    @overload
    def __sub__(self, v: DVector) -> DPoint:
        r"""
        @brief Subtract one vector from a point


        Subtract vector v from from self by subtracting the coordinates. This renders a point.

        This method has been added in version 0.27.
        """
    def __truediv__(self, d: float) -> DPoint:
        r"""
        @brief Division by some divisor


        Returns the scaled object. All coordinates are divided with the given divisor and if necessary rounded.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def abs(self) -> float:
        r"""
        @brief The absolute value of the point (Euclidian distance to 0,0)

        The returned value is 'sqrt(x*x+y*y)'.

        This method has been introduced in version 0.23.
        """
    def assign(self, other: DPoint) -> None:
        r"""
        @brief Assigns another object to self
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def distance(self, d: DPoint) -> float:
        r"""
        @brief The Euclidian distance to another point


        @param d The other point to compute the distance to.
        """
    def dup(self) -> DPoint:
        r"""
        @brief Creates a copy of self
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given point. This method enables points as hash keys.

        This method has been introduced in version 0.25.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def sq_abs(self) -> float:
        r"""
        @brief The square of the absolute value of the point (Euclidian distance to 0,0)

        The returned value is 'x*x+y*y'.

        This method has been introduced in version 0.23.
        """
    def sq_distance(self, d: DPoint) -> float:
        r"""
        @brief The square Euclidian distance to another point


        @param d The other point to compute the distance to.
        """
    def to_itype(self, dbu: Optional[float] = ...) -> Point:
        r"""
        @brief Converts the point to an integer coordinate point

        The database unit can be specified to translate the floating-point coordinate point in micron units to an integer-coordinate point in database units. The point's' coordinates will be divided by the database unit.

        This method has been introduced in version 0.25.
        """
    def to_s(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief String conversion.
        If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def to_v(self) -> DVector:
        r"""
        @brief Turns the point into a vector
        This method returns a vector representing the distance from (0,0) to the point.This method has been introduced in version 0.25.
        """

class Point:
    r"""
    @brief An integer point class
    Points represent a coordinate in the two-dimensional coordinate space of layout. They are not geometrical objects by itself. But they are frequently used in the database API for various purposes.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    x: int
    r"""
    Getter:
    @brief Accessor to the x coordinate

    Setter:
    @brief Write accessor to the x coordinate
    """
    y: int
    r"""
    Getter:
    @brief Accessor to the y coordinate

    Setter:
    @brief Write accessor to the y coordinate
    """
    @classmethod
    def from_dpoint(cls, dpoint: DPoint) -> Point:
        r"""
        @brief Creates an integer coordinate point from a floating-point coordinate point

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dpoint'.
        """
    @classmethod
    def from_s(cls, s: str) -> Point:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> Point:
        r"""
        @brief Default constructor: creates a point at 0,0
        """
    @overload
    @classmethod
    def new(cls, dpoint: DPoint) -> Point:
        r"""
        @brief Creates an integer coordinate point from a floating-point coordinate point

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dpoint'.
        """
    @overload
    @classmethod
    def new(cls, v: Vector) -> Point:
        r"""
        @brief Default constructor: creates a point at from an vector
        This constructor is equivalent to computing point(0,0)+v.
        This method has been introduced in version 0.25.
        """
    @overload
    @classmethod
    def new(cls, x: int, y: int) -> Point:
        r"""
        @brief Constructor for a point from two coordinate values

        """
    def __add__(self, v: Vector) -> Point:
        r"""
        @brief Adds a vector to a point


        Adds vector v to self by adding the coordinates.

        Starting with version 0.25, this method expects a vector argument.
        """
    def __copy__(self) -> Point:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> Point:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, p: object) -> bool:
        r"""
        @brief Equality test operator

        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given point. This method enables points as hash keys.

        This method has been introduced in version 0.25.
        """
    def __imul__(self, f: float) -> Point:
        r"""
        @brief Scaling by some factor


        Scales object in place. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor: creates a point at 0,0
        """
    @overload
    def __init__(self, dpoint: DPoint) -> None:
        r"""
        @brief Creates an integer coordinate point from a floating-point coordinate point

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dpoint'.
        """
    @overload
    def __init__(self, v: Vector) -> None:
        r"""
        @brief Default constructor: creates a point at from an vector
        This constructor is equivalent to computing point(0,0)+v.
        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self, x: int, y: int) -> None:
        r"""
        @brief Constructor for a point from two coordinate values

        """
    def __itruediv__(self, d: float) -> Point:
        r"""
        @brief Division by some divisor


        Divides the object in place. All coordinates are divided with the given divisor and if necessary rounded.
        """
    def __lt__(self, p: Point) -> bool:
        r"""
        @brief "less" comparison operator


        This operator is provided to establish a sorting
        order
        """
    def __mul__(self, f: float) -> Point:
        r"""
        @brief Scaling by some factor


        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __ne__(self, p: object) -> bool:
        r"""
        @brief Inequality test operator

        """
    def __neg__(self) -> Point:
        r"""
        @brief Compute the negative of a point


        Returns a new point with -x, -y.

        This method has been added in version 0.23.
        """
    def __rmul__(self, f: float) -> Point:
        r"""
        @brief Scaling by some factor


        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __str__(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief String conversion.
        If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    @overload
    def __sub__(self, p: Point) -> Vector:
        r"""
        @brief Subtract one point from another


        Subtract point p from self by subtracting the coordinates. This renders a vector.

        Starting with version 0.25, this method renders a vector.
        """
    @overload
    def __sub__(self, v: Vector) -> Point:
        r"""
        @brief Subtract one vector from a point


        Subtract vector v from from self by subtracting the coordinates. This renders a point.

        This method has been added in version 0.27.
        """
    def __truediv__(self, d: float) -> Point:
        r"""
        @brief Division by some divisor


        Returns the scaled object. All coordinates are divided with the given divisor and if necessary rounded.
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def abs(self) -> float:
        r"""
        @brief The absolute value of the point (Euclidian distance to 0,0)

        The returned value is 'sqrt(x*x+y*y)'.

        This method has been introduced in version 0.23.
        """
    def assign(self, other: Point) -> None:
        r"""
        @brief Assigns another object to self
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def distance(self, d: Point) -> float:
        r"""
        @brief The Euclidian distance to another point


        @param d The other point to compute the distance to.
        """
    def dup(self) -> Point:
        r"""
        @brief Creates a copy of self
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given point. This method enables points as hash keys.

        This method has been introduced in version 0.25.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def sq_abs(self) -> float:
        r"""
        @brief The square of the absolute value of the point (Euclidian distance to 0,0)

        The returned value is 'x*x+y*y'.

        This method has been introduced in version 0.23.
        """
    def sq_distance(self, d: Point) -> float:
        r"""
        @brief The square Euclidian distance to another point


        @param d The other point to compute the distance to.
        """
    def to_dtype(self, dbu: Optional[float] = ...) -> DPoint:
        r"""
        @brief Converts the point to a floating-point coordinate point

        The database unit can be specified to translate the integer-coordinate point into a floating-point coordinate point in micron units. The database unit is basically a scaling factor.

        This method has been introduced in version 0.25.
        """
    def to_s(self, dbu: Optional[float] = ...) -> str:
        r"""
        @brief String conversion.
        If a DBU is given, the output units will be micrometers.

        The DBU argument has been added in version 0.27.6.
        """
    def to_v(self) -> Vector:
        r"""
        @brief Turns the point into a vector
        This method returns a vector representing the distance from (0,0) to the point.This method has been introduced in version 0.25.
        """

class SimplePolygon:
    r"""
    @brief A simple polygon class

    A simple polygon consists of an outer hull only. To support polygons with holes, use \Polygon.
    The hull contour consists of several points. The point
    list is normalized such that the leftmost, lowest point is 
    the first one. The orientation is normalized such that
    the orientation of the hull contour is clockwise.

    It is in no way checked that the contours are not overlapping
    This must be ensured by the user of the object
    when filling the contours.

    The \SimplePolygon class stores coordinates in integer format. A class that stores floating-point coordinates is \DSimplePolygon.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    @property
    def points(self) -> None:
        r"""
        WARNING: This variable can only be set, not retrieved.
        @brief Sets the points of the simple polygon

        @param pts An array of points to assign to the simple polygon

        See the constructor description for details about raw mode.
        """
    @classmethod
    def ellipse(cls, box: Box, n: int) -> SimplePolygon:
        r"""
        @brief Creates a simple polygon approximating an ellipse

        @param box The bounding box of the ellipse
        @param n The number of points that will be used to approximate the ellipse

        This method has been introduced in version 0.23.
        """
    @classmethod
    def from_dpoly(cls, dpolygon: DSimplePolygon) -> SimplePolygon:
        r"""
        @brief Creates an integer coordinate polygon from a floating-point coordinate polygon

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dpoly'.
        """
    @classmethod
    def from_s(cls, s: str) -> SimplePolygon:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> SimplePolygon:
        r"""
        @brief Default constructor: creates an empty (invalid) polygon
        """
    @overload
    @classmethod
    def new(cls, box: Box) -> SimplePolygon:
        r"""
        @brief Constructor converting a box to a polygon

        @param box The box to convert to a polygon
        """
    @overload
    @classmethod
    def new(cls, dpolygon: DSimplePolygon) -> SimplePolygon:
        r"""
        @brief Creates an integer coordinate polygon from a floating-point coordinate polygon

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dpoly'.
        """
    @overload
    @classmethod
    def new(cls, pts: Sequence[Point], raw: Optional[bool] = ...) -> SimplePolygon:
        r"""
        @brief Constructor given the points of the simple polygon

        @param pts The points forming the simple polygon
        @param raw If true, the points are taken as they are (see below)

        If the 'raw' argument is set to true, the points are taken as they are. Specifically no removal of redundant points or joining of coincident edges will take place. In effect, polygons consisting of a single point or two points can be constructed as well as polygons with duplicate points. Note that such polygons may cause problems in some applications.

        Regardless of raw mode, the point list will be adjusted such that the first point is the lowest-leftmost one and the orientation is clockwise always.

        The 'raw' argument has been added in version 0.24.
        """
    def __copy__(self) -> SimplePolygon:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> SimplePolygon:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, p: object) -> bool:
        r"""
        @brief Returns a value indicating whether self is equal to p
        @param p The object to compare against
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given polygon. This method enables polygons as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor: creates an empty (invalid) polygon
        """
    @overload
    def __init__(self, box: Box) -> None:
        r"""
        @brief Constructor converting a box to a polygon

        @param box The box to convert to a polygon
        """
    @overload
    def __init__(self, dpolygon: DSimplePolygon) -> None:
        r"""
        @brief Creates an integer coordinate polygon from a floating-point coordinate polygon

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dpoly'.
        """
    @overload
    def __init__(self, pts: Sequence[Point], raw: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor given the points of the simple polygon

        @param pts The points forming the simple polygon
        @param raw If true, the points are taken as they are (see below)

        If the 'raw' argument is set to true, the points are taken as they are. Specifically no removal of redundant points or joining of coincident edges will take place. In effect, polygons consisting of a single point or two points can be constructed as well as polygons with duplicate points. Note that such polygons may cause problems in some applications.

        Regardless of raw mode, the point list will be adjusted such that the first point is the lowest-leftmost one and the orientation is clockwise always.

        The 'raw' argument has been added in version 0.24.
        """
    def __lt__(self, p: SimplePolygon) -> bool:
        r"""
        @brief Returns a value indicating whether self is less than p
        @param p The object to compare against
        This operator is provided to establish some, not necessarily a certain sorting order

        This method has been introduced in version 0.25.
        """
    def __mul__(self, f: float) -> SimplePolygon:
        r"""
        @brief Scales the polygon by some factor

        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __ne__(self, p: object) -> bool:
        r"""
        @brief Returns a value indicating whether self is not equal to p
        @param p The object to compare against
        """
    def __rmul__(self, f: float) -> SimplePolygon:
        r"""
        @brief Scales the polygon by some factor

        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __str__(self) -> str:
        r"""
        @brief Returns a string representing the polygon
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def area(self) -> int:
        r"""
        @brief Gets the area of the polygon
        The area is correct only if the polygon is not self-overlapping and the polygon is oriented clockwise.
        """
    def area2(self) -> int:
        r"""
        @brief Gets the double area of the polygon
        This method is provided because the area for an integer-type polygon is a multiple of 1/2. Hence the double area can be expresses precisely as an integer for these types.

        This method has been introduced in version 0.26.1
        """
    def assign(self, other: SimplePolygon) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> Box:
        r"""
        @brief Returns the bounding box of the simple polygon
        """
    def compress(self, remove_reflected: bool) -> None:
        r"""
        @brief Compressed the simple polygon.

        This method removes redundant points from the polygon, such as points being on a line formed by two other points.
        If remove_reflected is true, points are also removed if the two adjacent edges form a spike.

        @param remove_reflected See description of the functionality.

        This method was introduced in version 0.18.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> SimplePolygon:
        r"""
        @brief Creates a copy of self
        """
    def each_edge(self) -> Iterator[Edge]:
        r"""
        @brief Iterates over the edges that make up the simple polygon
        """
    def each_point(self) -> Iterator[Point]:
        r"""
        @brief Iterates over the points that make up the simple polygon
        """
    def extract_rad(self) -> List[Any]:
        r"""
        @brief Extracts the corner radii from a rounded polygon

        Attempts to extract the radii of rounded corner polygon. This is essentially the inverse of the \round_corners method. If this method succeeds, if will return an array of four elements: @ul
        @li The polygon with the rounded corners replaced by edgy ones @/li
        @li The radius of the inner corners @/li
        @li The radius of the outer corners @/li
        @li The number of points per full circle @/li
        @/ul

        This method is based on some assumptions and may fail. In this case, an empty array is returned.

        If successful, the following code will more or less render the original polygon and parameters

        @code
        p = ...   # some polygon
        p.round_corners(ri, ro, n)
        (p2, ri2, ro2, n2) = p.extract_rad
        # -> p2 == p, ro2 == ro, ri2 == ri, n2 == n (within some limits)
        @/code

        This method was introduced in version 0.25.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given polygon. This method enables polygons as hash keys.

        This method has been introduced in version 0.25.
        """
    def inside(self, p: Point) -> bool:
        r"""
        @brief Gets a value indicating whether the given point is inside the polygon
        If the given point is inside or on the edge the polygon, true is returned. This tests works well only if the polygon is not self-overlapping and oriented clockwise. 
        """
    def is_box(self) -> bool:
        r"""
        @brief Returns a value indicating whether the polygon is a simple box.

        A polygon is a box if it is identical to it's bounding box.

        @return True if the polygon is a box.

        This method was introduced in version 0.23.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_empty(self) -> bool:
        r"""
        @brief Returns a value indicating whether the polygon is empty
        """
    def is_halfmanhattan(self) -> bool:
        r"""
        @brief Returns a value indicating whether the polygon is half-manhattan
        Half-manhattan polygons have edges which are multiples of 45 degree. These polygons can be clipped at a rectangle without potential grid snapping.

        This predicate was introduced in version 0.27.
        """
    def is_rectilinear(self) -> bool:
        r"""
        @brief Returns a value indicating whether the polygon is rectilinear
        """
    @overload
    def minkowski_sum(self, b: Box, resolve_holes: bool) -> Polygon:
        r"""
        @brief Computes the Minkowski sum of a polygon and a box

        @param b The box.
        @param resolve_holes If true, the output polygon will not contain holes, but holes are resolved by joining the holes with the hull.

        @return The new polygon representing the Minkowski sum of self and b.

        This method was introduced in version 0.22.
        """
    @overload
    def minkowski_sum(self, c: Sequence[Point], resolve_holes: bool) -> Polygon:
        r"""
        @brief Computes the Minkowski sum of a polygon and a contour of points (a trace)

        @param c The contour (a series of points forming the trace).
        @param resolve_holes If true, the output polygon will not contain holes, but holes are resolved by joining the holes with the hull.

        @return The new polygon representing the Minkowski sum of self and c.

        This method was introduced in version 0.22.
        """
    @overload
    def minkowski_sum(self, e: Edge, resolve_holes: bool) -> Polygon:
        r"""
        @brief Computes the Minkowski sum of a polygon and an edge

        @param e The edge.
        @param resolve_holes If true, the output polygon will not contain holes, but holes are resolved by joining the holes with the hull.

        @return The new polygon representing the Minkowski sum of self and e.

        This method was introduced in version 0.22.
        """
    @overload
    def minkowski_sum(self, p: SimplePolygon, resolve_holes: bool) -> Polygon:
        r"""
        @brief Computes the Minkowski sum of a polygon and a polygon

        @param p The other polygon.
        @param resolve_holes If true, the output polygon will not contain holes, but holes are resolved by joining the holes with the hull.

        @return The new polygon representing the Minkowski sum of self and p.

        This method was introduced in version 0.22.
        """
    @overload
    def minkowsky_sum(self, b: Box, resolve_holes: bool) -> Polygon:
        r"""
        @brief Computes the Minkowski sum of a polygon and a box

        @param b The box.
        @param resolve_holes If true, the output polygon will not contain holes, but holes are resolved by joining the holes with the hull.

        @return The new polygon representing the Minkowski sum of self and b.

        This method was introduced in version 0.22.
        """
    @overload
    def minkowsky_sum(self, c: Sequence[Point], resolve_holes: bool) -> Polygon:
        r"""
        @brief Computes the Minkowski sum of a polygon and a contour of points (a trace)

        @param c The contour (a series of points forming the trace).
        @param resolve_holes If true, the output polygon will not contain holes, but holes are resolved by joining the holes with the hull.

        @return The new polygon representing the Minkowski sum of self and c.

        This method was introduced in version 0.22.
        """
    @overload
    def minkowsky_sum(self, e: Edge, resolve_holes: bool) -> Polygon:
        r"""
        @brief Computes the Minkowski sum of a polygon and an edge

        @param e The edge.
        @param resolve_holes If true, the output polygon will not contain holes, but holes are resolved by joining the holes with the hull.

        @return The new polygon representing the Minkowski sum of self and e.

        This method was introduced in version 0.22.
        """
    @overload
    def minkowsky_sum(self, p: SimplePolygon, resolve_holes: bool) -> Polygon:
        r"""
        @brief Computes the Minkowski sum of a polygon and a polygon

        @param p The other polygon.
        @param resolve_holes If true, the output polygon will not contain holes, but holes are resolved by joining the holes with the hull.

        @return The new polygon representing the Minkowski sum of self and p.

        This method was introduced in version 0.22.
        """
    @overload
    def move(self, p: Vector) -> SimplePolygon:
        r"""
        @brief Moves the simple polygon.

        Moves the simple polygon by the given offset and returns the 
        moved simple polygon. The polygon is overwritten.

        @param p The distance to move the simple polygon.

        @return The moved simple polygon.
        """
    @overload
    def move(self, x: int, y: int) -> SimplePolygon:
        r"""
        @brief Moves the polygon.

        Moves the polygon by the given offset and returns the 
        moved polygon. The polygon is overwritten.

        @param x The x distance to move the polygon.
        @param y The y distance to move the polygon.

        @return The moved polygon (self).
        """
    @overload
    def moved(self, p: Vector) -> SimplePolygon:
        r"""
        @brief Returns the moved simple polygon

        Moves the simple polygon by the given offset and returns the 
        moved simple polygon. The polygon is not modified.

        @param p The distance to move the simple polygon.

        @return The moved simple polygon.
        """
    @overload
    def moved(self, x: int, y: int) -> SimplePolygon:
        r"""
        @brief Returns the moved polygon (does not modify self)

        Moves the polygon by the given offset and returns the 
        moved polygon. The polygon is not modified.

        @param x The x distance to move the polygon.
        @param y The y distance to move the polygon.

        @return The moved polygon.

        This method has been introduced in version 0.23.
        """
    def num_points(self) -> int:
        r"""
        @brief Gets the number of points
        """
    def perimeter(self) -> int:
        r"""
        @brief Gets the perimeter of the polygon
        The perimeter is sum of the lengths of all edges making up the polygon.
        """
    def point(self, p: int) -> Point:
        r"""
        @brief Gets a specific point of the contour@param p The index of the point to get
        If the index of the point is not a valid index, a default value is returned.
        This method was introduced in version 0.18.
        """
    def round_corners(self, rinner: float, router: float, n: int) -> SimplePolygon:
        r"""
        @brief Rounds the corners of the polygon

        Replaces the corners of the polygon with circle segments.

        @param rinner The circle radius of inner corners (in database units).
        @param router The circle radius of outer corners (in database units).
        @param n The number of points per full circle.

        @return The new polygon.

        This method was introduced in version 0.22 for integer coordinates and in 0.25 for all coordinate types.
        """
    def set_points(self, pts: Sequence[Point], raw: Optional[bool] = ...) -> None:
        r"""
        @brief Sets the points of the simple polygon

        @param pts An array of points to assign to the simple polygon
        @param raw If true, the points are taken as they are

        See the constructor description for details about raw mode.

        This method has been added in version 0.24.
        """
    def split(self) -> List[SimplePolygon]:
        r"""
        @brief Splits the polygon into two or more parts
        This method will break the polygon into parts. The exact breaking algorithm is unspecified, the result are smaller polygons of roughly equal number of points and 'less concave' nature. Usually the returned polygon set consists of two polygons, but there can be more. The merged region of the resulting polygons equals the original polygon with the exception of small snapping effects at new vertexes.

        The intended use for this method is a iteratively split polygons until the satisfy some maximum number of points limit.

        This method has been introduced in version 0.25.3.
        """
    def to_dtype(self, dbu: Optional[float] = ...) -> DSimplePolygon:
        r"""
        @brief Converts the polygon to a floating-point coordinate polygon

        The database unit can be specified to translate the integer-coordinate polygon into a floating-point coordinate polygon in micron units. The database unit is basically a scaling factor.

        This method has been introduced in version 0.25.
        """
    def to_s(self) -> str:
        r"""
        @brief Returns a string representing the polygon
        """
    @overload
    def touches(self, box: Box) -> bool:
        r"""
        @brief Returns true, if the polygon touches the given box.
        The box and the polygon touch if they overlap or their contours share at least one point.

        This method was introduced in version 0.25.1.
        """
    @overload
    def touches(self, edge: Edge) -> bool:
        r"""
        @brief Returns true, if the polygon touches the given edge.
        The edge and the polygon touch if they overlap or the edge shares at least one point with the polygon's contour.

        This method was introduced in version 0.25.1.
        """
    @overload
    def touches(self, polygon: Polygon) -> bool:
        r"""
        @brief Returns true, if the polygon touches the other polygon.
        The polygons touch if they overlap or their contours share at least one point.

        This method was introduced in version 0.25.1.
        """
    @overload
    def touches(self, simple_polygon: SimplePolygon) -> bool:
        r"""
        @brief Returns true, if the polygon touches the other polygon.
        The polygons touch if they overlap or their contours share at least one point.

        This method was introduced in version 0.25.1.
        """
    @overload
    def transform(self, t: ICplxTrans) -> SimplePolygon:
        r"""
        @brief Transforms the simple polygon with a complex transformation (in-place)

        Transforms the simple polygon with the given complex transformation.
        Modifies self and returns self. An out-of-place version which does not modify self is \transformed.

        @param t The transformation to apply.

        This method has been introduced in version 0.24.
        """
    @overload
    def transform(self, t: Trans) -> SimplePolygon:
        r"""
        @brief Transforms the simple polygon (in-place)

        Transforms the simple polygon with the given transformation.
        Modifies self and returns self. An out-of-place version which does not modify self is \transformed.

        @param t The transformation to apply.

        This method has been introduced in version 0.24.
        """
    @overload
    def transformed(self, t: CplxTrans) -> DSimplePolygon:
        r"""
        @brief Transforms the simple polygon.

        Transforms the simple polygon with the given complex transformation.
        Does not modify the simple polygon but returns the transformed polygon.

        @param t The transformation to apply.

        @return The transformed simple polygon.

        With version 0.25, the original 'transformed_cplx' method is deprecated and 'transformed' takes both simple and complex transformations.
        """
    @overload
    def transformed(self, t: ICplxTrans) -> SimplePolygon:
        r"""
        @brief Transforms the simple polygon.

        Transforms the simple polygon with the given complex transformation.
        Does not modify the simple polygon but returns the transformed polygon.

        @param t The transformation to apply.

        @return The transformed simple polygon (in this case an integer coordinate object).

        This method has been introduced in version 0.18.
        """
    @overload
    def transformed(self, t: Trans) -> SimplePolygon:
        r"""
        @brief Transforms the simple polygon.

        Transforms the simple polygon with the given transformation.
        Does not modify the simple polygon but returns the transformed polygon.

        @param t The transformation to apply.

        @return The transformed simple polygon.
        """
    def transformed_cplx(self, t: CplxTrans) -> DSimplePolygon:
        r"""
        @brief Transforms the simple polygon.

        Transforms the simple polygon with the given complex transformation.
        Does not modify the simple polygon but returns the transformed polygon.

        @param t The transformation to apply.

        @return The transformed simple polygon.

        With version 0.25, the original 'transformed_cplx' method is deprecated and 'transformed' takes both simple and complex transformations.
        """

class DSimplePolygon:
    r"""
    @brief A simple polygon class

    A simple polygon consists of an outer hull only. To support polygons with holes, use \DPolygon.
    The contour consists of several points. The point
    list is normalized such that the leftmost, lowest point is 
    the first one. The orientation is normalized such that
    the orientation of the hull contour is clockwise.

    It is in no way checked that the contours are not over-
    lapping. This must be ensured by the user of the object
    when filling the contours.

    The \DSimplePolygon class stores coordinates in floating-point format which gives a higher precision for some operations. A class that stores integer coordinates is \SimplePolygon.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    @property
    def points(self) -> None:
        r"""
        WARNING: This variable can only be set, not retrieved.
        @brief Sets the points of the simple polygon

        @param pts An array of points to assign to the simple polygon

        See the constructor description for details about raw mode.
        """
    @classmethod
    def ellipse(cls, box: DBox, n: int) -> DSimplePolygon:
        r"""
        @brief Creates a simple polygon approximating an ellipse

        @param box The bounding box of the ellipse
        @param n The number of points that will be used to approximate the ellipse

        This method has been introduced in version 0.23.
        """
    @classmethod
    def from_ipoly(cls, polygon: SimplePolygon) -> DSimplePolygon:
        r"""
        @brief Creates a floating-point coordinate polygon from an integer coordinate polygon
        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipoly'.
        """
    @classmethod
    def from_s(cls, s: str) -> DSimplePolygon:
        r"""
        @brief Creates an object from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> DSimplePolygon:
        r"""
        @brief Default constructor: creates an empty (invalid) polygon
        """
    @overload
    @classmethod
    def new(cls, box: DBox) -> DSimplePolygon:
        r"""
        @brief Constructor converting a box to a polygon

        @param box The box to convert to a polygon
        """
    @overload
    @classmethod
    def new(cls, polygon: SimplePolygon) -> DSimplePolygon:
        r"""
        @brief Creates a floating-point coordinate polygon from an integer coordinate polygon
        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipoly'.
        """
    @overload
    @classmethod
    def new(cls, pts: Sequence[DPoint], raw: Optional[bool] = ...) -> DSimplePolygon:
        r"""
        @brief Constructor given the points of the simple polygon

        @param pts The points forming the simple polygon
        @param raw If true, the points are taken as they are (see below)

        If the 'raw' argument is set to true, the points are taken as they are. Specifically no removal of redundant points or joining of coincident edges will take place. In effect, polygons consisting of a single point or two points can be constructed as well as polygons with duplicate points. Note that such polygons may cause problems in some applications.

        Regardless of raw mode, the point list will be adjusted such that the first point is the lowest-leftmost one and the orientation is clockwise always.

        The 'raw' argument has been added in version 0.24.
        """
    def __copy__(self) -> DSimplePolygon:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> DSimplePolygon:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, p: object) -> bool:
        r"""
        @brief Returns a value indicating whether self is equal to p
        @param p The object to compare against
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given polygon. This method enables polygons as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Default constructor: creates an empty (invalid) polygon
        """
    @overload
    def __init__(self, box: DBox) -> None:
        r"""
        @brief Constructor converting a box to a polygon

        @param box The box to convert to a polygon
        """
    @overload
    def __init__(self, polygon: SimplePolygon) -> None:
        r"""
        @brief Creates a floating-point coordinate polygon from an integer coordinate polygon
        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_ipoly'.
        """
    @overload
    def __init__(self, pts: Sequence[DPoint], raw: Optional[bool] = ...) -> None:
        r"""
        @brief Constructor given the points of the simple polygon

        @param pts The points forming the simple polygon
        @param raw If true, the points are taken as they are (see below)

        If the 'raw' argument is set to true, the points are taken as they are. Specifically no removal of redundant points or joining of coincident edges will take place. In effect, polygons consisting of a single point or two points can be constructed as well as polygons with duplicate points. Note that such polygons may cause problems in some applications.

        Regardless of raw mode, the point list will be adjusted such that the first point is the lowest-leftmost one and the orientation is clockwise always.

        The 'raw' argument has been added in version 0.24.
        """
    def __lt__(self, p: DSimplePolygon) -> bool:
        r"""
        @brief Returns a value indicating whether self is less than p
        @param p The object to compare against
        This operator is provided to establish some, not necessarily a certain sorting order

        This method has been introduced in version 0.25.
        """
    def __mul__(self, f: float) -> DSimplePolygon:
        r"""
        @brief Scales the polygon by some factor

        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __ne__(self, p: object) -> bool:
        r"""
        @brief Returns a value indicating whether self is not equal to p
        @param p The object to compare against
        """
    def __rmul__(self, f: float) -> DSimplePolygon:
        r"""
        @brief Scales the polygon by some factor

        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __str__(self) -> str:
        r"""
        @brief Returns a string representing the polygon
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def area(self) -> float:
        r"""
        @brief Gets the area of the polygon
        The area is correct only if the polygon is not self-overlapping and the polygon is oriented clockwise.
        """
    def area2(self) -> float:
        r"""
        @brief Gets the double area of the polygon
        This method is provided because the area for an integer-type polygon is a multiple of 1/2. Hence the double area can be expresses precisely as an integer for these types.

        This method has been introduced in version 0.26.1
        """
    def assign(self, other: DSimplePolygon) -> None:
        r"""
        @brief Assigns another object to self
        """
    def bbox(self) -> DBox:
        r"""
        @brief Returns the bounding box of the simple polygon
        """
    def compress(self, remove_reflected: bool) -> None:
        r"""
        @brief Compressed the simple polygon.

        This method removes redundant points from the polygon, such as points being on a line formed by two other points.
        If remove_reflected is true, points are also removed if the two adjacent edges form a spike.

        @param remove_reflected See description of the functionality.

        This method was introduced in version 0.18.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> DSimplePolygon:
        r"""
        @brief Creates a copy of self
        """
    def each_edge(self) -> Iterator[DEdge]:
        r"""
        @brief Iterates over the edges that make up the simple polygon
        """
    def each_point(self) -> Iterator[DPoint]:
        r"""
        @brief Iterates over the points that make up the simple polygon
        """
    def extract_rad(self) -> List[Any]:
        r"""
        @brief Extracts the corner radii from a rounded polygon

        Attempts to extract the radii of rounded corner polygon. This is essentially the inverse of the \round_corners method. If this method succeeds, if will return an array of four elements: @ul
        @li The polygon with the rounded corners replaced by edgy ones @/li
        @li The radius of the inner corners @/li
        @li The radius of the outer corners @/li
        @li The number of points per full circle @/li
        @/ul

        This method is based on some assumptions and may fail. In this case, an empty array is returned.

        If successful, the following code will more or less render the original polygon and parameters

        @code
        p = ...   # some polygon
        p.round_corners(ri, ro, n)
        (p2, ri2, ro2, n2) = p.extract_rad
        # -> p2 == p, ro2 == ro, ri2 == ri, n2 == n (within some limits)
        @/code

        This method was introduced in version 0.25.
        """
    def hash(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given polygon. This method enables polygons as hash keys.

        This method has been introduced in version 0.25.
        """
    def inside(self, p: DPoint) -> bool:
        r"""
        @brief Gets a value indicating whether the given point is inside the polygon
        If the given point is inside or on the edge the polygon, true is returned. This tests works well only if the polygon is not self-overlapping and oriented clockwise. 
        """
    def is_box(self) -> bool:
        r"""
        @brief Returns a value indicating whether the polygon is a simple box.

        A polygon is a box if it is identical to it's bounding box.

        @return True if the polygon is a box.

        This method was introduced in version 0.23.
        """
    def is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def is_empty(self) -> bool:
        r"""
        @brief Returns a value indicating whether the polygon is empty
        """
    def is_halfmanhattan(self) -> bool:
        r"""
        @brief Returns a value indicating whether the polygon is half-manhattan
        Half-manhattan polygons have edges which are multiples of 45 degree. These polygons can be clipped at a rectangle without potential grid snapping.

        This predicate was introduced in version 0.27.
        """
    def is_rectilinear(self) -> bool:
        r"""
        @brief Returns a value indicating whether the polygon is rectilinear
        """
    @overload
    def move(self, p: DVector) -> DSimplePolygon:
        r"""
        @brief Moves the simple polygon.

        Moves the simple polygon by the given offset and returns the 
        moved simple polygon. The polygon is overwritten.

        @param p The distance to move the simple polygon.

        @return The moved simple polygon.
        """
    @overload
    def move(self, x: float, y: float) -> DSimplePolygon:
        r"""
        @brief Moves the polygon.

        Moves the polygon by the given offset and returns the 
        moved polygon. The polygon is overwritten.

        @param x The x distance to move the polygon.
        @param y The y distance to move the polygon.

        @return The moved polygon (self).
        """
    @overload
    def moved(self, p: DVector) -> DSimplePolygon:
        r"""
        @brief Returns the moved simple polygon

        Moves the simple polygon by the given offset and returns the 
        moved simple polygon. The polygon is not modified.

        @param p The distance to move the simple polygon.

        @return The moved simple polygon.
        """
    @overload
    def moved(self, x: float, y: float) -> DSimplePolygon:
        r"""
        @brief Returns the moved polygon (does not modify self)

        Moves the polygon by the given offset and returns the 
        moved polygon. The polygon is not modified.

        @param x The x distance to move the polygon.
        @param y The y distance to move the polygon.

        @return The moved polygon.

        This method has been introduced in version 0.23.
        """
    def num_points(self) -> int:
        r"""
        @brief Gets the number of points
        """
    def perimeter(self) -> float:
        r"""
        @brief Gets the perimeter of the polygon
        The perimeter is sum of the lengths of all edges making up the polygon.
        """
    def point(self, p: int) -> DPoint:
        r"""
        @brief Gets a specific point of the contour@param p The index of the point to get
        If the index of the point is not a valid index, a default value is returned.
        This method was introduced in version 0.18.
        """
    def round_corners(self, rinner: float, router: float, n: int) -> DSimplePolygon:
        r"""
        @brief Rounds the corners of the polygon

        Replaces the corners of the polygon with circle segments.

        @param rinner The circle radius of inner corners (in database units).
        @param router The circle radius of outer corners (in database units).
        @param n The number of points per full circle.

        @return The new polygon.

        This method was introduced in version 0.22 for integer coordinates and in 0.25 for all coordinate types.
        """
    def set_points(self, pts: Sequence[DPoint], raw: Optional[bool] = ...) -> None:
        r"""
        @brief Sets the points of the simple polygon

        @param pts An array of points to assign to the simple polygon
        @param raw If true, the points are taken as they are

        See the constructor description for details about raw mode.

        This method has been added in version 0.24.
        """
    def split(self) -> List[DSimplePolygon]:
        r"""
        @brief Splits the polygon into two or more parts
        This method will break the polygon into parts. The exact breaking algorithm is unspecified, the result are smaller polygons of roughly equal number of points and 'less concave' nature. Usually the returned polygon set consists of two polygons, but there can be more. The merged region of the resulting polygons equals the original polygon with the exception of small snapping effects at new vertexes.

        The intended use for this method is a iteratively split polygons until the satisfy some maximum number of points limit.

        This method has been introduced in version 0.25.3.
        """
    def to_itype(self, dbu: Optional[float] = ...) -> SimplePolygon:
        r"""
        @brief Converts the polygon to an integer coordinate polygon
        The database unit can be specified to translate the floating-point coordinate polygon in micron units to an integer-coordinate polygon in database units. The polygon's' coordinates will be divided by the database unit.

        This method has been introduced in version 0.25.
        """
    def to_s(self) -> str:
        r"""
        @brief Returns a string representing the polygon
        """
    @overload
    def touches(self, box: DBox) -> bool:
        r"""
        @brief Returns true, if the polygon touches the given box.
        The box and the polygon touch if they overlap or their contours share at least one point.

        This method was introduced in version 0.25.1.
        """
    @overload
    def touches(self, edge: DEdge) -> bool:
        r"""
        @brief Returns true, if the polygon touches the given edge.
        The edge and the polygon touch if they overlap or the edge shares at least one point with the polygon's contour.

        This method was introduced in version 0.25.1.
        """
    @overload
    def touches(self, polygon: DPolygon) -> bool:
        r"""
        @brief Returns true, if the polygon touches the other polygon.
        The polygons touch if they overlap or their contours share at least one point.

        This method was introduced in version 0.25.1.
        """
    @overload
    def touches(self, simple_polygon: DSimplePolygon) -> bool:
        r"""
        @brief Returns true, if the polygon touches the other polygon.
        The polygons touch if they overlap or their contours share at least one point.

        This method was introduced in version 0.25.1.
        """
    @overload
    def transform(self, t: DCplxTrans) -> DSimplePolygon:
        r"""
        @brief Transforms the simple polygon with a complex transformation (in-place)

        Transforms the simple polygon with the given complex transformation.
        Modifies self and returns self. An out-of-place version which does not modify self is \transformed.

        @param t The transformation to apply.

        This method has been introduced in version 0.24.
        """
    @overload
    def transform(self, t: DTrans) -> DSimplePolygon:
        r"""
        @brief Transforms the simple polygon (in-place)

        Transforms the simple polygon with the given transformation.
        Modifies self and returns self. An out-of-place version which does not modify self is \transformed.

        @param t The transformation to apply.

        This method has been introduced in version 0.24.
        """
    @overload
    def transformed(self, t: DCplxTrans) -> DSimplePolygon:
        r"""
        @brief Transforms the simple polygon.

        Transforms the simple polygon with the given complex transformation.
        Does not modify the simple polygon but returns the transformed polygon.

        @param t The transformation to apply.

        @return The transformed simple polygon.

        With version 0.25, the original 'transformed_cplx' method is deprecated and 'transformed' takes both simple and complex transformations.
        """
    @overload
    def transformed(self, t: DTrans) -> DSimplePolygon:
        r"""
        @brief Transforms the simple polygon.

        Transforms the simple polygon with the given transformation.
        Does not modify the simple polygon but returns the transformed polygon.

        @param t The transformation to apply.

        @return The transformed simple polygon.
        """
    @overload
    def transformed(self, t: VCplxTrans) -> SimplePolygon:
        r"""
        @brief Transforms the polygon with the given complex transformation

        @param t The magnifying transformation to apply
        @return The transformed polygon (in this case an integer coordinate polygon)

        This method has been introduced in version 0.25.
        """
    def transformed_cplx(self, t: DCplxTrans) -> DSimplePolygon:
        r"""
        @brief Transforms the simple polygon.

        Transforms the simple polygon with the given complex transformation.
        Does not modify the simple polygon but returns the transformed polygon.

        @param t The transformation to apply.

        @return The transformed simple polygon.

        With version 0.25, the original 'transformed_cplx' method is deprecated and 'transformed' takes both simple and complex transformations.
        """

class Polygon:
    r"""
    @brief A polygon class

    A polygon consists of an outer hull and zero to many
    holes. Each contour consists of several points. The point
    list is normalized such that the leftmost, lowest point is 
    the first one. The orientation is normalized such that
    the orientation of the hull contour is clockwise, while
    the orientation of the holes is counterclockwise.

    It is in no way checked that the contours are not overlapping.
    This must be ensured by the user of the object
    when filling the contours.

    A polygon can be asked for the number of holes using the \holes method. \each_point_hull delivers the points of the hull contour. \each_point_hole delivers the points of a specific hole. \each_edge delivers the edges (point-to-point connections) of both hull and holes. \bbox delivers the bounding box, \area the area and \perimeter the perimeter of the polygon.

    Here's an example of how to create a polygon:

    @code
    hull =  [ RBA::Point::new(0, 0),       RBA::Point::new(6000, 0), 
              RBA::Point::new(6000, 3000), RBA::Point::new(0, 3000) ]
    hole1 = [ RBA::Point::new(1000, 1000), RBA::Point::new(2000, 1000), 
              RBA::Point::new(2000, 2000), RBA::Point::new(1000, 2000) ]
    hole2 = [ RBA::Point::new(3000, 1000), RBA::Point::new(4000, 1000), 
              RBA::Point::new(4000, 2000), RBA::Point::new(3000, 2000) ]
    poly = RBA::Polygon::new(hull)
    poly.insert_hole(hole1)
    poly.insert_hole(hole2)

    # ask the polygon for some properties
    poly.holes      # -> 2
    poly.area       # -> 16000000
    poly.perimeter  # -> 26000
    poly.bbox       # -> (0,0;6000,3000)
    @/code

    The \Polygon class stores coordinates in integer format. A class that stores floating-point coordinates is \DPolygon.

    See @<a href="/programming/database_api.xml">The Database API@</a> for more details about the database objects.
    """
    PO_any: ClassVar[int]
    r"""
    @brief A value for the preferred orientation parameter of \decompose_convex
    This value indicates that there is not cut preference
    This constant has been introduced in version 0.25.
    """
    PO_horizontal: ClassVar[int]
    r"""
    @brief A value for the preferred orientation parameter of \decompose_convex
    This value indicates that there only horizontal cuts are allowed
    This constant has been introduced in version 0.25.
    """
    PO_htrapezoids: ClassVar[int]
    r"""
    @brief A value for the preferred orientation parameter of \decompose_convex
    This value indicates that cuts shall favor decomposition into horizontal trapezoids
    This constant has been introduced in version 0.25.
    """
    PO_vertical: ClassVar[int]
    r"""
    @brief A value for the preferred orientation parameter of \decompose_convex
    This value indicates that there only vertical cuts are allowed
    This constant has been introduced in version 0.25.
    """
    PO_vtrapezoids: ClassVar[int]
    r"""
    @brief A value for the preferred orientation parameter of \decompose_convex
    This value indicates that cuts shall favor decomposition into vertical trapezoids
    This constant has been introduced in version 0.25.
    """
    TD_htrapezoids: ClassVar[int]
    r"""
    @brief A value for the mode parameter of \decompose_trapezoids
    This value indicates simple decomposition mode. This mode produces horizontal trapezoids and tries to minimize the number of trapezoids.
    This constant has been introduced in version 0.25.
    """
    TD_simple: ClassVar[int]
    r"""
    @brief A value for the mode parameter of \decompose_trapezoids
    This value indicates simple decomposition mode. This mode is fast but does not make any attempts to produce less trapezoids.
    This constant has been introduced in version 0.25.
    """
    TD_vtrapezoids: ClassVar[int]
    r"""
    @brief A value for the mode parameter of \decompose_trapezoids
    This value indicates simple decomposition mode. This mode produces vertical trapezoids and tries to minimize the number of trapezoids.
    """
    @property
    def hull(self) -> None:
        r"""
        WARNING: This variable can only be set, not retrieved.
        @brief Sets the points of the hull of polygon
        @param p An array of points to assign to the polygon's hull
        The 'assign_hull' variant is provided in analogy to 'assign_hole'.
        """
    @classmethod
    def ellipse(cls, box: Box, n: int) -> Polygon:
        r"""
        @brief Creates a simple polygon approximating an ellipse

        @param box The bounding box of the ellipse
        @param n The number of points that will be used to approximate the ellipse

        This method has been introduced in version 0.23.
        """
    @classmethod
    def from_dpoly(cls, dpolygon: DPolygon) -> Polygon:
        r"""
        @brief Creates an integer coordinate polygon from a floating-point coordinate polygon

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dpolygon'.
        """
    @classmethod
    def from_s(cls, s: str) -> Polygon:
        r"""
        @brief Creates a polygon from a string
        Creates the object from a string representation (as returned by \to_s)

        This method has been added in version 0.23.
        """
    @overload
    @classmethod
    def new(cls) -> Polygon:
        r"""
        @brief Creates an empty (invalid) polygon
        """
    @overload
    @classmethod
    def new(cls, box: Box) -> Polygon:
        r"""
        @brief Creates a polygon from a box

        @param box The box to convert to a polygon
        """
    @overload
    @classmethod
    def new(cls, dpolygon: DPolygon) -> Polygon:
        r"""
        @brief Creates an integer coordinate polygon from a floating-point coordinate polygon

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dpolygon'.
        """
    @overload
    @classmethod
    def new(cls, sp: SimplePolygon) -> Polygon:
        r"""
        @brief Creates a polygon from a simple polygon
        @param sp The simple polygon that is converted into the polygon
        This method was introduced in version 0.22.
        """
    @overload
    @classmethod
    def new(cls, pts: Sequence[Point], raw: Optional[bool] = ...) -> Polygon:
        r"""
        @brief Creates a polygon from a point array for the hull

        @param pts The points forming the polygon hull
        @param raw If true, the point list won't be modified (see \assign_hull)

        The 'raw' argument was added in version 0.24.
        """
    def __copy__(self) -> Polygon:
        r"""
        @brief Creates a copy of self
        """
    def __deepcopy__(self) -> Polygon:
        r"""
        @brief Creates a copy of self
        """
    def __eq__(self, p: object) -> bool:
        r"""
        @brief Returns a value indicating whether the polygons are equal
        @param p The object to compare against
        """
    def __hash__(self) -> int:
        r"""
        @brief Computes a hash value
        Returns a hash value for the given polygon. This method enables polygons as hash keys.

        This method has been introduced in version 0.25.
        """
    @overload
    def __init__(self) -> None:
        r"""
        @brief Creates an empty (invalid) polygon
        """
    @overload
    def __init__(self, box: Box) -> None:
        r"""
        @brief Creates a polygon from a box

        @param box The box to convert to a polygon
        """
    @overload
    def __init__(self, dpolygon: DPolygon) -> None:
        r"""
        @brief Creates an integer coordinate polygon from a floating-point coordinate polygon

        This constructor has been introduced in version 0.25 and replaces the previous static method 'from_dpolygon'.
        """
    @overload
    def __init__(self, sp: SimplePolygon) -> None:
        r"""
        @brief Creates a polygon from a simple polygon
        @param sp The simple polygon that is converted into the polygon
        This method was introduced in version 0.22.
        """
    @overload
    def __init__(self, pts: Sequence[Point], raw: Optional[bool] = ...) -> None:
        r"""
        @brief Creates a polygon from a point array for the hull

        @param pts The points forming the polygon hull
        @param raw If true, the point list won't be modified (see \assign_hull)

        The 'raw' argument was added in version 0.24.
        """
    def __lt__(self, p: Polygon) -> bool:
        r"""
        @brief Returns a value indicating whether self is less than p
        @param p The object to compare against
        This operator is provided to establish some, not necessarily a certain sorting order
        """
    def __mul__(self, f: float) -> Polygon:
        r"""
        @brief Scales the polygon by some factor

        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __ne__(self, p: object) -> bool:
        r"""
        @brief Returns a value indicating whether the polygons are not equal
        @param p The object to compare against
        """
    def __rmul__(self, f: float) -> Polygon:
        r"""
        @brief Scales the polygon by some factor

        Returns the scaled object. All coordinates are multiplied with the given factor and if necessary rounded.
        """
    def __str__(self) -> str:
        r"""
        @brief Returns a string representing the polygon
        """
    def _create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def _destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def _destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def _is_const_object(self) -> bool:
        r"""
        @brief Returns a value indicating whether the reference is a const reference
        This method returns true, if self is a const reference.
        In that case, only const methods may be called on self.
        """
    def _manage(self) -> None:
        r"""
        @brief Marks the object as managed by the script side.
        After calling this method on an object, the script side will be responsible for the management of the object. This method may be called if an object is returned from a C++ function and the object is known not to be owned by any C++ instance. If necessary, the script side may delete the object if the script's reference is no longer required.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def _unmanage(self) -> None:
        r"""
        @brief Marks the object as no longer owned by the script side.
        Calling this method will make this object no longer owned by the script's memory management. Instead, the object must be managed in some other way. Usually this method may be called if it is known that some C++ object holds and manages this object. Technically speaking, this method will turn the script's reference into a weak reference. After the script engine decides to delete the reference, the object itself will still exist. If the object is not managed otherwise, memory leaks will occur.

        Usually it's not required to call this method. It has been introduced in version 0.24.
        """
    def area(self) -> int:
        r"""
        @brief Gets the area of the polygon
        The area is correct only if the polygon is not self-overlapping and the polygon is oriented clockwise.Orientation is ensured automatically in most cases.
        """
    def area2(self) -> int:
        r"""
        @brief Gets the double area of the polygon
        This method is provided because the area for an integer-type polygon is a multiple of 1/2. Hence the double area can be expresses precisely as an integer for these types.

        This method has been introduced in version 0.26.1
        """
    def assign(self, other: Polygon) -> None:
        r"""
        @brief Assigns another object to self
        """
    @overload
    def assign_hole(self, n: int, b: Box) -> None:
        r"""
        @brief Sets the box as the given hole of the polygon
        @param n The index of the hole to which the points should be assigned
        @param b The box to assign to the polygon's hole
        If the hole index is not valid, this method does nothing.
        This method was introduced in version 0.23.
        """
    @overload
    def assign_hole(self, n: int, p: Sequence[Point], raw: Optional[bool] = ...) -> None:
        r"""
        @brief Sets the points of the given hole of the polygon
        @param n The index of the hole to which the points should be assigned
        @param p An array of points to assign to the polygon's hole
        @param raw If true, the points won't be compressed (see \assign_hull)
        If the hole index is not valid, this method does nothing.

        This method was introduced in version 0.18.
        The 'raw' argument was added in version 0.24.
        """
    def assign_hull(self, p: Sequence[Point], raw: Optional[bool] = ...) -> None:
        r"""
        @brief Sets the points of the hull of polygon
        @param p An array of points to assign to the polygon's hull
        @param raw If true, the points won't be compressed

        If the 'raw' argument is set to true, the points are taken as they are. Specifically no removal of redundant points or joining of coincident edges will take place. In effect, polygons consisting of a single point or two points can be constructed as well as polygons with duplicate points. Note that such polygons may cause problems in some applications.

        Regardless of raw mode, the point list will be adjusted such that the first point is the lowest-leftmost one and the orientation is clockwise always.

        The 'assign_hull' variant is provided in analogy to 'assign_hole'.

        The 'raw' argument was added in version 0.24.
        """
    def bbox(self) -> Box:
        r"""
        @brief Returns the bounding box of the polygon
        The bounding box is the box enclosing all points of the polygon.
        """
    def compress(self, remove_reflected: bool) -> None:
        r"""
        @brief Compresses the polygon.

        This method removes redundant points from the polygon, such as points being on a line formed by two other points.
        If remove_reflected is true, points are also removed if the two adjacent edges form a spike.

        @param remove_reflected See description of the functionality.

        This method was introduced in version 0.18.
        """
    def create(self) -> None:
        r"""
        @brief Ensures the C++ object is created
        Use this method to ensure the C++ object is created, for example to ensure that resources are allocated. Usually C++ objects are created on demand and not necessarily when the script object is created.
        """
    def decompose_convex(self, preferred_orientation: Optional[int] = ...) -> List[SimplePolygon]:
        r"""
        @brief Decomposes the polygon into convex pieces

        This method returns a decomposition of the polygon that contains convex pieces only.
        If the polygon was convex already, the list returned has a single element which is the
        original polygon.

        @param preferred_orientation One of the PO_... constants

        This method was introduced in version 0.25.
        """
    def decompose_trapezoids(self, mode: Optional[int] = ...) -> List[SimplePolygon]:
        r"""
        @brief Decomposes the polygon into trapezoids

        This method returns a decomposition of the polygon into trapezoid pieces.
        It supports different modes for various applications. See the TD_... constants for details.

        @param mode One of the TD_... constants

        This method was introduced in version 0.25.
        """
    def destroy(self) -> None:
        r"""
        @brief Explicitly destroys the object
        Explicitly destroys the object on C++ side if it was owned by the script interpreter. Subsequent access to this object will throw an exception.
        If the object is not owned by the script, this method will do nothing.
        """
    def destroyed(self) -> bool:
        r"""
        @brief Returns a value indicating whether the object was already destroyed
        This method returns true, if the object was destroyed, either explicitly or by the C++ side.
        The latter may happen, if the object is owned by a C++ object which got destroyed itself.
        """
    def dup(self) -> Polygon:
        r"""
        @brief Creates a copy of self
        """
    @overload
    def each_edge(self) -> Iterator[Edge]:
        r"""
        @brief Iterates over the edges that m