/*
 * Copyright 2025 Bloomberg Finance LP
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef BUILDBOXCOMMON_OCIMANIFEST_H
#define BUILDBOXCOMMON_OCIMANIFEST_H

#include <nlohmann/json.hpp>
#include <optional>
#include <string>
#include <vector>

namespace buildboxcommon {

// Represents a single layer in a container image
struct OciManifestLayer {
    std::string d_mediaType;
    std::string d_digest;
    std::optional<size_t> d_size;

    nlohmann::json toJson() const;
    static OciManifestLayer fromJson(const nlohmann::json &j);
};

// Main structure representing a complete OCI container image manifest
// media type application/vnd.oci.image.config.v1+json represents the oci image
// spec
// https://specs.opencontainers.org/image-spec/manifest/#example-image-manifest

// media type application/vnd.docker.distribution.manifest.v2+json represents
// the docker image spec
// https://docs.docker.com/reference/cli/docker/manifest/#inspect-an-images-manifest-object

// The structure is designed to be compatible with both OCI and Docker
// specifications, as Docker's V2 manifest is a subset of the OCI image spec.

struct OciManifest {
    // Registry metadata
    std::string d_registryUri;
    std::string d_repository;
    std::string d_reference;

    // Manifest data
    int d_schemaVersion;
    std::string d_mediaType;
    std::vector<OciManifestLayer> d_layers;
    // config and annotations are not supported

    nlohmann::json toJson() const;
    static OciManifest fromJson(const nlohmann::json &j);
};

// Represents platform information for multi-platform images
struct OciPlatform {
    std::string d_architecture;
    std::string d_os;

    nlohmann::json toJson() const;
    static OciPlatform fromJson(const nlohmann::json &j);
};

// Represents a single entry in a manifest list/index
struct OciManifestIndexEntry {
    std::string d_mediaType;
    std::string d_digest;
    std::optional<size_t> d_size;
    OciPlatform d_platform;
    std::string d_reference;

    nlohmann::json toJson() const;
    static OciManifestIndexEntry fromJson(const nlohmann::json &j);
};

// Represents a manifest list/index for multi-platform images
struct OciManifestIndex {
    // Registry metadata
    std::string d_registryUri;
    std::string d_repository;
    std::string d_reference;

    // Manifest list data
    int d_schemaVersion;
    std::string d_mediaType;
    std::vector<OciManifestIndexEntry> d_manifests;

    nlohmann::json toJson() const;
    static OciManifestIndex fromJson(const nlohmann::json &j);
};

} // namespace buildboxcommon

#endif // BUILDBOXCOMMON_OCIMANIFEST_H
