#!/usr/bin/env python3

# Copyright (c) the JPEG XL Project Authors. All rights reserved.
#
# Use of this source code is governed by a BSD-style
# license that can be found in the LICENSE file.

"""Demo for upscaling.

Given an upscaling factor which can be 2, 4 or 8 we demo upscaling an image by
that factor.

usage: upscaler_demo.py [-h] [--upscaling_factor N] input_filename output_filename

Upscaling of an image by a factor of 2, 4 or 8.

positional arguments:
  input_filename        of the PNG image to be upscaled.
  output_filename       where the upscaled image is written as PNG.

optional arguments:
  -h, --help            show this help message and exit
  --upscaling_factor N  where N must be 2, 4 (default) or 8.
"""
from PIL import Image

import argparse
import numpy as np


def convolution(pixels, kernel):
  """
  Returns the convolution of `pixels` with `kernel`.

  Uses padding such that the shape of the returned convoluted array is the
  same as the shape of `pixels`, scaled by the upscaling_factor implied by the
  `kernel`.

  Args:
    pixels: A [heigth, width]- or [height, width, num_channels]-array
    representing an image.

    kernel: A [upscaling_factor, upscaling_factor, kernel_size,
     kernel_size]-array used for the convolution.

  Returns:
    A [upscaling_factor*heigth, upscaling_factor*width]- or
    [upscaling_factor*height, upscaling_factor*width, num_channels]-array representing the
    convoluted upscaled image.
  """
  upscaling_factor, _, kernel_size, _ = kernel.shape
  output_shape = list(pixels.shape)
  output_shape[0] *= upscaling_factor
  output_shape[1] *= upscaling_factor
  shaped_pixels = pixels.reshape(pixels.shape[:2] + (-1,))
  pad_width = kernel_size//2
  padded_pixels = np.pad(
      shaped_pixels, 2*[2*[pad_width]] + [[0, 0]], mode='edge')
  x, y, _ = shaped_pixels.shape
  convoluted = np.block([[np.einsum('rc...,RCrc->...RC',
                                    padded_pixels[i - pad_width: i + pad_width + 1,
                                                  j - pad_width: j + pad_width + 1],
                                    kernel)
                          for j in range(pad_width, pad_width + y)]
                         for i in range(pad_width, pad_width + x)])
  return np.moveaxis(convoluted, 0, -1).reshape(output_shape)


def main():
    parser = argparse.ArgumentParser(
        description="Upscaling of an image by a factor of 2, 4 or 8.")
    parser.add_argument(
        "--upscaling_factor",
        type=int,
        help="where N must be  2, 4 (default) or 8.",
        nargs=1,
        default=[4],
        metavar='N')

    parser.add_argument(
        "input_filename",
        type=str,
        help="of the PNG image to be upscaled."
    )
    parser.add_argument(
        "output_filename",
        type=str,
        help="where the upscaled image is written as PNG."
    )

    args = parser.parse_args()
    upscaling_factor = args.upscaling_factor[0]
    kernel_size = 5
    if upscaling_factor not in (2, 4, 8):
        raise ValueError("upscaling_factor must be 2, 4 or 8.")
    kernel = np.array(_get_scaling_kernels()[upscaling_factor])
    assert kernel.shape == (
        upscaling_factor, upscaling_factor, kernel_size, kernel_size)
    orig_raw = Image.open(args.input_filename)
    orig = orig_raw.convert('RGB') if orig_raw.mode == 'P' else orig_raw
    upscaled_float = convolution(np.array(orig), kernel)

    upscaled = Image.fromarray(
        np.rint(np.clip(upscaled_float, 0, 255)).astype(np.uint8), orig.mode)
    upscaled.save(args.output_filename)


def _get_scaling_kernels():
    return {2: [[[[-0.017162003089909145, -0.0345230259724203, -0.04022174342753632,
    -0.029210135410064335, -0.006246448474415789], [-0.0345230259724203,
    0.14111091126932612, 0.28896754962953114, 0.0027871809188615613,
    -0.016102674925096382], [-0.04022174342753632, 0.28896754962953114,
    0.5666155013385713, 0.037776067445408776, -0.01986694439461126],
    [-0.029210135410064335, 0.0027871809188615535, 0.03777606744540877,
    -0.031447310821961526, -0.011850679991269755], [-0.006246448474415788,
    -0.01610267492509638, -0.019866944394611258, -0.011850679991269755,
    -0.0021353894928012747]], [[-0.006246448474415787, -0.029210135410064328,
    -0.040221743427536316, -0.034523025972420296, -0.01716200308990914],
    [-0.01610267492509638, 0.0027871809188615582, 0.2889675496295311,
    0.1411109112693261, -0.034523025972420296], [-0.019866944394611254,
    0.037776067445408755, 0.5666155013385712, 0.2889675496295311,
    -0.04022174342753631], [-0.011850679991269751, -0.03144731082196152,
    0.037776067445408755, 0.00278718091886156, -0.029210135410064324],
    [-0.0021353894928012743, -0.011850679991269751, -0.01986694439461125,
    -0.016102674925096375, -0.006246448474415786]]], [[[-0.006246448474415788,
    -0.01610267492509638, -0.019866944394611258, -0.011850679991269755,
    -0.0021353894928012747], [-0.02921013541006433, 0.002787180918861557,
    0.03777606744540876, -0.031447310821961526, -0.011850679991269755],
    [-0.04022174342753632, 0.28896754962953114, 0.5666155013385712,
    0.037776067445408776, -0.019866944394611258], [-0.0345230259724203,
    0.14111091126932612, 0.28896754962953114, 0.0027871809188615595,
    -0.016102674925096382], [-0.017162003089909145, -0.03452302597242031,
    -0.04022174342753633, -0.029210135410064335, -0.006246448474415789]],
    [[-0.0021353894928012747, -0.011850679991269755, -0.019866944394611258,
    -0.01610267492509638, -0.006246448474415788], [-0.011850679991269755,
    -0.031447310821961526, 0.03777606744540876, 0.002787180918861564,
    -0.02921013541006433], [-0.019866944394611258, 0.037776067445408776,
    0.5666155013385712, 0.28896754962953114, -0.040221743427536316],
    [-0.016102674925096382, 0.002787180918861556, 0.28896754962953114,
    0.14111091126932615, -0.0345230259724203], [-0.006246448474415789,
    -0.029210135410064335, -0.04022174342753633, -0.0345230259724203,
    -0.017162003089909145]]]],
    4: [[[[-0.024190672183733018, -0.03491987403959535, -0.036933511116288356,
    -0.03094284535390427, -0.005297851729507614], [-0.03491987403959535,
    0.23651958284942343, 0.3339294481745815, -0.010735433431237009,
    -0.013131808617501706], [-0.036933511116288356, 0.3339294481745815,
    0.4691419769580017, -0.0020927007975838127, -0.014845888917802386],
    [-0.030942845353904277, -0.010735433431237024, -0.0020927007975838035,
    -0.035516824721615874, -0.007548300818273063], [-0.005297851729507614,
    -0.013131808617501708, -0.014845888917802386, -0.007548300818273063,
    -0.0009165296078520004]], [[-0.01663431734052121, -0.03556862997573282,
    -0.0388890539890255, -0.035168498619353575, -0.009894687488916538],
    [-0.03556694367519552, 0.13048175192612746, 0.40103024797994685,
    0.03951149796198834, -0.02077584470399766], [-0.04064806042030105,
    0.18942529580147974, 0.5627989220290085, 0.06674400125646836,
    -0.023354943007463536], [-0.0226791877794674, -0.023635779153108244,
    0.0031580414703133823, -0.03399097960642573, -0.013595188211470589],
    [-0.003354666868160516, -0.011632944561351362, -0.016102939237729652,
    -0.00974087766582541, -0.0019162161212866041]], [[-0.009894687488916542,
    -0.03516849861935358, -0.03888905398902551, -0.035568629975732825,
    -0.016634317340521215], [-0.020775844703997664, 0.03951149796198835,
    0.4010302479799469, 0.13048175192612743, -0.03556694367519553],
    [-0.02335494300746354, 0.06674400125646836, 0.5627989220290086,
    0.18942529580147976, -0.04064806042030106], [-0.01359518821147059,
    -0.033990979606425734, 0.003158041470313383, -0.02363577915310824,
    -0.022679187779467407], [-0.0019162161212866043, -0.00974087766582541,
    -0.016102939237729656, -0.011632944561351364, -0.0033546668681605166]],
    [[-0.005297851729507613, -0.030942845353904264, -0.036933511116288356,
    -0.034919874039595344, -0.024190672183733015], [-0.013131808617501703,
    -0.010735433431237012, 0.33392944817458137, 0.23651958284942343,
    -0.03491987403959533], [-0.014845888917802382, -0.0020927007975838153,
    0.4691419769580016, 0.33392944817458153, -0.03693351111628834],
    [-0.007548300818273061, -0.03551682472161587, -0.0020927007975838053,
    -0.010735433431237016, -0.030942845353904264], [-0.0009165296078520002,
    -0.007548300818273061, -0.014845888917802382, -0.013131808617501704,
    -0.005297851729507613]]], [[[-0.01663431734052122, -0.03556694367519555,
    -0.040648060420301065, -0.02267918777946741, -0.0033546668681605175],
    [-0.03556862997573284, 0.13048175192612746, 0.18942529580147982,
    -0.023635779153108258, -0.011632944561351367], [-0.038889053989025514,
    0.401030247979947, 0.5627989220290087, 0.0031580414703133814,
    -0.01610293923772966], [-0.03516849861935359, 0.03951149796198835,
    0.06674400125646837, -0.03399097960642574, -0.009740877665825412],
    [-0.009894687488916542, -0.020775844703997664, -0.023354943007463547,
    -0.01359518821147059, -0.0019162161212866046]], [[-0.01095445961681655,
    -0.0319846366701879, -0.04455120920314033, -0.027997902912581793,
    -0.006459118117528576], [-0.0319846366701879, 0.06390599280769027,
    0.22963887988104975, 0.006309810655924714, -0.018973492447769916],
    [-0.04455120920314033, 0.2296388798810498, 0.67537268393182,
    0.08483369316914859, -0.025349935472536677], [-0.027997902912581786,
    0.006309810655924713, 0.08483369316914857, -0.02205197197850368,
    -0.016679994683747115], [-0.006459118117528575, -0.018973492447769913,
    -0.02534993547253667, -0.016679994683747115, -0.0038444335414517822]],
    [[-0.006459118117528576, -0.02799790291258179, -0.04455120920314034,
    -0.0319846366701879, -0.010954459616816552], [-0.01897349244776992,
    0.006309810655924714, 0.22963887988104978, 0.06390599280769028,
    -0.03198463667018791], [-0.025349935472536677, 0.08483369316914859,
    0.6753726839318202, 0.22963887988104975, -0.04455120920314034],
    [-0.016679994683747118, -0.022051971978503677, 0.08483369316914859,
    0.0063098106559247085, -0.02799790291258179], [-0.0038444335414517827,
    -0.016679994683747115, -0.02534993547253667, -0.018973492447769916,
    -0.006459118117528575]], [[-0.0033546668681605166, -0.022679187779467407,
    -0.04064806042030106, -0.03556694367519554, -0.016634317340521218],
    [-0.011632944561351364, -0.023635779153108254, 0.18942529580147976,
    0.13048175192612743, -0.035568629975732825], [-0.016102939237729656,
    0.0031580414703133762, 0.5627989220290086, 0.40103024797994696,
    -0.038889053989025486], [-0.009740877665825409, -0.033990979606425734,
    0.06674400125646834, 0.03951149796198835, -0.035168498619353575],
    [-0.0019162161212866041, -0.013595188211470589, -0.02335494300746354,
    -0.02077584470399766, -0.00989468748891654]]], [[[-0.009894687488916542,
    -0.020775844703997664, -0.023354943007463547, -0.01359518821147059,
    -0.0019162161212866046], [-0.03516849861935359, 0.03951149796198835,
    0.06674400125646836, -0.03399097960642574, -0.009740877665825412],
    [-0.03888905398902551, 0.401030247979947, 0.5627989220290087,
    0.0031580414703133814, -0.01610293923772966], [-0.03556862997573284,
    0.13048175192612746, 0.18942529580147982, -0.023635779153108258,
    -0.011632944561351367], [-0.016634317340521218, -0.03556694367519555,
    -0.040648060420301065, -0.022679187779467418, -0.0033546668681605175]],
    [[-0.006459118117528575, -0.018973492447769916, -0.02534993547253667,
    -0.016679994683747118, -0.0038444335414517827], [-0.02799790291258179,
    0.006309810655924723, 0.08483369316914856, -0.022051971978503684,
    -0.016679994683747118], [-0.04455120920314034, 0.22963887988104978,
    0.6753726839318203, 0.0848336931691486, -0.02534993547253667],
    [-0.03198463667018791, 0.06390599280769028, 0.22963887988104978,
    0.006309810655924709, -0.018973492447769923], [-0.010954459616816552,
    -0.03198463667018791, -0.04455120920314034, -0.027997902912581796,
    -0.006459118117528576]], [[-0.0038444335414517822, -0.01667999468374711,
    -0.02534993547253667, -0.018973492447769913, -0.006459118117528575],
    [-0.016679994683747115, -0.02205197197850368, 0.08483369316914854,
    0.006309810655924723, -0.027997902912581786], [-0.02534993547253667,
    0.08483369316914859, 0.6753726839318202, 0.22963887988104975,
    -0.04455120920314033], [-0.01897349244776992, 0.006309810655924712,
    0.22963887988104975, 0.06390599280769027, -0.0319846366701879],
    [-0.006459118117528576, -0.027997902912581786, -0.04455120920314033,
    -0.0319846366701879, -0.01095445961681655]], [[-0.0019162161212866041,
    -0.013595188211470589, -0.02335494300746354, -0.02077584470399766,
    -0.00989468748891654], [-0.009740877665825409, -0.033990979606425734,
    0.06674400125646834, 0.03951149796198835, -0.03516849861935358],
    [-0.016102939237729656, 0.0031580414703133762, 0.5627989220290086,
    0.40103024797994696, -0.03888905398902548], [-0.011632944561351364,
    -0.023635779153108254, 0.18942529580147976, 0.1304817519261275,
    -0.035568629975732825], [-0.0033546668681605166, -0.022679187779467414,
    -0.04064806042030106, -0.03556694367519554, -0.016634317340521215]]],
    [[[-0.005297851729507615, -0.013131808617501711, -0.01484588891780239,
    -0.007548300818273065, -0.0009165296078520006], [-0.030942845353904277,
    -0.010735433431237028, -0.0020927007975838087, -0.03551682472161588,
    -0.007548300818273065], [-0.03693351111628837, 0.3339294481745815,
    0.4691419769580017, -0.002092700797583813, -0.01484588891780239],
    [-0.03491987403959536, 0.23651958284942348, 0.33392944817458153,
    -0.010735433431237012, -0.01313180861750171], [-0.024190672183733025,
    -0.034919874039595365, -0.03693351111628837, -0.030942845353904277,
    -0.005297851729507615]], [[-0.0033546668681605166, -0.011632944561351364,
    -0.016102939237729656, -0.009740877665825412, -0.0019162161212866043],
    [-0.022679187779467404, -0.023635779153108247, 0.003158041470313383,
    -0.033990979606425734, -0.013595188211470589], [-0.04064806042030106,
    0.18942529580147982, 0.5627989220290085, 0.06674400125646837,
    -0.023354943007463547], [-0.03556694367519553, 0.1304817519261275,
    0.4010302479799469, 0.03951149796198835, -0.020775844703997653],
    [-0.016634317340521215, -0.035568629975732825, -0.038889053989025514,
    -0.035168498619353575, -0.009894687488916542]], [[-0.0019162161212866048,
    -0.009740877665825414, -0.01610293923772966, -0.011632944561351367,
    -0.0033546668681605175], [-0.01359518821147059, -0.03399097960642574,
    0.0031580414703133836, -0.023635779153108254, -0.022679187779467407],
    [-0.023354943007463554, 0.06674400125646839, 0.5627989220290086,
    0.18942529580147982, -0.040648060420301065], [-0.020775844703997657,
    0.03951149796198836, 0.401030247979947, 0.13048175192612746,
    -0.035566943675195535], [-0.009894687488916544, -0.03516849861935359,
    -0.03888905398902552, -0.03556862997573283, -0.016634317340521218]],
    [[-0.0009165296078520004, -0.007548300818273063, -0.014845888917802386,
    -0.013131808617501708, -0.005297851729507614], [-0.007548300818273063,
    -0.035516824721615874, -0.0020927007975838083, -0.010735433431237009,
    -0.03094284535390427], [-0.014845888917802386, -0.0020927007975838166,
    0.4691419769580016, 0.3339294481745815, -0.036933511116288356],
    [-0.013131808617501706, -0.010735433431237014, 0.3339294481745815,
    0.23651958284942348, -0.03491987403959534], [-0.005297851729507614,
    -0.03094284535390427, -0.03693351111628836, -0.03491987403959535,
    -0.024190672183733018]]]],
    8: [[[[-0.029286133281073247, -0.03706352644207269, -0.0378381168526885,
    -0.03324558280295302, -0.004476318148146651], [-0.0370635264420727,
    0.29895328454745274, 0.3575770812164143, -0.024475522375569658,
    -0.010817484288013228], [-0.0378381168526885, 0.35757708121641435,
    0.42720050241527285, -0.0224893852885426, -0.01155272937910007],
    [-0.03324558280295302, -0.024475522375569672, -0.022489385288542597,
    -0.03680917952171095, -0.005422291349995999], [-0.00447631814814665,
    -0.01081748428801323, -0.011552729379100074, -0.005422291349995998,
    -0.00045072273860512197]], [[-0.02519406150475052, -0.037526010691823306,
    -0.03901507994141054, -0.03663285147762567, -0.006466489422914399],
    [-0.043145939817870266, 0.23903219477825294, 0.41119300519363017,
    -0.005730455022054139, -0.014502394951723473], [-0.04562755195174026,
    0.28689495518965613, 0.4909386897413151, -7.890574314417001e-05,
    -0.015459264122748742], [-0.029204772772557758, -0.02788574061911041,
    -0.021181804710686657, -0.039424021044039116, -0.007755474877032563],
    [-0.003601096394526256, -0.010202069931803576, -0.012319067611648214,
    -0.006389875713059274, -0.0007159165805851706]], [[-0.020664074967504838,
    -0.03838632575427139, -0.04002101086742024, -0.03900035414027985,
    -0.009019734953997754], [-0.042468451339058966, 0.1756761813778118,
    0.45220642702382896, 0.02287757117854141, -0.019367833372750356],
    [-0.045626588213857136, 0.21238920010551757, 0.5398093391410694,
    0.033694739393926816, -0.020702111700092594], [-0.024336140047717,
    -0.03193943219458267, -0.020308275361446707, -0.04044013741654317,
    -0.010740155274818487], [-0.002791220988040244, -0.009571146384946013,
    -0.012883266171804216, -0.007309372111524051, -0.0010778269600400276]],
    [[-0.016263925397518374, -0.039541478550530786, -0.04046620032608076,
    -0.03979621423581153, -0.012244853215160445], [-0.03583254566206615,
    0.11572472115297627, 0.47416733354946305, 0.06284440084948137,
    -0.026850659249274114], [-0.038669884759381434, 0.1422954970729258,
    0.5659339775075575, 0.08045180751196822, -0.028882977402423956],
    [-0.01930821727497102, -0.03620398561701563, -0.019741250657301437,
    -0.03919545281633189, -0.014560933634183603], [-0.0021015621671157305,
    -0.008907053401106528, -0.013176682690936201, -0.008138951872408835,
    -0.0015349087147535298]], [[-0.01224485321516044, -0.03979621423581152,
    -0.04046620032608074, -0.03954147855053078, -0.016263925397518367],
    [-0.0268506592492741, 0.06284440084948137, 0.47416733354946283,
    0.11572472115297619, -0.03583254566206614], [-0.028882977402423942,
    0.0804518075119682, 0.5659339775075574, 0.14229549707292571,
    -0.03866988475938142], [-0.014560933634183596, -0.03919545281633188,
    -0.01974125065730143, -0.03620398561701561, -0.019308217274971014],
    [-0.0015349087147535291, -0.008138951872408828, -0.013176682690936196,
    -0.008907053401106523, -0.002101562167115729]], [[-0.00901973495399775,
    -0.039000354140279844, -0.040021010867420236, -0.03838632575427138,
    -0.020664074967504838], [-0.019367833372750352, 0.02287757117854141,
    0.4522064270238289, 0.17567618137781174, -0.04246845133905896],
    [-0.020702111700092587, 0.03369473939392681, 0.5398093391410693,
    0.21238920010551757, -0.04562658821385712], [-0.010740155274818485,
    -0.04044013741654316, -0.020308275361446707, -0.031939432194582666,
    -0.024336140047717], [-0.0010778269600400273, -0.007309372111524049,
    -0.012883266171804212, -0.00957114638494601, -0.0027912209880402426]],
    [[-0.006466489422914402, -0.03663285147762569, -0.03901507994141056,
    -0.03752601069182331, -0.02519406150475053], [-0.014502394951723478,
    -0.005730455022054147, 0.4111930051936302, 0.23903219477825297,
    -0.04314593981787026], [-0.015459264122748746, -7.890574314417718e-05,
    0.4909386897413152, 0.2868949551896563, -0.045627551951740265],
    [-0.007755474877032565, -0.03942402104403913, -0.021181804710686664,
    -0.027885740619110408, -0.029204772772557765], [-0.0007159165805851706,
    -0.006389875713059275, -0.012319067611648218, -0.01020206993180358,
    -0.003601096394526257]], [[-0.00447631814814665, -0.03324558280295302,
    -0.0378381168526885, -0.03706352644207268, -0.02928613328107324],
    [-0.01081748428801323, -0.024475522375569672, 0.3575770812164142,
    0.2989532845474528, -0.03706352644207268], [-0.01155272937910007,
    -0.02248938528854261, 0.42720050241527285, 0.35757708121641446,
    -0.037838116852688494], [-0.005422291349995998, -0.03680917952171095,
    -0.022489385288542604, -0.024475522375569658, -0.03324558280295301],
    [-0.0004507227386051219, -0.005422291349995998, -0.01155272937910007,
    -0.010817484288013232, -0.00447631814814665]]], [[[-0.025194061504750523,
    -0.043145939817870266, -0.04562755195174026, -0.02920477277255776,
    -0.0036010963945262565], [-0.037526010691823306, 0.23903219477825288,
    0.28689495518965624, -0.0278857406191104, -0.010202069931803576],
    [-0.03901507994141054, 0.4111930051936302, 0.4909386897413151,
    -0.021181804710686657, -0.01231906761164821], [-0.03663285147762567,
    -0.005730455022054155, -7.890574314415865e-05, -0.039424021044039116,
    -0.006389875713059272], [-0.0064664894229143986, -0.014502394951723471,
    -0.015459264122748746, -0.00775547487703256, -0.0007159165805851706]],
    [[-0.02128481178805433, -0.04173044153813555, -0.04831487472573022,
    -0.03293190035303922, -0.005252595229206095], [-0.041730441538135564,
    0.18968272846778533, 0.3306368426789878, -0.013001053856678076,
    -0.01372950329294693], [-0.04831487472573022, 0.3306368426789878,
    0.5640812622041927, 0.004583518760872409, -0.016482266055193047],
    [-0.03293190035303923, -0.013001053856678086, 0.004583518760872414,
    -0.040827417160105635, -0.009045186119473492], [-0.005252595229206096,
    -0.01372950329294693, -0.016482266055193047, -0.00904518611947349,
    -0.0011168422627331077]], [[-0.017203222289937238, -0.040527364499551265,
    -0.050457063493932794, -0.036073170059570094, -0.007380297997922879],
    [-0.0401746501391571, 0.13727831636454993, 0.3640223411093611,
    0.010278898793053761, -0.01832107424986819], [-0.04887867620762643,
    0.24585519478421125, 0.6202613509857569, 0.04314806591631964,
    -0.022137366266623233], [-0.02790922286627615, -0.021178184193661707,
    0.007986619792820032, -0.039957113612285294, -0.012434273033433196],
    [-0.00411203529942813, -0.012971303942569701, -0.01723725281482718,
    -0.010225452530604957, -0.0016530642487971611]], [[-0.013417641633011908,
    -0.03965629331558996, -0.051516162733405924, -0.038148863041386254,
    -0.010058190693394595], [-0.03365072121724163, 0.08734505711506498,
    0.38194295165025005, 0.04338227748703876, -0.025259934728481214],
    [-0.04158013952905281, 0.16637288777763284, 0.6502702298731253,
    0.0962163605307964, -0.031013880437287037], [-0.02231705358706074,
    -0.02946265951499448, 0.009920547334197453, -0.03600283468483377,
    -0.01684919502363355], [-0.003131096848010505, -0.012180160279609381,
    -0.01763265975706309, -0.011256197301616299, -0.0023166274424323116]],
    [[-0.01005819069339459, -0.03814886304138624, -0.0515161627334059,
    -0.03965629331558995, -0.013417641633011903], [-0.025259934728481193,
    0.04338227748703876, 0.38194295165024994, 0.08734505711506492,
    -0.033650721217241615], [-0.031013880437287013, 0.09621636053079634,
    0.6502702298731251, 0.1663728877776328, -0.04158013952905279],
    [-0.016849195023633544, -0.03600283468483376, 0.009920547334197446,
    -0.029462659514994466, -0.022317053587060723], [-0.0023166274424323103,
    -0.01125619730161629, -0.017632659757063088, -0.012180160279609381,
    -0.0031310968480105037]], [[-0.007380297997922879, -0.0360731700595701,
    -0.0504570634939328, -0.040527364499551265, -0.01720322228993724],
    [-0.01832107424986819, 0.010278898793053765, 0.36402234110936105,
    0.1372783163645499, -0.040174650139157095], [-0.022137366266623233,
    0.043148065916319624, 0.6202613509857569, 0.24585519478421133,
    -0.04887867620762643], [-0.012434273033433196, -0.039957113612285294,
    0.007986619792820032, -0.0211781841936617, -0.027909222866276145],
    [-0.0016530642487971611, -0.010225452530604957, -0.01723725281482718,
    -0.012971303942569701, -0.004112035299428131]], [[-0.005252595229206095,
    -0.03293190035303923, -0.04831487472573021, -0.04173044153813554,
    -0.021284811788054327], [-0.013729503292946926, -0.013001053856678083,
    0.33063684267898774, 0.1896827284677853, -0.04173044153813554],
    [-0.016482266055193047, 0.004583518760872396, 0.5640812622041926,
    0.33063684267898785, -0.0483148747257302], [-0.00904518611947349,
    -0.04082741716010563, 0.0045835187608724145, -0.013001053856678076,
    -0.03293190035303922], [-0.0011168422627331075, -0.009045186119473489,
    -0.016482266055193043, -0.013729503292946926, -0.005252595229206093]],
    [[-0.0036010963945262565, -0.029204772772557765, -0.04562755195174027,
    -0.04314593981787027, -0.02519406150475053], [-0.01020206993180358,
    -0.02788574061911041, 0.2868949551896562, 0.239032194778253,
    -0.037526010691823306], [-0.012319067611648214, -0.02118180471068667,
    0.4909386897413152, 0.41119300519363033, -0.039015079941410534],
    [-0.0063898757130592745, -0.03942402104403913, -7.890574314417213e-05,
    -0.00573045502205414, -0.036632851477625676], [-0.0007159165805851707,
    -0.007755474877032561, -0.015459264122748746, -0.014502394951723478,
    -0.0064664894229143986]]], [[[-0.020664074967504838,
    -0.042468451339058966, -0.04562658821385713, -0.024336140047717003,
    -0.002791220988040243], [-0.03838632575427139, 0.1756761813778117,
    0.21238920010551754, -0.031939432194582666, -0.00957114638494601],
    [-0.04002101086742023, 0.4522064270238289, 0.5398093391410693,
    -0.020308275361446703, -0.012883266171804212], [-0.039000354140279844,
    0.022877571178541382, 0.03369473939392682, -0.04044013741654317,
    -0.007309372111524049], [-0.00901973495399775, -0.019367833372750352,
    -0.02070211170009259, -0.010740155274818482, -0.0010778269600400273]],
    [[-0.017203222289937245, -0.040174650139157116, -0.04887867620762643,
    -0.027909222866276156, -0.004112035299428132], [-0.040527364499551286,
    0.13727831636454993, 0.24585519478421136, -0.021178184193661704,
    -0.01297130394256971], [-0.05045706349393281, 0.3640223411093611,
    0.6202613509857569, 0.007986619792820032, -0.017237252814827183],
    [-0.03607317005957011, 0.010278898793053753, 0.04314806591631965,
    -0.03995711361228531, -0.010225452530604962], [-0.007380297997922879,
    -0.01832107424986819, -0.022137366266623236, -0.012434273033433195,
    -0.0016530642487971616]], [[-0.013741489638205826, -0.037976197105406395,
    -0.05142937279813894, -0.031173068184164848, -0.005819138225232018],
    [-0.03797619710540639, 0.09628103622608752, 0.271299908889608,
    -0.003537793416633379, -0.017341510615908634], [-0.05142937279813893,
    0.271299908889608, 0.6821432561027145, 0.050180479222290235,
    -0.023208515458651935], [-0.031173068184164848, -0.003537793416633386,
    0.050180479222290235, -0.03637638898995256, -0.013943731217351598],
    [-0.005819138225232015, -0.017341510615908627, -0.023208515458651935,
    -0.013943731217351598, -0.002408535881464665]], [[-0.010640027531642969,
    -0.03608088767126983, -0.05272168029782533, -0.03375669845324461,
    -0.007955856657996018], [-0.03153980584721341, 0.05686230181726655,
    0.28500998074905703, 0.02230594207226229, -0.023749554287216885],
    [-0.04383615619088237, 0.1845947431815049, 0.7151797455936234,
    0.10805612743320024, -0.03263677274980321], [-0.02511202585552835,
    -0.017286364047844695, 0.054073310615547404, -0.028675684605006257,
    -0.018931312997898533], [-0.004465109850519687, -0.01636186809305578,
    -0.023770526019940293, -0.01522847594949364, -0.0033333443560800225]],
    [[-0.007955856657996014, -0.033756698453244596, -0.052721680297825306,
    -0.03608088767126982, -0.010640027531642969], [-0.023749554287216878,
    0.022305942072262296, 0.285009980749057, 0.056862301817266515,
    -0.031539805847213394], [-0.0326367727498032, 0.10805612743320023,
    0.7151797455936234, 0.18459474318150482, -0.04383615619088237],
    [-0.018931312997898533, -0.028675684605006243, 0.05407331061554739,
    -0.0172863640478447, -0.025112025855528346], [-0.0033333443560800216,
    -0.015228475949493633, -0.023770526019940286, -0.016361868093055777,
    -0.004465109850519686]], [[-0.005819138225232017, -0.031173068184164845,
    -0.05142937279813894, -0.037976197105406395, -0.013741489638205831],
    [-0.01734151061590863, -0.003537793416633379, 0.27129990888960803,
    0.09628103622608748, -0.03797619710540639], [-0.023208515458651945,
    0.05018047922229022, 0.6821432561027146, 0.27129990888960803,
    -0.05142937279813893], [-0.013943731217351596, -0.03637638898995256,
    0.050180479222290235, -0.003537793416633377, -0.03117306818416484],
    [-0.0024085358814646654, -0.013943731217351596, -0.023208515458651942,
    -0.01734151061590863, -0.005819138225232016]], [[-0.004112035299428132,
    -0.02790922286627615, -0.04887867620762644, -0.04017465013915711,
    -0.017203222289937245], [-0.012971303942569708, -0.021178184193661718,
    0.24585519478421136, 0.13727831636454993, -0.04052736449955127],
    [-0.017237252814827183, 0.007986619792820018, 0.620261350985757,
    0.36402234110936116, -0.050457063493932794], [-0.01022545253060496,
    -0.039957113612285315, 0.043148065916319644, 0.010278898793053767,
    -0.0360731700595701], [-0.0016530642487971618, -0.012434273033433193,
    -0.022137366266623233, -0.018321074249868192, -0.007380297997922878]],
    [[-0.0027912209880402413, -0.02433614004771699, -0.04562658821385711,
    -0.042468451339058945, -0.020664074967504827], [-0.009571146384946006,
    -0.03193943219458265, 0.21238920010551743, 0.17567618137781163,
    -0.038386325754271367], [-0.012883266171804209, -0.020308275361446707,
    0.5398093391410691, 0.4522064270238288, -0.0400210108674202],
    [-0.007309372111524046, -0.040440137416543155, 0.033694739393926795,
    0.022877571178541393, -0.039000354140279817], [-0.001077826960040027,
    -0.010740155274818476, -0.02070211170009258, -0.019367833372750345,
    -0.009019734953997745]]], [[[-0.01626392539751837, -0.035832545662066145,
    -0.03866988475938143, -0.01930821727497102, -0.0021015621671157296],
    [-0.03954147855053079, 0.11572472115297622, 0.14229549707292574,
    -0.03620398561701562, -0.008907053401106523], [-0.04046620032608075,
    0.47416733354946294, 0.5659339775075575, -0.019741250657301427,
    -0.013176682690936197], [-0.039796214235811526, 0.06284440084948134,
    0.08045180751196822, -0.03919545281633188, -0.008138951872408828],
    [-0.012244853215160443, -0.0268506592492741, -0.02888297740242396,
    -0.014560933634183601, -0.0015349087147535293]], [[-0.013417641633011906,
    -0.03365072121724163, -0.04158013952905282, -0.022317053587060733,
    -0.003131096848010505], [-0.039656293315589966, 0.08734505711506495,
    0.16637288777763282, -0.029462659514994483, -0.012180160279609385],
    [-0.051516162733405924, 0.3819429516502501, 0.6502702298731253,
    0.00992054733419745, -0.01763265975706309], [-0.03814886304138625,
    0.04338227748703875, 0.09621636053079638, -0.03600283468483378,
    -0.011256197301616295], [-0.010058190693394593, -0.02525993472848121,
    -0.03101388043728703, -0.016849195023633558, -0.0023166274424323116]],
    [[-0.01064002753164297, -0.0315398058472134, -0.04383615619088237,
    -0.02511202585552835, -0.004465109850519686], [-0.03608088767126983,
    0.05686230181726653, 0.18459474318150484, -0.0172863640478447,
    -0.01636186809305578], [-0.05272168029782532, 0.2850099807490571,
    0.7151797455936235, 0.0540733106155474, -0.02377052601994029],
    [-0.0337566984532446, 0.02230594207226228, 0.1080561274332002,
    -0.028675684605006246, -0.01522847594949364], [-0.007955856657996014,
    -0.02374955428721688, -0.03263677274980321, -0.018931312997898533,
    -0.003333344356080022]], [[-0.008199753617852345, -0.02964168716094745,
    -0.04499286779343149, -0.02745350495005966, -0.006124077091711166],
    [-0.02964168716094745, 0.027274160377919326, 0.19446599876518117,
    0.0015983184753035505, -0.022324728394118268], [-0.04499286779343149,
    0.19446599876518125, 0.7498250634433566, 0.11452620166036631,
    -0.03348047712449868], [-0.027453504950059663, 0.0015983184753035505,
    0.11452620166036631, -0.016056808817843177, -0.02070338975868157],
    [-0.006124077091711163, -0.022324728394118268, -0.03348047712449867,
    -0.02070338975868157, -0.004582234640385923]], [[-0.006124077091711165,
    -0.027453504950059653, -0.04499286779343149, -0.02964168716094745,
    -0.008199753617852345], [-0.022324728394118268, 0.0015983184753035479,
    0.19446599876518117, 0.027274160377919316, -0.02964168716094745],
    [-0.03348047712449866, 0.11452620166036634, 0.7498250634433566,
    0.19446599876518117, -0.04499286779343149], [-0.020703389758681575,
    -0.016056808817843174, 0.11452620166036631, 0.0015983184753035442,
    -0.02745350495005966], [-0.004582234640385922, -0.020703389758681568,
    -0.03348047712449866, -0.022324728394118268, -0.006124077091711163]],
    [[-0.004465109850519687, -0.025112025855528342, -0.04383615619088236,
    -0.03153980584721341, -0.01064002753164297], [-0.01636186809305578,
    -0.017286364047844702, 0.18459474318150484, 0.056862301817266536,
    -0.03608088767126984], [-0.023770526019940296, 0.054073310615547404,
    0.7151797455936236, 0.285009980749057, -0.05272168029782532],
    [-0.015228475949493642, -0.028675684605006246, 0.10805612743320021,
    0.022305942072262292, -0.03375669845324461], [-0.003333344356080022,
    -0.018931312997898533, -0.03263677274980321, -0.023749554287216885,
    -0.007955856657996013]], [[-0.003131096848010504, -0.02231705358706072,
    -0.04158013952905278, -0.03365072121724162, -0.013417641633011903],
    [-0.01218016027960938, -0.029462659514994476, 0.16637288777763273,
    0.0873450571150649, -0.03965629331558995], [-0.017632659757063088,
    0.009920547334197435, 0.6502702298731252, 0.38194295165024994,
    -0.051516162733405875], [-0.01125619730161629, -0.036002834684833764,
    0.09621636053079632, 0.04338227748703877, -0.03814886304138623],
    [-0.0023166274424323103, -0.01684919502363354, -0.031013880437287016,
    -0.025259934728481197, -0.010058190693394588]], [[-0.0021015621671157296,
    -0.01930821727497101, -0.03866988475938142, -0.035832545662066145,
    -0.016263925397518367], [-0.008907053401106521, -0.03620398561701562,
    0.14229549707292571, 0.11572472115297618, -0.03954147855053078],
    [-0.013176682690936197, -0.019741250657301437, 0.5659339775075574,
    0.4741673335494629, -0.04046620032608073], [-0.008138951872408826,
    -0.03919545281633188, 0.08045180751196819, 0.06284440084948135,
    -0.039796214235811506], [-0.0015349087147535291, -0.014560933634183593,
    -0.028882977402423952, -0.026850659249274097, -0.01224485321516044]]],
    [[[-0.012244853215160442, -0.0268506592492741, -0.02888297740242396,
    -0.0145609336341836, -0.0015349087147535293], [-0.039796214235811526,
    0.06284440084948134, 0.08045180751196819, -0.03919545281633189,
    -0.008138951872408828], [-0.040466200326080747, 0.47416733354946294,
    0.5659339775075575, -0.019741250657301427, -0.013176682690936197],
    [-0.03954147855053079, 0.1157247211529762, 0.14229549707292577,
    -0.03620398561701562, -0.008907053401106523], [-0.016263925397518374,
    -0.035832545662066145, -0.03866988475938143, -0.019308217274971024,
    -0.00210156216711573]], [[-0.010058190693394592, -0.025259934728481204,
    -0.031013880437287023, -0.016849195023633547, -0.0023166274424323103],
    [-0.03814886304138625, 0.04338227748703876, 0.09621636053079634,
    -0.036002834684833764, -0.011256197301616292], [-0.05151616273340591,
    0.38194295165025, 0.6502702298731253, 0.009920547334197446,
    -0.017632659757063088], [-0.039656293315589966, 0.08734505711506492,
    0.16637288777763276, -0.029462659514994476, -0.012180160279609383],
    [-0.013417641633011903, -0.03365072121724163, -0.041580139529052804,
    -0.02231705358706074, -0.0031310968480105046]], [[-0.007955856657996016,
    -0.02374955428721689, -0.03263677274980321, -0.01893131299789854,
    -0.003333344356080023], [-0.03375669845324461, 0.02230594207226229,
    0.10805612743320021, -0.02867568460500625, -0.01522847594949364],
    [-0.05272168029782533, 0.28500998074905703, 0.7151797455936236,
    0.05407331061554741, -0.0237705260199403], [-0.03608088767126984,
    0.05686230181726652, 0.18459474318150484, -0.017286364047844702,
    -0.016361868093055783], [-0.01064002753164297, -0.03153980584721341,
    -0.04383615619088236, -0.025112025855528356, -0.004465109850519687]],
    [[-0.006124077091711165, -0.022324728394118264, -0.03348047712449866,
    -0.020703389758681568, -0.004582234640385924], [-0.02745350495005966,
    0.0015983184753035565, 0.11452620166036628, -0.016056808817843167,
    -0.020703389758681568], [-0.04499286779343149, 0.19446599876518122,
    0.7498250634433568, 0.1145262016603663, -0.03348047712449867],
    [-0.02964168716094745, 0.027274160377919326, 0.1944659987651812,
    0.0015983184753035424, -0.022324728394118264], [-0.008199753617852345,
    -0.02964168716094745, -0.04499286779343149, -0.027453504950059663,
    -0.006124077091711164]], [[-0.004582234640385923, -0.02070338975868156,
    -0.03348047712449866, -0.022324728394118268, -0.006124077091711164],
    [-0.020703389758681568, -0.01605680881784317, 0.11452620166036626,
    0.0015983184753035535, -0.02745350495005966], [-0.03348047712449866,
    0.11452620166036631, 0.7498250634433566, 0.1944659987651812,
    -0.04499286779343149], [-0.022324728394118268, 0.0015983184753035503,
    0.19446599876518122, 0.02727416037791932, -0.02964168716094745],
    [-0.006124077091711164, -0.027453504950059653, -0.04499286779343149,
    -0.02964168716094745, -0.008199753617852345]], [[-0.0033333443560800216,
    -0.018931312997898523, -0.0326367727498032, -0.023749554287216878,
    -0.007955856657996013], [-0.015228475949493635, -0.028675684605006243,
    0.10805612743320019, 0.022305942072262285, -0.0337566984532446],
    [-0.02377052601994029, 0.05407331061554739, 0.7151797455936234,
    0.2850099807490569, -0.052721680297825306], [-0.016361868093055777,
    -0.0172863640478447, 0.18459474318150482, 0.05686230181726653,
    -0.03608088767126982], [-0.004465109850519686, -0.025112025855528346,
    -0.04383615619088235, -0.03153980584721339, -0.010640027531642966]],
    [[-0.002316627442432311, -0.01684919502363354, -0.031013880437287023,
    -0.025259934728481207, -0.010058190693394592], [-0.011256197301616293,
    -0.036002834684833764, 0.09621636053079634, 0.04338227748703875,
    -0.03814886304138625], [-0.01763265975706309, 0.009920547334197437,
    0.6502702298731253, 0.38194295165025, -0.05151616273340589],
    [-0.012180160279609383, -0.029462659514994483, 0.16637288777763276,
    0.08734505711506496, -0.03965629331558995], [-0.0031310968480105046,
    -0.022317053587060733, -0.04158013952905281, -0.03365072121724162,
    -0.013417641633011903]], [[-0.00153490871475353, -0.014560933634183603,
    -0.028882977402423966, -0.02685065924927412, -0.012244853215160445],
    [-0.008138951872408833, -0.039195452816331904, 0.08045180751196822,
    0.06284440084948137, -0.03979621423581154], [-0.013176682690936204,
    -0.019741250657301448, 0.5659339775075575, 0.47416733354946317,
    -0.04046620032608075], [-0.008907053401106528, -0.036203985617015634,
    0.1422954970729258, 0.1157247211529763, -0.0395414785505308],
    [-0.002101562167115731, -0.01930821727497103, -0.03866988475938145,
    -0.035832545662066166, -0.01626392539751838]]], [[[-0.00901973495399775,
    -0.01936783337275035, -0.02070211170009259, -0.010740155274818482,
    -0.001077826960040027], [-0.039000354140279844, 0.022877571178541386,
    0.033694739393926795, -0.04044013741654317, -0.007309372111524048],
    [-0.040021010867420236, 0.452206427023829, 0.5398093391410694,
    -0.020308275361446707, -0.01288326617180421], [-0.038386325754271394,
    0.1756761813778117, 0.21238920010551754, -0.031939432194582666,
    -0.00957114638494601], [-0.020664074967504838, -0.042468451339058966,
    -0.04562658821385713, -0.024336140047717003, -0.002791220988040243]],
    [[-0.007380297997922877, -0.018321074249868192, -0.022137366266623233,
    -0.012434273033433195, -0.0016530642487971611], [-0.0360731700595701,
    0.01027889879305376, 0.04314806591631964, -0.03995711361228531,
    -0.010225452530604959], [-0.05045706349393281, 0.3640223411093611,
    0.6202613509857569, 0.007986619792820032, -0.017237252814827183],
    [-0.04052736449955128, 0.13727831636454993, 0.24585519478421136,
    -0.021178184193661704, -0.012971303942569708], [-0.017203222289937245,
    -0.040174650139157116, -0.04887867620762645, -0.027909222866276163,
    -0.004112035299428132]], [[-0.005819138225232015, -0.01734151061590863,
    -0.02320851545865194, -0.0139437312173516, -0.0024085358814646654],
    [-0.031173068184164845, -0.0035377934166333767, 0.05018047922229021,
    -0.03637638898995257, -0.013943731217351596], [-0.05142937279813893,
    0.27129990888960803, 0.6821432561027146, 0.050180479222290235,
    -0.023208515458651942], [-0.0379761971054064, 0.0962810362260875,
    0.27129990888960803, -0.0035377934166333793, -0.017341510615908634],
    [-0.01374148963820583, -0.0379761971054064, -0.05142937279813894,
    -0.031173068184164855, -0.005819138225232018]], [[-0.004465109850519687,
    -0.016361868093055777, -0.023770526019940293, -0.015228475949493638,
    -0.0033333443560800233], [-0.02511202585552834, -0.017286364047844695,
    0.054073310615547376, -0.028675684605006243, -0.018931312997898533],
    [-0.04383615619088236, 0.1845947431815049, 0.7151797455936236,
    0.10805612743320021, -0.03263677274980321], [-0.03153980584721341,
    0.05686230181726654, 0.285009980749057, 0.02230594207226228,
    -0.023749554287216885], [-0.010640027531642969, -0.03608088767126984,
    -0.052721680297825334, -0.03375669845324461, -0.007955856657996016]],
    [[-0.003333344356080022, -0.015228475949493635, -0.023770526019940282,
    -0.016361868093055777, -0.004465109850519686], [-0.018931312997898526,
    -0.02867568460500624, 0.054073310615547356, -0.01728636404784469,
    -0.02511202585552834], [-0.0326367727498032, 0.10805612743320023,
    0.7151797455936234, 0.18459474318150484, -0.04383615619088235],
    [-0.023749554287216878, 0.022305942072262296, 0.285009980749057,
    0.05686230181726651, -0.03153980584721339], [-0.007955856657996014,
    -0.033756698453244596, -0.05272168029782531, -0.03608088767126983,
    -0.010640027531642967]], [[-0.002408535881464665, -0.013943731217351592,
    -0.023208515458651935, -0.017341510615908627, -0.005819138225232014],
    [-0.013943731217351592, -0.03637638898995256, 0.050180479222290214,
    -0.003537793416633366, -0.031173068184164845], [-0.02320851545865193,
    0.05018047922229023, 0.6821432561027146, 0.271299908889608,
    -0.051429372798138924], [-0.017341510615908627, -0.003537793416633378,
    0.271299908889608, 0.0962810362260875, -0.03797619710540638],
    [-0.005819138225232016, -0.03117306818416484, -0.05142937279813893,
    -0.03797619710540639, -0.013741489638205826]], [[-0.0016530642487971614,
    -0.012434273033433195, -0.022137366266623233, -0.01832107424986819,
    -0.007380297997922878], [-0.01022545253060496, -0.03995711361228531,
    0.04314806591631963, 0.010278898793053765, -0.0360731700595701],
    [-0.017237252814827183, 0.007986619792820022, 0.6202613509857569,
    0.3640223411093611, -0.05045706349393281], [-0.012971303942569706,
    -0.021178184193661718, 0.24585519478421136, 0.13727831636454998,
    -0.04052736449955127], [-0.004112035299428132, -0.027909222866276156,
    -0.04887867620762645, -0.04017465013915711, -0.017203222289937245]],
    [[-0.0010778269600400273, -0.010740155274818482, -0.02070211170009259,
    -0.01936783337275035, -0.009019734953997749], [-0.007309372111524049,
    -0.04044013741654317, 0.03369473939392679, 0.022877571178541403,
    -0.039000354140279844], [-0.012883266171804212, -0.020308275361446713,
    0.5398093391410693, 0.4522064270238289, -0.04002101086742022],
    [-0.00957114638494601, -0.031939432194582666, 0.2123892001055175,
    0.17567618137781177, -0.03838632575427138], [-0.002791220988040243,
    -0.024336140047717003, -0.04562658821385713, -0.04246845133905897,
    -0.020664074967504838]]], [[[-0.006466489422914399, -0.014502394951723476,
    -0.015459264122748746, -0.007755474877032561, -0.0007159165805851706],
    [-0.036632851477625676, -0.005730455022054154, -7.890574314417718e-05,
    -0.03942402104403913, -0.006389875713059275], [-0.039015079941410555,
    0.4111930051936302, 0.4909386897413152, -0.021181804710686668,
    -0.012319067611648218], [-0.03752601069182331, 0.239032194778253,
    0.2868949551896562, -0.027885740619110408, -0.010202069931803578],
    [-0.025194061504750533, -0.04314593981787028, -0.04562755195174027,
    -0.029204772772557768, -0.003601096394526257]], [[-0.005252595229206095,
    -0.013729503292946928, -0.016482266055193047, -0.009045186119473492,
    -0.0011168422627331079], [-0.03293190035303923, -0.013001053856678081,
    0.004583518760872412, -0.040827417160105635, -0.009045186119473489],
    [-0.04831487472573022, 0.33063684267898774, 0.5640812622041926,
    0.004583518760872408, -0.01648226605519305], [-0.04173044153813555,
    0.18968272846778533, 0.3306368426789878, -0.01300105385667808,
    -0.013729503292946928], [-0.02128481178805433, -0.041730441538135564,
    -0.048314874725730234, -0.03293190035303923, -0.005252595229206095]],
    [[-0.004112035299428132, -0.012971303942569708, -0.017237252814827186,
    -0.010225452530604966, -0.0016530642487971618], [-0.027909222866276156,
    -0.02117818419366171, 0.007986619792820025, -0.039957113612285315,
    -0.0124342730334332], [-0.04887867620762645, 0.24585519478421144,
    0.620261350985757, 0.04314806591631966, -0.022137366266623243],
    [-0.040174650139157116, 0.13727831636454998, 0.3640223411093612,
    0.010278898793053765, -0.018321074249868192], [-0.017203222289937245,
    -0.04052736449955129, -0.05045706349393283, -0.03607317005957011,
    -0.007380297997922882]], [[-0.0031310968480105046, -0.012180160279609381,
    -0.017632659757063088, -0.011256197301616295, -0.0023166274424323108],
    [-0.022317053587060723, -0.02946265951499447, 0.009920547334197437,
    -0.036002834684833764, -0.016849195023633544], [-0.0415801395290528,
    0.16637288777763282, 0.6502702298731252, 0.09621636053079637,
    -0.031013880437287027], [-0.03365072121724162, 0.08734505711506496,
    0.38194295165025005, 0.04338227748703875, -0.025259934728481197],
    [-0.013417641633011903, -0.03965629331558996, -0.05151616273340592,
    -0.03814886304138624, -0.010058190693394595]], [[-0.0023166274424323103,
    -0.011256197301616293, -0.017632659757063088, -0.012180160279609383,
    -0.0031310968480105046], [-0.016849195023633544, -0.036002834684833764,
    0.009920547334197428, -0.029462659514994466, -0.02231705358706073],
    [-0.03101388043728702, 0.0962163605307964, 0.6502702298731252,
    0.1663728877776328, -0.041580139529052804], [-0.025259934728481197,
    0.043382277487038774, 0.38194295165025005, 0.08734505711506493,
    -0.033650721217241615], [-0.010058190693394593, -0.03814886304138624,
    -0.05151616273340592, -0.03965629331558995, -0.013417641633011903]],
    [[-0.0016530642487971614, -0.01022545253060496, -0.017237252814827183,
    -0.012971303942569706, -0.00411203529942813], [-0.012434273033433196,
    -0.03995711361228531, 0.007986619792820018, -0.021178184193661697,
    -0.027909222866276152], [-0.022137366266623233, 0.04314806591631965,
    0.6202613509857569, 0.24585519478421133, -0.04887867620762643],
    [-0.018321074249868185, 0.010278898793053763, 0.3640223411093611,
    0.13727831636454993, -0.040174650139157095], [-0.00738029799792288,
    -0.0360731700595701, -0.05045706349393282, -0.04052736449955128,
    -0.01720322228993724]], [[-0.0011168422627331075, -0.009045186119473489,
    -0.016482266055193047, -0.013729503292946922, -0.005252595229206093],
    [-0.00904518611947349, -0.04082741716010563, 0.004583518760872401,
    -0.01300105385667807, -0.032931900353039216], [-0.016482266055193047,
    0.004583518760872399, 0.5640812622041926, 0.33063684267898774,
    -0.0483148747257302], [-0.013729503292946922, -0.01300105385667808,
    0.33063684267898774, 0.1896827284677853, -0.04173044153813553],
    [-0.005252595229206094, -0.03293190035303922, -0.048314874725730206,
    -0.04173044153813555, -0.021284811788054327]], [[-0.0007159165805851706,
    -0.00775547487703256, -0.015459264122748746, -0.014502394951723471,
    -0.006466489422914398], [-0.006389875713059273, -0.039424021044039116,
    -7.890574314417675e-05, -0.0057304550220541334, -0.03663285147762567],
    [-0.012319067611648212, -0.02118180471068667, 0.4909386897413151,
    0.4111930051936303, -0.03901507994141054], [-0.010202069931803573,
    -0.02788574061911041, 0.28689495518965613, 0.23903219477825297,
    -0.037526010691823306], [-0.0036010963945262557, -0.029204772772557758,
    -0.04562755195174025, -0.043145939817870266, -0.025194061504750526]]],
    [[[-0.004476318148146651, -0.010817484288013232, -0.011552729379100072,
    -0.005422291349995998, -0.0004507227386051219], [-0.03324558280295301,
    -0.02447552237556967, -0.022489385288542604, -0.03680917952171095,
    -0.005422291349995998], [-0.0378381168526885, 0.35757708121641424,
    0.4272005024152728, -0.022489385288542604, -0.011552729379100074],
    [-0.0370635264420727, 0.2989532845474528, 0.35757708121641435,
    -0.024475522375569658, -0.010817484288013228], [-0.029286133281073247,
    -0.037063526442072704, -0.037838116852688494, -0.03324558280295301,
    -0.00447631814814665]], [[-0.0036010963945262574, -0.01020206993180358,
    -0.012319067611648216, -0.006389875713059276, -0.0007159165805851708],
    [-0.029204772772557765, -0.02788574061911041, -0.02118180471068666,
    -0.03942402104403913, -0.007755474877032563], [-0.04562755195174027,
    0.2868949551896562, 0.490938689741315, -7.890574314416898e-05,
    -0.015459264122748749], [-0.043145939817870266, 0.239032194778253,
    0.41119300519363033, -0.005730455022054142, -0.014502394951723474],
    [-0.025194061504750533, -0.03752601069182331, -0.039015079941410555,
    -0.036632851477625676, -0.0064664894229144]], [[-0.002791220988040242,
    -0.009571146384946008, -0.012883266171804207, -0.007309372111524051,
    -0.0010778269600400271], [-0.024336140047716993, -0.03193943219458266,
    -0.020308275361446703, -0.040440137416543155, -0.010740155274818482],
    [-0.04562658821385712, 0.2123892001055175, 0.5398093391410692,
    0.033694739393926816, -0.020702111700092587], [-0.042468451339058945,
    0.1756761813778117, 0.4522064270238289, 0.022877571178541396,
    -0.019367833372750342], [-0.020664074967504824, -0.03838632575427138,
    -0.040021010867420236, -0.03900035414027984, -0.00901973495399775]],
    [[-0.0021015621671157296, -0.008907053401106525, -0.013176682690936196,
    -0.008138951872408831, -0.0015349087147535293], [-0.019308217274971017,
    -0.03620398561701561, -0.019741250657301434, -0.03919545281633189,
    -0.014560933634183596], [-0.03866988475938143, 0.1422954970729258,
    0.5659339775075574, 0.0804518075119682, -0.028882977402423963],
    [-0.035832545662066145, 0.11572472115297625, 0.47416733354946283,
    0.06284440084948135, -0.026850659249274097], [-0.01626392539751837,
    -0.039541478550530786, -0.04046620032608076, -0.03979621423581152,
    -0.012244853215160443]], [[-0.0015349087147535293, -0.008138951872408831,
    -0.013176682690936201, -0.008907053401106528, -0.00210156216711573],
    [-0.0145609336341836, -0.0391954528163319, -0.019741250657301437,
    -0.03620398561701563, -0.01930821727497102], [-0.028882977402423963,
    0.08045180751196825, 0.5659339775075575, 0.14229549707292577,
    -0.038669884759381434], [-0.0268506592492741, 0.06284440084948138,
    0.47416733354946305, 0.11572472115297622, -0.03583254566206615],
    [-0.012244853215160445, -0.03979621423581153, -0.04046620032608077,
    -0.03954147855053079, -0.016263925397518374]], [[-0.0010778269600400273,
    -0.00730937211152405, -0.01288326617180421, -0.00957114638494601,
    -0.002791220988040242], [-0.010740155274818482, -0.04044013741654317,
    -0.020308275361446707, -0.031939432194582666, -0.024336140047717],
    [-0.020702111700092594, 0.03369473939392682, 0.5398093391410693,
    0.21238920010551754, -0.04562658821385713], [-0.019367833372750342,
    0.022877571178541396, 0.452206427023829, 0.17567618137781177,
    -0.04246845133905895], [-0.009019734953997752, -0.039000354140279844,
    -0.040021010867420236, -0.03838632575427138, -0.02066407496750483]],
    [[-0.0007159165805851705, -0.006389875713059274, -0.012319067611648212,
    -0.010202069931803576, -0.0036010963945262565], [-0.007755474877032563,
    -0.03942402104403912, -0.02118180471068666, -0.0278857406191104,
    -0.029204772772557758], [-0.015459264122748742, -7.890574314417695e-05,
    0.4909386897413151, 0.2868949551896562, -0.04562755195174026],
    [-0.01450239495172347, -0.0057304550220541465, 0.4111930051936302,
    0.23903219477825297, -0.04314593981787025], [-0.0064664894229144,
    -0.03663285147762567, -0.03901507994141055, -0.037526010691823306,
    -0.025194061504750526]], [[-0.0004507227386051219, -0.005422291349995998,
    -0.01155272937910007, -0.01081748428801323, -0.00447631814814665],
    [-0.005422291349995998, -0.03680917952171095, -0.022489385288542607,
    -0.024475522375569655, -0.03324558280295302], [-0.011552729379100074,
    -0.02248938528854261, 0.4272005024152728, 0.3575770812164143,
    -0.0378381168526885], [-0.010817484288013228, -0.02447552237556967,
    0.3575770812164143, 0.29895328454745285, -0.03706352644207268],
    [-0.00447631814814665, -0.03324558280295302, -0.0378381168526885,
    -0.03706352644207269, -0.029286133281073243]]]]
}

if __name__ == "__main__":
    main()